package ru.yandex.infra.auth.nanny;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.MoreObjects;

public class NannyServiceInfo {
    private final String name;
    private final Map<String, NannyAuthGroup> authGroups;

    private final String snapshotId;
    private final String commitComment;
    private final String commitAuthor;
    private final long commitTimestamp;

    public NannyServiceInfo(String name, Map<String, NannyAuthGroup> authGroups,
                            String snapshotId, String commitComment, String commitAuthor, long commitTimestamp) {
        this.name = name;
        this.authGroups = authGroups;
        this.snapshotId = snapshotId;
        this.commitComment = commitComment;
        this.commitAuthor = commitAuthor;
        this.commitTimestamp = commitTimestamp;
    }

    public static NannyServiceInfo fromJsonNode(JsonNode node) {
        var serviceId = node.get("_id").textValue();
        return fromAuthAttrsNode(serviceId, node.get("auth_attrs"));
    }

    public static NannyServiceInfo fromAuthAttrsNode(String serviceId, JsonNode node) {
        Map<String, NannyAuthGroup> groups = new HashMap<>();
        Iterator<Map.Entry<String, JsonNode>> groupsIterator = node.get("content").fields();

        while (groupsIterator.hasNext()) {
            Map.Entry<String, JsonNode> entry = groupsIterator.next();
            groups.put(entry.getKey(), NannyAuthGroup.fromJsonNode(entry.getValue()));
        }

        final JsonNode changeInfo = node.get("change_info");
        return new NannyServiceInfo(serviceId,
                groups,
                node.get("_id").textValue(),
                changeInfo.get("comment").textValue(),
                changeInfo.get("author").textValue(),
                changeInfo.get("ctime").longValue());
    }

    public String getName() {
        return name;
    }

    public String getCommitAuthor() {
        return commitAuthor;
    }

    @JsonProperty("content")
    public Map<String, NannyAuthGroup> getAuthGroups() {
        return authGroups;
    }

    @JsonProperty("snapshot_id")
    public String getSnapshotId() {
        return snapshotId;
    }

    @JsonProperty("comment")
    public String getCommitComment() {
        return commitComment;
    }

    public NannyServiceInfo withAuthGroups(Map<String, NannyAuthGroup> authGroups, String commitComment) {
        return new NannyServiceInfo(name, authGroups, snapshotId, commitComment, commitAuthor, commitTimestamp);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("name", name)
                .add("snapshotId", snapshotId)
                .add("authGroups", authGroups)
                .add("commitComment", commitComment)
                .add("commitAuthor", commitAuthor)
                .add("commitTimestamp", commitTimestamp)
                .toString();
    }
}
