package ru.yandex.infra.auth.servlets;

import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Charsets;
import com.google.common.net.MediaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubjectsProvider;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.auth.idm.service.IdmService;

public class ApiGetProjectAclServlet extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(ApiGetProjectAclServlet.class);
    private static final String REQUEST_ID_HEADER = "X-Auth-Request-Id";
    private static final String PROJECT_ID_PARAMETER = "project_id";

    private final ObjectMapper jsonMapper = new ObjectMapper();
    private final RolesInfo rolesInfo;
    private final RoleSubjectsProvider roleSubjectsProvider;
    private final boolean use404ForMissedProject;
    private final AtomicLong metricRequestsCount;
    private final AtomicLong metricFailedRequestsCount;

    public ApiGetProjectAclServlet(IdmService idmService, RolesInfo rolesInfo, RoleSubjectsProvider roleSubjectsProvider, boolean use404ForMissedProject) {
        this.rolesInfo = rolesInfo;
        this.roleSubjectsProvider = roleSubjectsProvider;
        this.use404ForMissedProject = use404ForMissedProject;
        this.metricRequestsCount = idmService.getRequestsMetric(IdmService.RequestType.GET_PROJECT_ACL);
        this.metricFailedRequestsCount = idmService.getFailedRequestsMetric(IdmService.RequestType.GET_PROJECT_ACL);

    }

    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        metricRequestsCount.incrementAndGet();

        try {
            resp.setCharacterEncoding(Charsets.UTF_8.name());
            resp.setContentType(MediaType.JSON_UTF_8.toString());

            final long start = System.currentTimeMillis();

            String projectId = req.getParameter(PROJECT_ID_PARAMETER);

            if (projectId != null && !projectId.isEmpty()) {
                String response = doGetProjectAcl(projectId);
                if (response == null) {
                    metricFailedRequestsCount.incrementAndGet();
                    resp.sendError(404, "Unknown project id: " + projectId);
                } else {
                    resp.getWriter().print(response);
                }
            } else {
                metricFailedRequestsCount.incrementAndGet();
                resp.sendError(400, "Invalid query. project_id parameter is missing");
            }

            String requestId = req.getHeader(REQUEST_ID_HEADER);
            String requestInfo = requestId != null ? REQUEST_ID_HEADER + ": " + requestId : "";
            LOG.info("Processing GET-PROJECT-ACL request for {} took: {} ms.{}",
                    projectId, System.currentTimeMillis() - start, requestInfo);
        } catch (Throwable exception) {
            LOG.error("Failed to process /get-project-acl request", exception);
            metricFailedRequestsCount.incrementAndGet();
            resp.sendError(500, "Internal authctl error: " + exception.getMessage());
        }
    }

    private String doGetProjectAcl(String projectId) throws IOException {

        Map<Role, Set<String>> roleMembers = roleSubjectsProvider.getRoleMembers(projectId);

        ProjectAcl.Builder builder = new ProjectAcl.Builder(rolesInfo);
        if (roleMembers != null) {
            builder.addDefaultProjectRoles(projectId);
            roleMembers.forEach((role, members) -> members.stream().sorted().forEach(member -> builder.addAcl(role, member)));
        }
        else if(use404ForMissedProject) {
            return null;
        }
        ProjectAcl projectAcl = builder.build();

        return jsonMapper.writerWithDefaultPrettyPrinter().writeValueAsString(projectAcl);
    }
}
