package ru.yandex.infra.auth.servlets;

import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.yp.YpGroupsClient;
import ru.yandex.infra.auth.yp.YpService;

import static ru.yandex.infra.auth.yp.YpGroupsHelper.SYSTEM_LABEL_KEY;

public class GroupsGCServlet extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(PrivateApiServlet.class);
    private final YpService ypService;

    public GroupsGCServlet(YpService ypService) {
        this.ypService = ypService;
    }

    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        checkGarbageGroups(req, resp, false);
    }

    public void doPost(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        if (!"true".equals(req.getParameter("fix"))) {
            resp.getWriter().println("Removing garbage yp groups is not allowed by request parameter 'fix' = false");
            return;
        }
        checkGarbageGroups(req, resp, true);
    }

    private void checkGarbageGroups(HttpServletRequest req, HttpServletResponse resp, boolean isAllowedToRemove) throws IOException {
        final long start = System.currentTimeMillis();
        final PrintWriter writer = resp.getWriter();

        String slaveCluster = req.getParameter("slave-cluster");
        if (slaveCluster != null) {
            boolean removeOnlyEmptyGroups = "true".equals(req.getParameter("empty"));
            removeGroupsFromSlaveClusterIfMissedOnMaster(slaveCluster, writer, isAllowedToRemove, removeOnlyEmptyGroups);
        } else {
            removeGarbageGroups(writer, isAllowedToRemove);
        }

        LOG.info("Processing {} {} request took: {} ms", req.getMethod(), req.getServletPath(), System.currentTimeMillis() - start);
    }

    private void removeGarbageGroups(PrintWriter writer, boolean isAllowedToRemove) {
        List<String> groupsToRemove = ypService.getGarbageGroups()
                .stream()
                .sorted()
                .collect(Collectors.toList());

        groupsToRemove.forEach(writer::println);

        if (isAllowedToRemove) {
            removeGroups(ypService.getMasterClusterClients().getGroupsClient(), groupsToRemove);
        }
    }

    private void removeGroupsFromSlaveClusterIfMissedOnMaster(String slaveCluster,
                                                              PrintWriter writer,
                                                              boolean isAllowedToRemove,
                                                              boolean removeOnlyEmptyGroups) {
        try {
            var system = ypService.getSystemName();
            var labels = ImmutableMap.of(SYSTEM_LABEL_KEY, system);

            YpGroupsClient masterClient = ypService.getMasterClusterClients().getGroupsClient();
            YpGroupsClient slaveClient = ypService.getSlaveClusterClients().get(slaveCluster).getGroupsClient();

            LOG.info("[{}] Loading all deploy groups with labels {}...", slaveCluster, labels);
            Map<String, Set<String>> allSlaveGroupsWithMembers = slaveClient.getGroupsWithLabels(labels).get();
            Set<String> slaveGroups = removeOnlyEmptyGroups ? allSlaveGroupsWithMembers
                    .entrySet()
                    .stream().filter(e -> e.getValue().isEmpty())
                    .map(Map.Entry::getKey)
                    .collect(Collectors.toSet()) :
                    allSlaveGroupsWithMembers.keySet();
            LOG.info("[{}] Loaded {} groups", slaveCluster, allSlaveGroupsWithMembers.size());

            LOG.info("[master] Loading all deploy groups with labels {}...", labels);
            Set<String> masterGroups = masterClient.getGroupsWithLabels(labels).get().keySet();
            LOG.info("[master] Loaded {} groups", masterGroups.size());

            List<String> groupsMissedOnMaster = Sets.difference(slaveGroups, masterGroups).stream().sorted().collect(Collectors.toList());
            LOG.info("Found {} groups of {} total present on {} but missed on master",
                    groupsMissedOnMaster.size(), allSlaveGroupsWithMembers.size(), slaveCluster);

            groupsMissedOnMaster.forEach(writer::println);

            if (isAllowedToRemove) {
                removeGroups(slaveClient, groupsMissedOnMaster);
            }
        } catch (Exception exception) {
            LOG.error("Failed to clean slave groups on {}: {}", slaveCluster, exception);
        }
    }

    private void removeGroups(YpGroupsClient groupsClient, List<String> groups) {
        try {
            List<String> bunch = new ArrayList<>();
            for (int i = 0; i < groups.size(); i++) {
                bunch.add(groups.get(i));
                if (bunch.size() == 100 || i == groups.size()-1) {
                    groupsClient.removeGroups(bunch).get();
                    bunch.clear();
                    LOG.info("Successful removed group up to {}", i);
                }
            }
        } catch (InterruptedException | ExecutionException e) {
            LOG.error("YP error while removing group {}: {}", groups, e);
        }
    }
}
