package ru.yandex.infra.auth.servlets;

import java.io.IOException;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Charsets;
import com.google.common.net.MediaType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.idm.service.IdmService;

public class IdmServlet extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(IdmServlet.class);

    protected final IdmService service;
    protected final ObjectMapper jsonMapper = new ObjectMapper();
    protected final IdmService.RequestType requestType;
    private final IdmAuthorization authorizationStrategy;

    public IdmServlet(IdmService idmService, IdmAuthorization authorizationStrategy, IdmService.RequestType requestType) {
        this.service = idmService;
        this.authorizationStrategy = authorizationStrategy;
        this.requestType = requestType;
    }

    @FunctionalInterface
    private interface Function<T, R> {
        R apply(T t) throws ServletException, IOException;
    }

    public static void logRequest(HttpServletRequest req) {
        String idmRequestId = req.getHeader("X-IDM-Request-Id");
        LOG.info("Received {} {} {}{}", req.getMethod(), req.getServletPath(), req.getParameterMap(),
                idmRequestId != null ? " X-IDM-Request-Id: " + idmRequestId : "");
    }

    private void processRequest(HttpServletRequest req,
                                HttpServletResponse resp,
                                Function<HttpServletRequest, String> requestHandler) throws ServletException, IOException {
        final long start = System.currentTimeMillis();
        try {
            service.getRequestsMetric(requestType).incrementAndGet();
            logRequest(req);

            resp.setCharacterEncoding(Charsets.UTF_8.name());
            resp.setContentType(MediaType.JSON_UTF_8.toString());

            if (authorizationStrategy.checkAuthorization(req, resp)) {
                String response = requestHandler.apply(req);
                if (response != null && !response.isEmpty()) {
                    resp.getWriter().print(response);
                }
            }

        } finally {
            LOG.info("Processing {} {} request took: {} ms", req.getMethod(), req.getServletPath(), System.currentTimeMillis() - start);
        }
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        processRequest(req, resp, this::doPostInternal);
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        processRequest(req, resp, this::doGetInternal);
    }

    protected String doPostInternal(HttpServletRequest req) throws IOException {
        return "";
    }

    protected String doGetInternal(HttpServletRequest req) throws IOException {
        return "";
    }
}
