package ru.yandex.infra.auth.servlets;

import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RoleSubjectsProvider;
import ru.yandex.infra.auth.yp.YpObjectsTreeGetterError;
import ru.yandex.infra.auth.yp.YpService;
import ru.yandex.infra.auth.yp.YpServiceException;
import ru.yandex.infra.auth.yp.YpServiceReadOnlyImpl;

import static java.lang.String.format;
import static ru.yandex.infra.auth.Role.ROLE_NAME_DELIMITER;

public class PrivateApiServlet extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(PrivateApiServlet.class);
    private static final String ROLE_SUBJECT_DELIMITER = ":";
    private static final String REQUEST_PARAMETER = "id";
    private static final String REQUEST_UUID = "uuid";

    private final YpService ypService;
    private final PrivateOperation operation;
    private final RoleSubjectsProvider roleSubjectsProvider;

    public PrivateApiServlet(YpService ypService, PrivateOperation operation) {
        this(ypService, operation, null);
    }

    public PrivateApiServlet(YpService ypService, PrivateOperation operation, RoleSubjectsProvider roleSubjectsProvider) {
        this.ypService = ypService;
        this.operation = operation;
        this.roleSubjectsProvider = roleSubjectsProvider;
    }

    @Override
    public void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        final long start = System.currentTimeMillis();
        String id = req.getParameter(REQUEST_PARAMETER);
        if (id == null) {
            id = "";
        }

        String response = "";
        switch (operation) {
            case LIST_ROLE_NODES:
                response = doListRoleNodes(id);
                break;
            case LIST_ROLE_SUBJECTS:
                response = doListRoleSubjects(id);
                break;
            case LIST_PROJECTS_WITHOUT_OWNER:
                response = doListProjectsWithoutOwner();
                break;
            default:
                response = format("Unexpected GET operation %s\n", operation);
        }

        resp.getWriter().print(response);
        LOG.info("Processing GET {} request took: {} ms", operation, System.currentTimeMillis() - start);
    }

    private String doListProjectsWithoutOwner() {
        return String.join("\n", roleSubjectsProvider.getProjectsWithoutOwner());
    }

    @Override
    public void doPost(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        final long start = System.currentTimeMillis();
        String id = getRequestParameterOrEmpty(req, REQUEST_PARAMETER);
        String uuid = getRequestParameterOrEmpty(req, REQUEST_UUID);

        String response = "";
        switch (operation) {
            case CLEANUP_NODES:
                response = doCleanupNode();
                break;
            case REMOVE_ROLE_SUBJECT:
                response = doRemoveRoleSubject(id, uuid);
                break;
            case UPDATE_ROLE_ACLS:
                response = doUpdateRoleAcls(id);
                break;
            default:
                response = format("Unexpected POST operation %s\n", operation);
        }

        resp.getWriter().print(response);
        LOG.info("Processing POST {} request took: {} ms", operation, System.currentTimeMillis() - start);
    }

    private String getRequestParameterOrEmpty(HttpServletRequest request, String parameterName) {
        String value = request.getParameter(parameterName);
        if (value == null) {
            return "";
        }
        return value;
    }

    private String doCleanupNode() {
        ypService.setGlobalCleanupFlag();
        return "Done\n";
    }

    private String doUpdateRoleAcls(String projectId) {
        StringBuilder response = new StringBuilder();
        StringBuilder failedRoleSubjects = new StringBuilder();
        response.append(format("Updated ACL for %s project\n", projectId.isEmpty() ? "ALL" : projectId));

        getRoleSubjects(projectId).forEach(subject -> {
            try {
                ypService.updateRoleSubject(subject);
                response.append(subject);
                response.append("\n");
            } catch (YpServiceException e) {
                LOG.error("Failed to update acls for subject {}: {}", subject, e);
                failedRoleSubjects.append(subject);
                failedRoleSubjects.append("\n");
            }

        });

        response.append("\nFAILED ROLE SUBJECTS:\n");
        response.append(failedRoleSubjects);
        return response.toString();
    }

    private String doListRoleNodes(String projectId) {
        String response;
        try {
            SortedSet<Role> sortedRoleNodes = ypService.getRoles().stream()
                    .filter(role -> projectId.isEmpty() || role.getLevelName(0).orElse("").equals(projectId))
                    .collect(Collectors.toCollection(TreeSet::new));

            StringBuilder builder = new StringBuilder();
            sortedRoleNodes.forEach(role -> builder.append(format("%s %s\n", role.getLevelsJoinedWithDelimiter(), role.getCreator())));
            response = builder.toString();
        } catch (YpObjectsTreeGetterError e) {
            LOG.error("Failed to get list of role nodes:", e);
            response = "Failed\n";
        }
        return response;
    }

    private Set<RoleSubject> getRoleSubjects(String projectId) {
        if (projectId.isEmpty()) {
            return ypService.getRoleSubjects();
        }
        Map<Role, Set<String>> roleMembers = roleSubjectsProvider.getRoleMembers(projectId);
        return YpServiceReadOnlyImpl.getRoleSubjectsFromRoleMembers(roleMembers);
    }

    private String doListRoleSubjects(String projectId) {
        Set<RoleSubject> roleSubjects = getRoleSubjects(projectId);
        SortedSet<RoleSubject> sortedRoleNodes = new TreeSet<>(roleSubjects);

        StringBuilder builder = new StringBuilder();
        sortedRoleNodes.forEach(subject -> builder.append(
                format("%s:%s\n",
                        subject.getRole().getLevelsJoinedWithDelimiter(),
                        subject.isPersonal() ? subject.getLogin() : subject.getGroupId()
                ))
        );
        return builder.toString();
    }

    private String doRemoveRoleSubject(String subjectId, String uuid) {
        String[] chunks = subjectId.split(ROLE_SUBJECT_DELIMITER);
        if (chunks.length != 2) {
            return format("Invalid role subject: %s\n", subjectId);
        }

        String[] path = chunks[0].split("\\" + ROLE_NAME_DELIMITER);
        if (path.length == 0) {
            return format("Invalid role subject: %s\n", subjectId);
        }

        String leaf = path[path.length - 1];
        Role role = new Role(chunks[0].substring(0, chunks[0].length() - leaf.length()), leaf, "", uuid);
        RoleSubject roleSubject;
        try {
            roleSubject = new RoleSubject(
                    "",
                    Long.parseLong(chunks[1]),
                    role
            );
        } catch (NumberFormatException e) {
            roleSubject = new RoleSubject(
                    chunks[1],
                    0L,
                    role
            );
        }

        try {
            ypService.removeRoleSubject(roleSubject);
            return "Done\n";
        } catch (YpServiceException ex) {
            return format("Failed for %s : %s\n", roleSubject, ex.getMessage());
        }
    }

    public enum PrivateOperation {
        CLEANUP_NODES,
        LIST_ROLE_NODES,
        LIST_ROLE_SUBJECTS,
        REMOVE_ROLE_SUBJECT,
        UPDATE_ROLE_ACLS,
        LIST_PROJECTS_WITHOUT_OWNER,
    }
}
