package ru.yandex.infra.auth.servlets;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RolesInfo;

import static ru.yandex.infra.auth.yp.YpGroupsHelper.IDM_GROUP_PREFIX;

class ProjectAcl {
    private final Map<String, Map<String, List<String>>> projectAcl;
    private final Map<String, Map<String, List<String>>> stageAcl;
    private final Map<String, Map<String, List<String>>> nannyServiceAcl;
    private final Map<String, Map<String, Map<String, List<String>>>> boxTypeAcl;

    private ProjectAcl(Map<String, Map<String, List<String>>> projectAcl,
            Map<String, Map<String, List<String>>> stageAcl,
            Map<String, Map<String, List<String>>> nannyServiceAcl,
            Map<String, Map<String, Map<String, List<String>>>> boxTypeAcl) {
        this.projectAcl = projectAcl;
        this.stageAcl = stageAcl;
        this.nannyServiceAcl = nannyServiceAcl;
        this.boxTypeAcl = boxTypeAcl;
    }

    @JsonProperty("project_acl")
    public Map<String, Map<String, List<String>>> getProjectAcl() {
        return projectAcl;
    }

    @JsonProperty("stage_acl")
    public Map<String, Map<String, List<String>>> getStageAcl() {
        return stageAcl;
    }

    @JsonProperty("nanny_service_acl")
    public Map<String, Map<String, List<String>>> getNannyServiceAcl() {
        return nannyServiceAcl;
    }

    @JsonProperty("box_type_acl")
    public Map<String, Map<String, Map<String, List<String>>>> getBoxTypeAcl() {
        return boxTypeAcl;
    }

    enum Level {
        ROOT,
        PROJECT,
        STAGE,
        BOX
    }

    static class Builder {
        private final Map<String, Map<String, List<String>>> projectAcl = new HashMap<>();
        private final Map<String, Map<String, List<String>>> stageAcl = new HashMap<>();
        private final Map<String, Map<String, List<String>>> nannyServiceAcl = new HashMap<>();
        private final Map<String, Map<String, Map<String, List<String>>>> boxTypeAcl = new HashMap<>();
        private final RolesInfo rolesInfo;

        Builder(RolesInfo rolesInfo) {
            this.rolesInfo = rolesInfo;
        }

        Builder addDefaultProjectRoles(String projectId) {
            projectAcl.computeIfAbsent(projectId, ignored -> createAclMap(Level.PROJECT.ordinal()));
            return this;
        }

        Builder addAcl(Role role, String member) {
            if (member.startsWith(IDM_GROUP_PREFIX)) {
                member = member.replace(IDM_GROUP_PREFIX, "group:");
            }
            if (role.size() - 1 >= Level.values().length) {
                // ignore unexpected role format
                return this;
            }
            ProjectAcl.Level level = ProjectAcl.Level.values()[role.size() - 1];
            String objectName = role.getLevelName(role.size() - 2).orElse("");
            String parentObjectName = role.getLevelName(role.size() - 3).orElse("");
            String roleName = role.getLeaf().toLowerCase();

            if (role.isNannyRole()) {
                nannyServiceAcl.computeIfAbsent(objectName, ignored -> createNannyServiceAclMap())
                        .computeIfAbsent(roleName, ignored -> new ArrayList<>())
                        .add(member);
                return this;
            }

            switch (level) {
                case PROJECT:
                    if (!objectName.isEmpty()) {
                        addAcl(projectAcl, Level.PROJECT.ordinal(), objectName, roleName, member);
                    }
                    break;

                case STAGE:
                    addAcl(stageAcl, Level.STAGE.ordinal(), objectName, roleName, member);
                    break;

                case BOX:
                    Map<String, Map<String, List<String>>> acl = boxTypeAcl.computeIfAbsent(
                            parentObjectName,
                            ignored -> new HashMap<>()
                    );
                    addAcl(acl, Level.BOX.ordinal(), objectName, roleName, member);
                    break;

                default:
                    break;
            }
            return this;
        }

        ProjectAcl build() {
            return new ProjectAcl(projectAcl, stageAcl, nannyServiceAcl, boxTypeAcl);
        }

        private void addAcl(
                Map<String, Map<String, List<String>>> acl,
                int level,
                String object,
                String role,
                String subject
        ) {
            List<String> subjects = acl.computeIfAbsent(object, ignored -> createAclMap(level))
                    .get(role);
            if (subjects != null) {
                subjects.add(subject);
            }
        }

        private Map<String, List<String>> createAclMap(int level) {
            List<String> roles = rolesInfo.getVisibleRolesPerLevel(level);
            return createAclMap(roles);
        }

        private Map<String, List<String>> createNannyServiceAclMap() {
            Set<String> nannyServiceRoles = rolesInfo.getRolesPerLevel(RolesInfo.LevelName.NANNY_SERVICE);
            return createAclMap(nannyServiceRoles);
        }

        private Map<String, List<String>> createAclMap(Collection<String> roles) {
            Map<String, List<String>> acls = new LinkedHashMap<>();
            for (String r : roles) {
                acls.put(r.toLowerCase(), new ArrayList<>());
            }
            return acls;
        }
    }
}
