package ru.yandex.infra.auth.staff;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import com.typesafe.config.Config;
import org.apache.http.HttpStatus;
import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.utils.ApiClientWithAuthorization;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.util.ConfigUtils;

public class StaffApiImpl extends ApiClientWithAuthorization implements StaffApi {
    private static final Logger LOG = LoggerFactory.getLogger(StaffApiImpl.class);

    private final int limit;

    public StaffApiImpl(AsyncHttpClient httpClient, String host, String token, int groupsRequestLimit, GaugeRegistry gaugeRegistry) {
        super(httpClient, host, token, gaugeRegistry);
        limit = groupsRequestLimit;
    }

    public StaffApiImpl(Config config, AsyncHttpClient httpClient, GaugeRegistry gaugeRegistry) {
        this(httpClient,
                config.getString("host"),
                ConfigUtils.token(config.getString("token_file")),
                config.getInt("groups_request_limit"),
                gaugeRegistry);
    }

    enum GroupType {
        SERVICE("service"),
        SERVICE_ROLE("servicerole"),
        DEPARTMENT("department"),
        WIKI("wiki");

        public String name;

        GroupType(String name) {
            this.name = name;
        }
    }

    @Override
    public CompletableFuture<Long> getStaffGroupId(long abcServiceId) {

        String url = String.format("%s/v3/groups?service.id=%d&is_deleted=false&type=service", host, abcServiceId);

        return runWithAuthorization(httpClient.prepareGet(url), response -> {
            if (response.getStatusCode() != HttpStatus.SC_OK) {
                throw new RuntimeException(String.format("Staff API error: [%d] %s",
                        response.getStatusCode(),
                        response.getStatusText()));
            }

            try {
                GetGroupsResponse getGroupsResponse = mapper.readValue(response.getResponseBody(), GetGroupsResponse.class);
                if (getGroupsResponse.getTotal() == 0) {
                    LOG.error("Did not find any staff group");
                    return null;
                } else if (getGroupsResponse.getTotal() != 1) {
                    LOG.error("Found more than one staff group");
                    return null;
                }
                return getGroupsResponse.getResults().get(0).getId();
            } catch (IOException e) {
                LOG.error("Error while parsing response from staff", e);
                return null;
            }
        });
    }

    @Override
    public CompletableFuture<List<StaffGroup>> getAllDepartmentGroups() {
        return getAllGroups(GroupType.DEPARTMENT, "id,url,department.id");
    }

    @Override
    public CompletableFuture<List<StaffGroup>> getAllAbcServiceGroups() {
        return getAllGroups(GroupType.SERVICE, "id,service.id");
    }

    @Override
    public CompletableFuture<List<StaffGroup>> getAllAbcServiceRoleGroups() {
        return getAllGroups(GroupType.SERVICE_ROLE, "id,role_scope,parent.service.id");
    }

    private CompletableFuture<List<StaffGroup>> getAllGroups(GroupType groupType, String fields) {

        String url = String.format("%s/v3/groups?_limit=%d&type=%s&is_deleted=false&_fields=%s",
                host, limit, groupType.name, fields);

        return runWithAuthorization(httpClient.prepareGet(url), response -> {
            if (response.getStatusCode() != HttpStatus.SC_OK) {
                throw new RuntimeException(String.format("Staff API error: [%d] %s",
                        response.getStatusCode(),
                        response.getStatusText()));
            }

            try {
                GetGroupsResponse getGroupsResponse = mapper.readValue(response.getResponseBody(), GetGroupsResponse.class);
                return getGroupsResponse.getResults();
            } catch (IOException e) {
                throw new RuntimeException("Error while parsing response from Staff API", e);
            }
        });
    }
}
