package ru.yandex.infra.auth.tasks;

import java.time.Duration;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;

import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubjectsProvider;
import ru.yandex.infra.auth.yp.YpService;
import ru.yandex.infra.controller.RepeatedTask;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;

import static org.slf4j.LoggerFactory.getLogger;

public class RoleSubjectCacheUpdater implements RoleSubjectsProvider {

    private static final Logger LOG = getLogger(RoleSubjectCacheUpdater.class);

    static final String METRIC_TOTAL_ROLE_SUBJECTS = "total_roles_subjects";
    static final String METRIC_STAGES_COUNT = "stages_count";

    static final String METRIC_ACTIVE_ROLES = "active";
    static final String METRIC_PROJECTS_WITH_MISSED_OWNER = "projects_with_missed_owner";


    private Long metricTotalRoleSubjects;
    private Long metricStagesCount;

    private Long metricActiveRoles;
    private Integer metricProjectsWithMissedOwner;

    private final YpService ypService;
    private final ScheduledExecutorService executor;
    private final Duration roleSubjectsCacheUpdateRate;
    private final RepeatedTask mainLoop;

    private Map<String, Map<Role, Set<String>>> roleMembersByProjectId = new HashMap<>();

    public RoleSubjectCacheUpdater(YpService ypService,
                                   Duration roleSubjectCacheUpdateRate,
                                   Duration mainLoopTimeout,
                                   GaugeRegistry gaugeRegistry) {
        this.ypService = ypService;
        this.roleSubjectsCacheUpdateRate = roleSubjectCacheUpdateRate;
        this.executor = Executors.newSingleThreadScheduledExecutor(runnable -> new Thread(runnable, "roles_cache"));

        gaugeRegistry.add(METRIC_TOTAL_ROLE_SUBJECTS, new GolovanableGauge<>(() -> metricTotalRoleSubjects, "axxx"));
        gaugeRegistry.add(METRIC_STAGES_COUNT, new GolovanableGauge<>(() -> metricStagesCount, "axxx"));

        GaugeRegistry rolesRegistry = new NamespacedGaugeRegistry(gaugeRegistry, "roles");
        rolesRegistry.add(METRIC_ACTIVE_ROLES, new GolovanableGauge<>(() -> metricActiveRoles, "axxx"));
        rolesRegistry.add(METRIC_PROJECTS_WITH_MISSED_OWNER, new GolovanableGauge<>(() -> metricProjectsWithMissedOwner, "axxx"));

        mainLoop = new RepeatedTask(this::updateRoleSubjectCache,
                roleSubjectCacheUpdateRate,
                mainLoopTimeout,
                executor,
                Optional.of(rolesRegistry),
                LOG,
                false);
    }

    public void shutdown() {
        mainLoop.stop();
        executor.shutdown();
    }

    public void start() {
        if (roleSubjectsCacheUpdateRate.isZero()) {
            LOG.warn("RolesSubject cache is disabled by config option 'main.role_subject_cache_rate' = 0");
        } else {
            mainLoop.start();
        }
    }

    @VisibleForTesting
    public void updateRoleSubjectCache() {
        try {
            Map<Role, Set<String>> allRolesWithMembers = ypService.getRoleMembers();

            roleMembersByProjectId = allRolesWithMembers
                    .entrySet()
                    .stream()
                    .filter(entry -> entry.getKey().getProjectId().isPresent())
                    .collect(Collectors.groupingBy(entry -> entry.getKey().getProjectId().get(),
                            Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue)));

            metricStagesCount = allRolesWithMembers.keySet().stream()
                    .map(role -> role.getLevelName(1).orElse(null))
                    .filter(Objects::nonNull)
                    .distinct()
                    .count();

            Collection<Set<String>> allMembers = allRolesWithMembers.values();
            metricActiveRoles = allMembers.stream().filter(members -> !members.isEmpty()).count();
            metricTotalRoleSubjects = allMembers.stream().mapToLong(Set::size).sum();
            metricProjectsWithMissedOwner = getProjectsWithoutOwner().size();

        } catch (Exception exception) {
            throw new RuntimeException("Roles cache update failed", exception);
        }
    }

    @Override
    public Map<Role, Set<String>> getRoleMembers(String projectId) {
        return roleMembersByProjectId.get(projectId);
    }

    @Override
    public List<String> getProjectsWithoutOwner() {
        return roleMembersByProjectId.entrySet().stream()
                .filter(entry -> entry.getValue()
                        .entrySet()
                        .stream()
                        .noneMatch(roleMembers ->
                                "OWNER".equals(roleMembers.getKey().getLeaf()) && //Role is Project.OWNER
                                        !roleMembers.getValue().isEmpty())) //List of members not empty
                .map(Map.Entry::getKey)
                .collect(Collectors.toList());
    }
}
