package ru.yandex.infra.auth.tasks;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.time.Duration;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import ru.yandex.infra.auth.tvm.TvmTicketParser;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.passport.tvmauth.CheckedServiceTicket;
import ru.yandex.passport.tvmauth.Version;
import ru.yandex.passport.tvmauth.deprecated.ServiceContext;


public class TvmTicketUpdater implements TvmTicketParser {
    private static final Logger LOG = LoggerFactory.getLogger(TvmTicketUpdater.class);
    private static final String TVM_API_URL = "https://tvm-api.yandex.net/2/keys/?lib_version=" + Version.get();
    private static final String WATCHDOG_ID = "tvmUpdater";
    private static final int RESPONSE_BUFFER_CAPACITY = 512;

    private final int clientId;
    private final int sourceId;
    private final String secret;
    private final Watchdog watchdog;
    private volatile ServiceContext serviceContext;

    public TvmTicketUpdater(int clientId,
            int sourceId,
            String secret,
            ScheduledExecutorService executor,
            Duration tvmTicketUpdateRate,
            Duration tvmWatchdogTimeout,
            Watchdog watchdog)
    {
        LOG.info("Client ID: {}", clientId);
        this.clientId = clientId;
        this.sourceId = sourceId;
        this.secret = secret;
        this.watchdog = watchdog;

        watchdog.register(WATCHDOG_ID, tvmWatchdogTimeout.toMillis());

        // first initialization
        update();

        // start update loop
        executor.scheduleAtFixedRate(
                this::update,
                tvmTicketUpdateRate.toSeconds(), // first time we update tickets at construction time
                tvmTicketUpdateRate.toSeconds(),
                TimeUnit.SECONDS
        );
    }

    @Override
    public boolean parseTicket(String ticket) {
        CheckedServiceTicket serviceTicket = serviceContext.check(ticket);
        if (!serviceTicket.booleanValue()) {
            LOG.error("Invalid ticket:\nError: {}", serviceTicket.getStatus().toString());
            return false;
        }

        if (serviceTicket.getSrc() != sourceId) {
            LOG.error("Unexpected source ID: {}, [{}]", serviceTicket.getSrc(), sourceId);
            return false;
        }

        return true;
    }

    private void update() {
        LOG.info("Start updating TVM tickets ...");
        watchdog.touch(WATCHDOG_ID);
        serviceContext = new ServiceContext(clientId, secret, getTvmKeys());
        LOG.info("Updating TVM tickets - DONE");
    }

    private String getTvmKeys() {
        try {
            URL tvm = new URL(TVM_API_URL);
            URLConnection tvmConnection = tvm.openConnection();
            BufferedReader in = new BufferedReader(new InputStreamReader(tvmConnection.getInputStream()));
            String inputLine;
            StringBuilder result = new StringBuilder(RESPONSE_BUFFER_CAPACITY);

            while ((inputLine = in.readLine()) != null) {
                result.append(inputLine);
            }
            in.close();
            return result.toString();
        } catch (Exception ex) {
            LOG.error("getTvmKeys failed:", ex);
            return "";
        }
    }
}
