package ru.yandex.infra.auth.tasks;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.typesafe.config.Config;
import org.slf4j.Logger;

import ru.yandex.infra.controller.util.ExitUtils;

import static org.slf4j.LoggerFactory.getLogger;

public class Watchdog {
    private static final Logger LOG = getLogger(Watchdog.class);

    private final Map<String, WatchdogEntry> entries = new HashMap<>();
    private final ScheduledExecutorService executor = new ScheduledThreadPoolExecutor(1);

    private static class WatchdogEntry {
        public long timeout;
        public long lastTimeStamp;

        WatchdogEntry(long timeout) {
            this.timeout = timeout;
            this.lastTimeStamp = 0;
        }

        boolean isSuspended() {
            return lastTimeStamp == 0;
        }

        boolean isExpired() {
            return System.currentTimeMillis() - lastTimeStamp > timeout;
        }

        void touch() {
            lastTimeStamp = System.currentTimeMillis();
        }

        void suspend() {
            lastTimeStamp = 0;
        }
    }

    public Watchdog(Duration checkInterval) {
        long checkRate = checkInterval.toSeconds();
        // start update loop
        LOG.info("Watchdog started at fixed rate: {}s", checkRate);
        executor.scheduleAtFixedRate(
                this::check,
                checkRate, // first time we update tickets at construction time
                checkRate,
                TimeUnit.SECONDS
        );
    }

    public Watchdog(Config config) {
        this(config.getDuration("check_rate"));
    }

    public void shutdown() {
        executor.shutdown();
    }

    synchronized public void register(String id, long timeout) {
        LOG.info("Registered new watchdog entry: {} {}", id, timeout);
        entries.put(id, new WatchdogEntry(timeout));
    }

    synchronized public void touch(String id) {
        WatchdogEntry entry = entries.get(id);
        if (entry != null) {
            if (entry.isSuspended()) {
                LOG.info("Resume watchdog: {}", id);
            } else {
                LOG.debug("Touch watchdog: {}", id);
            }
            entry.touch();
        }
    }

    synchronized public void suspend(String id) {
        WatchdogEntry entry = entries.get(id);
        if (entry != null) {
            LOG.info("Suspend watchdog: {}", id);
            entry.suspend();
        }
    }

    synchronized public void check() {
        entries.forEach((id, entry) -> {
            if (!entry.isSuspended() && entry.isExpired()) {
                LOG.error("{} watchdog has expired. Immediately shutdown !!!", id);
                ExitUtils.gracefulExit(ExitUtils.WATCHDOG_EXPIRED);
            }
        });
    }
}
