package ru.yandex.infra.auth.utils;

import java.io.IOException;
import java.time.Duration;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Function;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.HttpStatus;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.BoundRequestBuilder;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.slf4j.Logger;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.controller.FailedHttpRequestException;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;

import static java.lang.String.format;
import static org.slf4j.LoggerFactory.getLogger;

public class ApiClientWithAuthorization {
    private static final Logger LOG = getLogger(ApiClientWithAuthorization.class);

    static final String METRIC_API_REQUESTS = "api_requests";
    static final String METRIC_FAILED_API_REQUESTS = "failed_api_requests";

    private final AtomicLong metricApiRequests = new AtomicLong();
    private final AtomicLong metricFailedApiRequests = new AtomicLong();

    protected final AsyncHttpClient httpClient;
    protected final String host;
    protected final String token;
    protected final ObjectMapper mapper = new ObjectMapper();

    public ApiClientWithAuthorization(AsyncHttpClient httpClient, String host, String token, GaugeRegistry gaugeRegistry) {
        this.httpClient = httpClient;
        this.host = host;
        this.token = token;

        gaugeRegistry.add(METRIC_API_REQUESTS, new GolovanableGauge<>(metricApiRequests::get, "dmmm"));
        gaugeRegistry.add(METRIC_FAILED_API_REQUESTS, new GolovanableGauge<>(metricFailedApiRequests::get, "dmmm"));
    }

    protected <T> CompletableFuture<T> executeRequest(String url, Function<JsonNode, T> mapResult) {

        return runWithAuthorization(httpClient.prepareGet(url), response -> {
            try {
                if (response.getStatusCode() != HttpStatus.SC_OK) {
                    String message = String.format("Received error response: [%d] %s\nRequest url: %s",
                            response.getStatusCode(),
                            response.getStatusText(),
                            url);
                    throw new FailedHttpRequestException(response.getStatusCode(), message);
                }

                JsonNode root = mapper.readTree(response.getResponseBodyAsBytes());
                return mapResult.apply(root);
            } catch (IOException e) {
                throw new RuntimeException("Error while parsing response for " + url, e);
            }
        });
    }

    public static <T> CompletableFuture<T> executeRequestWithRetries(Supplier<CompletableFuture<T>> requestSupplier,
                                                                     int maxAttemptsCount,
                                                                     Duration retryDelay,
                                                                     Set<Integer> statusCodesToStopRetries) {

        return requestSupplier.get()
                .thenApply(result -> Tuple2.tuple(result, (Throwable)null))
                .exceptionally(error -> Tuple2.tuple(null, error))
                .thenCompose(t -> {
                    if (t._1 != null) {
                        return CompletableFuture.completedFuture(t._1);
                    }

                    Throwable error = t._2.getCause();
                    if (maxAttemptsCount == 1 ||
                            (error instanceof FailedHttpRequestException && statusCodesToStopRetries.contains(((FailedHttpRequestException)error).getResponseCode()))) {
                        return CompletableFuture.failedFuture(t._2);
                    }

                    LOG.warn("Request failed: {}\n{} attempts left. Sleeping {} before next attempt.", error.getMessage(),  maxAttemptsCount - 1, retryDelay);

                    Executor delayed = CompletableFuture.delayedExecutor(retryDelay.toNanos(), TimeUnit.NANOSECONDS);
                    return CompletableFuture.completedFuture(null)
                            .thenComposeAsync(x -> executeRequestWithRetries(requestSupplier, maxAttemptsCount - 1, retryDelay, statusCodesToStopRetries), delayed);
                });
    }

    protected <FutureType> CompletableFuture<FutureType> runWithAuthorization(BoundRequestBuilder requestBuilder,
                                                                              Function<Response, FutureType> core) {
        metricApiRequests.incrementAndGet();

        Request request = requestBuilder
                .addHeader("Authorization", format("OAuth %s", token))
                .build();

        LOG.info("Sending {} request: {}", request.getMethod(), request.getUrl());

        return httpClient
                .executeRequest(request)
                .toCompletableFuture()
                .thenApply(core)
                .whenComplete((ignore, error) -> {
                    if (error != null) {
                        metricFailedApiRequests.incrementAndGet();
                    }
                });
    }

}
