package ru.yandex.infra.auth.utils;

import java.util.Collections;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.typesafe.config.Config;
import one.util.streamex.StreamEx;

import ru.yandex.infra.controller.dto.NannyServiceMeta;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.util.YpUtils;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.ObjectBuilderDescriptor;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.client.api.Autogen.TNannyServiceMeta;
import ru.yandex.yp.client.api.Autogen.TProjectMeta;
import ru.yandex.yp.client.api.Autogen.TSchemaMeta;
import ru.yandex.yp.client.api.Autogen.TStageMeta;
import ru.yandex.yp.client.api.DataModel.TGroupSpec;
import ru.yandex.yp.client.api.DataModel.TGroupStatus;
import ru.yandex.yp.client.api.DataModel.TUserSpec;
import ru.yandex.yp.client.api.DataModel.TUserStatus;
import ru.yandex.yp.client.api.TNannyServiceSpec;
import ru.yandex.yp.client.api.TNannyServiceStatus;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

public final class ConfigUtils {
    private static final ObjectBuilderDescriptor<TStageMeta, StageMeta> STAGE_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(TStageSpec::newBuilder, TStageStatus::newBuilder, StageMeta::fromProto,
                    TStageMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<TNannyServiceMeta, NannyServiceMeta> NANNY_SERVICE_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(TNannyServiceSpec::newBuilder, TNannyServiceStatus::newBuilder, NannyServiceMeta::fromProto,
                    TNannyServiceMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<TSchemaMeta, SchemaMeta> GROUP_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(TGroupSpec::newBuilder, TGroupStatus::newBuilder, SchemaMeta::fromProto,
                    TSchemaMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<TProjectMeta, ProjectMeta> PROJECT_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(TProjectSpec::newBuilder, TProjectStatus::newBuilder, ProjectMeta::fromProto,
                    TProjectMeta.getDefaultInstance());
    private static final ObjectBuilderDescriptor<TSchemaMeta, SchemaMeta> USER_DESCRIPTOR =
            new ObjectBuilderDescriptor<>(TUserSpec::newBuilder, TUserStatus::newBuilder, SchemaMeta::fromProto,
                    TSchemaMeta.getDefaultInstance());

    private ConfigUtils() {
    }

    public static Map<String, YpRawObjectService> ypClientsPerCluster(Config ypConfig,
                                                                      Config ypClientsConfig,
                                                                      GaugeRegistry gaugeRegistry,
                                                                      boolean isReadonlyMode) {
        return extractMap(ypClientsConfig, Config::getConfig).entrySet().stream().collect(
                Collectors.toMap(Map.Entry::getKey,
                        entry -> YpUtils.ypRawClient(entry.getValue(), ypConfig, gaugeRegistry, isReadonlyMode)));
    }

    private static <T> Map<String, T> extractMap(Config config, BiFunction<Config, String, T> extractor) {
        return StreamEx.of(config.root().keySet())
                .toMap(key -> extractor.apply(config, key));
    }

    public static LabelBasedRepository<StageMeta, TStageSpec, TStageStatus> ypStageRepository(
            YpRawObjectService objectService, int selectPageSize, int watchPageSize, Config stageLabelsConfig,
            GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.STAGE, getLabels(stageLabelsConfig), Optional.empty(),
                objectService, STAGE_DESCRIPTOR, selectPageSize, watchPageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<NannyServiceMeta, TNannyServiceSpec, TNannyServiceStatus> ypNannyServiceRepository(
            YpRawObjectService objectService, int selectPageSize, int watchPageSize, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.NANNY_SERVICE, Collections.emptyMap(), Optional.empty(),
                objectService, NANNY_SERVICE_DESCRIPTOR, selectPageSize, watchPageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository(
            YpRawObjectService objectService, int pageSize, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.GROUP, Collections.emptyMap(), Optional.empty(),
                objectService, GROUP_DESCRIPTOR, pageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository(
            YpRawObjectService objectService, int pageSize, GaugeRegistry gaugeRegistry,
            Map<String, String> allOperationsLabelsMap) {
        return new LabelBasedRepository<>(YpObjectType.GROUP, allOperationsLabelsMap, Optional.empty(),
                objectService, GROUP_DESCRIPTOR, pageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypGroupRepository(
            YpRawObjectService objectService, int pageSize, GaugeRegistry gaugeRegistry, String cluster) {
        return new LabelBasedRepository<>(YpObjectType.GROUP, Collections.emptyMap(), Optional.empty(),
                objectService, GROUP_DESCRIPTOR, pageSize, gaugeRegistry, cluster);
    }

    public static LabelBasedRepository<SchemaMeta, TUserSpec, TUserStatus> ypUserRepository(
            YpRawObjectService objectService, int pageSize, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.USER, Collections.emptyMap(), Optional.empty(),
                objectService, USER_DESCRIPTOR, pageSize, gaugeRegistry);
    }

    public static LabelBasedRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectRepository(
            YpRawObjectService objectService, int selectPageSize, int watchPageSize, GaugeRegistry gaugeRegistry) {
        return new LabelBasedRepository<>(YpObjectType.PROJECT, Collections.emptyMap(), Optional.empty(),
                objectService, PROJECT_DESCRIPTOR, selectPageSize, watchPageSize, gaugeRegistry);
    }

    public static Map<String, String> getLabels(Config config) {
        return StreamEx.of(config.root().keySet()).toMap(config::getString);
    }
}

