package ru.yandex.infra.auth.yp;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.infra.auth.Metrics;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.qe.telemetry.metrics.Gauges;
import ru.yandex.yp.model.YpTransaction;

import static java.lang.String.format;
import static java.util.concurrent.Executors.newFixedThreadPool;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.doAddMembersToGroup;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.doRemoveMembersFromGroup;

public class YpGroupsAsyncQueue {
    private static final Logger LOG = LoggerFactory.getLogger(YpGroupsAsyncQueue.class);
    public static final String TOTAL_YP_ASYNC_QUEUE = "yp_async_queue_size";
    private static final int QUEUE_CAPACITY = 1000000;

    private final List<Worker> workers;
    private final ExecutorService executorService;
    private final List<Future<?>> workerFutures;
    private final Map<String, BlockingQueue<Job>> queues;

    private static class Job {
        private final static int MAX_RETRIES = 10000;
        protected final String groupId;
        protected final Set<String> logins;

        private int retryCounter = 0;

        public Job(String groupId, Set<String> logins) {
            this.groupId = groupId;
            this.logins = logins;
        }

        public CompletableFuture<?> createFuture(YpGroupsClient groupsClient, YpTransaction transaction,
                String cluster) {
            throw new NotImplementedException();
        }

        public String getGroupId() {
            return groupId;
        }

        public int getRetryCounter() {
            return retryCounter;
        }

        public boolean shouldRetry() {
            retryCounter++;
            return retryCounter < MAX_RETRIES;
        }
    }

    private static class AddJob extends Job {
        public AddJob(String groupId, Set<String> logins) {
            super(groupId, logins);
        }

        public CompletableFuture<?> createFuture(YpGroupsClient groupsClient, YpTransaction transaction,
                String cluster) {
            return doAddMembersToGroup(groupsClient, groupId, logins, transaction, cluster, false);
        }
    }

    private static class RemoveJob extends Job {
        public RemoveJob(String groupId, Set<String> logins) {
            super(groupId, logins);
        }

        public CompletableFuture<?> createFuture(YpGroupsClient groupsClient, YpTransaction transaction,
                String cluster) {
            return doRemoveMembersFromGroup(groupsClient, groupId, logins, transaction, cluster);
        }
    }

    private static class SyncJob extends Job {
        public SyncJob(String groupId, Set<String> logins) {
            super(groupId, logins);
        }

        public CompletableFuture<?> createFuture(YpGroupsClient groupsClient, YpTransaction transaction,
                String cluster) {
            return doAddMembersToGroup(groupsClient, groupId, logins, transaction, cluster, true);
        }
    }

    private static class Worker extends Thread {
        private final static long RETRY_DELAY_MS = 10000L;
        private final String cluster;
        private final YpGroupsClient client;
        private final YpTransactionClient transactionClient;
        private final BlockingQueue<Job> queue;
        private final Metrics metrics;

        public Worker(String cluster,
                YpGroupsClient client,
                YpTransactionClient transactionClient,
                BlockingQueue<Job> queue,
                Metrics metrics) {
            super(cluster);
            this.cluster = cluster;
            this.client = client;
            this.transactionClient = transactionClient;
            this.queue = queue;
            this.metrics = metrics;
        }

        public void run() {
            while (true) {
                try {
                    Job job = queue.take();

                    boolean needToSleep = false;
                    try {
                        transactionClient
                                .runWithTransaction(transaction -> job.createFuture(client, transaction, cluster))
                                .get();
                    } catch (InterruptedException | ExecutionException e) {
                        metrics.addYpError();
                        LOG.error("[{}]: YP error while tried to update group {}: {}",
                                cluster, job.getGroupId(), e);
                        if (job.shouldRetry()) {
                            needToSleep = true;
                            queue.add(job);
                        } else {
                            LOG.error("[{}]: Stop retrying for {} !!!", cluster, job.getGroupId());
                        }
                    }
                    if (needToSleep) {
                        try {
                            LOG.info("[{}]: Going to sleep {} ms before next try #{}:", cluster, RETRY_DELAY_MS, job.getRetryCounter());
                            sleep(RETRY_DELAY_MS);
                        } catch (InterruptedException ex) {
                            LOG.error("[{}]: Error while trying to sleep: {}", cluster, ex.getMessage());
                        }
                    }
                } catch (Exception exception) {
                    System.err.println(format("[%s]: Exception while endless cycle: %s", cluster, exception.getMessage()));
                } catch (Throwable throwable) {
                    System.err.println(format("[%s]: Exception while endless cycle: %s", cluster, throwable.getMessage()));
                    throw throwable;
                }
            }
        }
    }

    private static class WorkersAliveChecker extends Thread {
        private final static int RETRY_DELAY_MS = 5 * 60 * 1000; // 5 min
        private final List<Worker> workers;
        private final List<Future<?>> workerFutures;
        private final ExecutorService executorService;

        WorkersAliveChecker(List<Worker> workers, List<Future<?>> workerFutures, ExecutorService executorService) {
            super("workers-alive-checker");
            this.workers = workers;
            this.workerFutures = workerFutures;
            this.executorService = executorService;
        }

        @Override
        public void run() {
            while (true) {
                for (int i = 0; i < workerFutures.size(); ++i) {
                    if (workerFutures.get(i).isDone()) {
                        workerFutures.set(i, executorService.submit(workers.get(i)));
                    }
                }
                try {
                    sleep(RETRY_DELAY_MS);
                } catch (InterruptedException e) {
                    LOG.error("Error while trying to sleep checker of threads liveness: ", e);
                }
            }
        }
    }

    public YpGroupsAsyncQueue(Map<String, YpClients> slaves,
            MetricRegistry metricRegistry,
            Metrics metrics) {

        Thread.setDefaultUncaughtExceptionHandler((thread, ex) -> ex.printStackTrace(System.err));

        queues = new HashMap<>();
        workers = slaves.entrySet().stream()
                .map(entry -> {
                    BlockingQueue<Job> queue = new ArrayBlockingQueue<>(QUEUE_CAPACITY);
                    final String cluster = entry.getKey();
                    queues.put(cluster, queue);
                    return new Worker(cluster,
                            entry.getValue().getGroupsClient(),
                            entry.getValue().getYpTransactionClient(),
                            queue, metrics);
                })
                .collect(Collectors.toList());

        executorService = newFixedThreadPool(Math.max(1, workers.size()));
        workerFutures = workers.stream().map(executorService::submit).collect(Collectors.toList());

        WorkersAliveChecker workersAliveChecker = new WorkersAliveChecker(workers, workerFutures, executorService);
        workersAliveChecker.start();

        Gauges.forSupplier(
                metricRegistry,
                TOTAL_YP_ASYNC_QUEUE,
                this::getTotalQueueSize
        );
    }

    private void addActionIntoQueueForAllSlaveClusters(String actionName,
                                                       BiFunction<String, Set<String>, ? extends Job> jobConstructorFactory,
                                                       String groupId, Set<String> logins) {
        queues.forEach((cluster, queue) -> {
            queue.add(jobConstructorFactory.apply(groupId, logins));
            LOG.info("[{}]: Job for {} members {} to/from group {} were added into queue", cluster, actionName, logins, groupId);
        });
    }

    public void addMembersToGroup(String groupId, Set<String> loginsToAdd) {
        addActionIntoQueueForAllSlaveClusters("adding", AddJob::new, groupId, loginsToAdd);
    }

    public void removeMembersFromGroup(String groupId, Set<String> loginsToRemove) {
        addActionIntoQueueForAllSlaveClusters("removing", RemoveJob::new, groupId, loginsToRemove);
    }

    public void syncMembersInGroup(String groupId, Set<String> logins) {
        addActionIntoQueueForAllSlaveClusters("syncing", SyncJob::new, groupId, logins);
    }

    public void syncMembersInGroup(String groupId, Set<String> logins, String cluster) {
        queues.get(cluster).add(new SyncJob(groupId, logins));
        LOG.info("[{}]: Job for syncing members {} in group {} were added into queue", cluster, logins, groupId);
    }

    private int getTotalQueueSize() {
        return queues.values().stream()
                .mapToInt(BlockingQueue::size)
                .sum();
    }
}
