package ru.yandex.infra.auth.yp;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.yp.CreateObjectRequest;
import ru.yandex.infra.controller.yp.Paths;
import ru.yandex.infra.controller.yp.SelectedObjects;
import ru.yandex.infra.controller.yp.UpdateYpObjectRequest;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.controller.yp.YpRequestWithPaging;
import ru.yandex.yp.client.api.AccessControl;
import ru.yandex.yp.client.api.DataModel.TGroupSpec;
import ru.yandex.yp.client.api.DataModel.TGroupStatus;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpTransaction;
import ru.yandex.yp.model.YpTypedId;

import static java.util.Collections.emptyMap;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.SYSTEM_LABEL_KEY;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC_LABELS;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC_WITH_TIMESTAMP;
import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class YpGroupsClientImpl implements YpGroupsClient {
    private final YpObjectTransactionalRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypClient;
    private final AccessControl.TAccessControlEntry defaultRobotACE;
    private final int selectAbcGroupPageSize;

    public YpGroupsClientImpl(YpObjectTransactionalRepository<SchemaMeta, TGroupSpec, TGroupStatus> ypClient,
            String robotName, int selectAbcGroupPageSize) {
        this.ypClient = ypClient;
        this.selectAbcGroupPageSize = selectAbcGroupPageSize;
        this.defaultRobotACE = AccessControl.TAccessControlEntry.newBuilder()
                .addPermissions(AccessControl.EAccessControlPermission.ACA_WRITE)
                .addSubjects(robotName)
                .setAction(AccessControl.EAccessControlAction.ACA_ALLOW)
                .build();
    }

    @Override
    public CompletableFuture<?> addGroup(String id, Set<String> members, YpTransaction transaction,
            Map<String, Object> labels) {
        CreateObjectRequest<TGroupSpec> createObjectRequest = new CreateObjectRequest.Builder<>(
                TGroupSpec.newBuilder()
                        .addAllMembers(members)
                        .build())
                .setAcl(new Acl(List.of(defaultRobotACE)))
                .setLabels(labels)
                .build();
        return ypClient.createObject(id, transaction, createObjectRequest);
    }

    @Override
    public CompletableFuture<?> removeGroup(String id, YpTransaction transaction) {
        return ypClient.removeObject(id, transaction);
    }

    @Override
    public CompletableFuture<?> removeGroup(String id) {
        return ypClient.removeObject(id);
    }

    @Override
    public CompletableFuture<?> removeGroups(Collection<String> ids) {
        List<YpTypedId> idsToRemove = ids.stream()
                .map(id -> new YpTypedId(id, YpObjectType.GROUP))
                .collect(Collectors.toList());
        return ypClient.getRawClient()
                .removeObjects(idsToRemove);
    }

    @Override
    public CompletableFuture<?> addMembers(String groupId, Set<String> memberIds, YpTransaction transaction) {
        return actionMembers(groupId, memberIds, (currentMembers, membersToAction) -> {
            currentMembers.addAll(membersToAction);
            return currentMembers;
        }, transaction);
    }

    @Override
    public CompletableFuture<?> removeMembers(String groupId, Set<String> memberIds, YpTransaction transaction) {
        return actionMembers(groupId, memberIds, (currentMembers, membersToAction) -> {
            currentMembers.removeAll(membersToAction);
            return currentMembers;
        }, transaction);
    }

    @Override
    public CompletableFuture<?> updateMembers(String groupId, Set<String> memberIds, YpTransaction transaction) {
        return updateMembers(groupId, memberIds, Optional.of(transaction));
    }

    @Override
    public CompletableFuture<?> updateMembers(String groupId, Set<String> memberIds) {
        return updateMembers(groupId, memberIds, Optional.empty());
    }

    public CompletableFuture<?> updateMembers(String groupId, Set<String> memberIds, Optional<YpTransaction> transaction) {
        final UpdateYpObjectRequest<TGroupSpec, TGroupStatus> request = new UpdateYpObjectRequest.Builder<TGroupSpec,
                TGroupStatus>()
                .setSpec(TGroupSpec.newBuilder().addAllMembers(memberIds).build())
                .build();
        return transaction.isPresent() ?
                ypClient.updateObject(groupId, transaction.get(), request) :
                ypClient.updateObject(groupId, request);
    }

    private CompletableFuture<?> actionMembers(String groupId, Set<String> memberToAction,
                                               BiFunction<Collection<String>, Collection<String>, Collection<String>> action, YpTransaction transaction) {
        return getMembers(groupId, transaction)
                .thenApply(currentMembers -> action.apply(new HashSet<>(currentMembers), memberToAction))
                .thenCompose(updatedMembers ->
                        ypClient.updateObject(groupId, transaction,
                                new UpdateYpObjectRequest.Builder<TGroupSpec, TGroupStatus>()
                                        .setSpec(TGroupSpec.newBuilder().addAllMembers(updatedMembers).build())
                                        .build()));
    }

    @Override
    public CompletableFuture<List<String>> getMembers(String groupId, YpTransaction transaction) {
        return ypClient.getObject(groupId, transaction.getStartTimestamp(), SPEC_WITH_TIMESTAMP)
                .thenApply(Optional::orElseThrow)
                .thenApply(YpObject::getSpec)
                .thenApply(TGroupSpec::getMembersList);
    }

    @Override
    public CompletableFuture<Boolean> exists(String groupId, YpTransaction transaction) {
        return ypClient.getObject(groupId, transaction.getStartTimestamp(), SPEC_WITH_TIMESTAMP)
                .thenApply(Optional::isPresent);
    }

    @Override
    public CompletableFuture<Map<String, Set<String>>> getGroupsWithPrefix(String prefix) {
        return ypClient.selectObjects(SPEC_WITH_TIMESTAMP, emptyMap())
                .thenApply(SelectedObjects::getObjects)
                .thenApply(objects -> filterByPrefix(objects, prefix));
    }

    @Override
    public CompletableFuture<Map<String, Set<String>>> getGroupsWithLabels(Map<String, String> labels) {
        return ypClient.generateTimestamp()
                .thenCompose(timestamp -> getGroupsWithLabels(labels, timestamp));
    }

    @Override
    public CompletableFuture<Map<String, Set<String>>> getGroupsWithLabels(Map<String, String> labels, long transactionTimestamp) {
        return ypClient.selectObjects(SPEC_WITH_TIMESTAMP, labels, transactionTimestamp)
                .thenApply(SelectedObjects::getObjects)
                .thenApply(objects -> filterByPrefix(objects,
                        labels.getOrDefault(SYSTEM_LABEL_KEY, "") + ":"));
    }

    private Map<String, Set<String>> filterByPrefix(
            Map<String, Try<YpObject<SchemaMeta, TGroupSpec, TGroupStatus>>> objects,
            String prefix) {
        return objects.entrySet().stream()
                .filter(entry -> entry.getKey().startsWith(prefix))
                .filter(entry -> entry.getValue().isSuccess())
                .collect(Collectors.toMap(Map.Entry::getKey,
                        entry -> Set.copyOf(entry.getValue().get().getSpec().getMembersList())));
    }

    @Override
    public CompletableFuture<Map<String, YpGroup>> getGroupsByLabels(Map<String, String> labels,
                                                                     Optional<Long> transactionTimestamp) {
        return (transactionTimestamp.map(CompletableFuture::completedFuture).orElseGet(ypClient::generateTimestamp))
                .thenCompose(timestamp -> ypClient.selectObjects(SPEC_LABELS, labels, timestamp))
                .thenApply(SelectedObjects::getObjects)
                .thenApply(objects ->
                        objects.entrySet().stream()
                                .filter(entry -> entry.getValue().isSuccess())
                                .collect(Collectors.toMap(Map.Entry::getKey,
                                        entry -> YpGroup.fromProto(entry.getKey(), entry.getValue().get())))
                        );
    }

    @Override
    public CompletableFuture<Set<String>> listAllIds() {
        return ypClient.listAllIds();
    }

    @Override
    public CompletableFuture<List<YpAbcRoleGroup>> getAbcRoleGroups() {
        YpSelectStatement.Builder builder = YpSelectStatement.builder(YpObjectType.GROUP, YpPayloadFormat.YSON)
                .addSelector(Paths.ID)
                .addSelector("/labels/abc/service_id")
                .addSelector("/labels/abc/scope_slug")
                .setFilter("[/labels/abc/type] = \"abc:service-scope\"");
        return YpRequestWithPaging.selectObjects(ypClient.getRawClient(), selectAbcGroupPageSize, builder, payloads -> {
            String groupId = payloadToYson(payloads.get(0)).stringValue();
            long serviceId = payloadToYson(payloads.get(1)).longValue();
            String scopeId = payloadToYson(payloads.get(2)).stringValue();

            return new YpAbcRoleGroup(groupId, scopeId, serviceId);
        });
    }

}
