package ru.yandex.infra.auth.yp;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yp.model.YpTransaction;

final public class YpGroupsHelper {
    public static final String SYSTEM_LABEL_KEY = "system";
    public static final String SYSTEM_IDM_LABEL_VALUE = "idm";
    public static final String IDM_GROUP_PREFIX = "idm:";

    private static final Logger LOG = LoggerFactory.getLogger(YpGroupsHelper.class);

    static public CompletableFuture<?> doAddMembersToGroup(YpGroupsClient groupsClient, String groupId,
            Set<String> loginsToAdd, YpTransaction transaction, String cluster, boolean replaceMembers) {
        return groupsClient.exists(groupId, transaction)
                .thenCompose(isExist -> (CompletableFuture<?>)
                        (isExist ? updateExistingGroupMembers(groupsClient, groupId, loginsToAdd, transaction, cluster, replaceMembers) :
                                   createNewGroup(groupsClient, groupId, loginsToAdd, transaction, cluster)));
    }

    static private CompletableFuture<?> createNewGroup(YpGroupsClient groupsClient, String groupId,
        Set<String> loginsToAdd, YpTransaction transaction, String cluster) {

        Map<String, Object> labels = extractLabelsFromGroupId(groupId);
        LOG.info("[{}]: Creating new group '{}' with labels '{}' and members {}", cluster, groupId, labels, loginsToAdd);
        return groupsClient.addGroup(groupId, loginsToAdd, transaction, labels);
    }

    static private CompletableFuture<?> updateExistingGroupMembers(YpGroupsClient groupsClient, String groupId,
                                                                   Set<String> logins, YpTransaction transaction,
                                                                   String cluster, boolean replaceMembers) {
        if (replaceMembers) {
            LOG.info("[{}]: Updating group '{}' with replacing members to {}", cluster, groupId, logins);
            return groupsClient.updateMembers(groupId, logins, transaction);
        } else {
            LOG.info("[{}]: Updating group '{}' appending members {}", cluster, groupId, logins);
            return groupsClient.addMembers(groupId, logins, transaction);
        }
    }

    static private Map<String, Object> extractLabelsFromGroupId(String groupId) {
        Map<String, Object> labels = new HashMap<>();
        if (groupId.startsWith(IDM_GROUP_PREFIX)) {
            labels.put(SYSTEM_LABEL_KEY, SYSTEM_IDM_LABEL_VALUE);
        } else if (groupId.contains(":")) {
            String systemLabel = groupId.substring(0, groupId.indexOf(":"));
            labels.put(SYSTEM_LABEL_KEY, systemLabel);
        }
        return labels;
    }

    static public CompletableFuture<?> doRemoveMembersFromGroup(YpGroupsClient groupsClient, String groupId,
            Set<String> loginsToRemove, YpTransaction transaction, String cluster) {
        return groupsClient.exists(groupId, transaction)
                .thenCompose(isExist -> {
                    if (!isExist) {
                        // Nothing to remove -> it's OK
                        return CompletableFuture.completedFuture(true).thenRun(
                                () -> LOG.info("[{}]: Group '{}' is missing, nothing to remove", cluster, groupId));
                    }
                    return groupsClient
                            .removeMembers(groupId, loginsToRemove, transaction)
                            .thenRun(() -> LOG.info("[{}]: Successfully removed members {} from group '{}'",
                                    cluster, loginsToRemove, groupId));
                });
    }
}
