package ru.yandex.infra.auth.yp;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.yp.UpdateYpObjectRequest;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.yp.client.api.AccessControl;
import ru.yandex.yp.model.YpTransaction;

import static ru.yandex.infra.auth.yp.YpUtils.generateACE;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.META;

public final class YpObjectPermissionsUpdaterImpl<Meta extends SchemaMeta, Spec extends Message, Status extends Message>
        implements YpObjectPermissionsUpdater {
    private static final Logger LOG = LoggerFactory.getLogger(YpObjectPermissionsUpdaterImpl.class);

    private final YpTransactionClient transactionClient;
    private final YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository;

    public YpObjectPermissionsUpdaterImpl(YpTransactionClient transactionClient,
            YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository) {
        this.transactionClient = transactionClient;
        this.ypRepository = ypRepository;
    }

    static private class GetRolesResult {
        final Acl acl;
        final YpTransaction transaction;

        private GetRolesResult(Acl acl, YpTransaction transaction) {
            this.acl = acl;
            this.transaction = transaction;
        }
    }

    private CompletableFuture<?> actionRole(String objectId, String subjectId,
            Function<List<AccessControl.TAccessControlEntry>, Boolean> condition,
            BiFunction<List<AccessControl.TAccessControlEntry>, List<AccessControl.TAccessControlEntry>, Boolean> action,
            YpTransaction transaction, String actionDescription) {
        return getAcl(objectId, transaction)
                .thenCompose(getRolesResult -> {
                    List<AccessControl.TAccessControlEntry> curAces = getRolesResult.acl.getEntries().stream()
                            .filter(entry -> entry.getSubjectsList().contains(subjectId))
                            .collect(Collectors.toList());

                    if (condition.apply(curAces)) {
                        List<AccessControl.TAccessControlEntry> entries = new ArrayList<>(getRolesResult.acl.getEntries());

                        action.apply(entries, curAces);
                        LOG.info("{} '{}' /meta/acl entry for group {}", actionDescription, objectId, subjectId);
                        return updateObject(objectId, new Acl(entries), getRolesResult.transaction)
                                .thenApply(x -> getRolesResult.transaction);
                    }
                    return CompletableFuture.completedFuture(getRolesResult.transaction);
                });
    }

    @Override
    public CompletableFuture<?> addRole(String objectId, String subjectId, List<RolesInfo.RoleAce> aces) {
        return transactionClient.runWithTransaction(transaction ->
                addRole(objectId, subjectId, aces, transaction));
    }

    @Override
    public CompletableFuture<?> addRole(String objectId,
            String subjectId,
            List<RolesInfo.RoleAce> aces,
            YpTransaction transaction) {
        return actionRole(objectId, subjectId, List::isEmpty,
                (entries, oldAces) -> entries.addAll(
                        aces.stream()
                                .map(ace -> generateACE(subjectId,
                                        ace.getRolePermissions(),
                                        ace.getRoleAttributePaths()))
                                .collect(Collectors.toList())
                ), transaction, "Adding");
    }

    @Override
    public CompletableFuture<?> updateRole(String objectId, String subjectId, List<RolesInfo.RoleAce> aces) {
        return transactionClient.runWithTransaction(transaction -> actionRole(objectId, subjectId, (ignored) -> true,
                (entries, oldAces) -> {
                    entries.removeAll(oldAces);
                    return entries.addAll(
                            aces.stream()
                                    .map(ace -> generateACE(subjectId,
                                            ace.getRolePermissions(),
                                            ace.getRoleAttributePaths()))
                                    .collect(Collectors.toList())
                    );
                }, transaction, "Updating"));
    }

    @Override
    public CompletableFuture<?> removeRoles(String objectId, Set<String> subjectIds) {
        return transactionClient.runWithTransaction(transaction -> getAcl(objectId, transaction)
                .thenCompose(getRolesResult -> {
                    List<AccessControl.TAccessControlEntry> acesAfterRemovedSubjects = getRolesResult.acl.getEntries().stream()
                            .filter(entry -> entry.getSubjectsList().stream().noneMatch(subjectIds::contains))
                            .collect(Collectors.toList());

                    int totalAcesCount = getRolesResult.acl.getEntries().size();
                    if (acesAfterRemovedSubjects.size() != totalAcesCount) {
                        LOG.info("Trying to remove '{}' /meta/acl entries for subjects: {}", objectId, subjectIds);
                        return updateObject(objectId, new Acl(acesAfterRemovedSubjects), getRolesResult.transaction)
                                .thenRun(() -> LOG.info("Removed {} of {} '{}' /meta/acl entries",
                                        totalAcesCount - acesAfterRemovedSubjects.size(), totalAcesCount, objectId))
                                .thenApply(x -> getRolesResult.transaction);
                    }
                    return CompletableFuture.completedFuture(getRolesResult.transaction);
                }));
    }

    private CompletableFuture<GetRolesResult> getAcl(String id, YpTransaction transaction) {
        return ypRepository.getObject(id, transaction.getStartTimestamp(), META)
                .thenCompose(object -> {
                    if (object.isEmpty()) {
                        throw new MissedYpObjectException(id, ypRepository.getObjectType());
                    }
                    return CompletableFuture.completedFuture(object.get().getMeta().getAcl());
                }).thenApply(acl -> new GetRolesResult(acl, transaction));
    }

    private CompletableFuture<?> updateObject(String id, Acl acl, YpTransaction transaction) {
        UpdateYpObjectRequest<Spec, Status> updateRequest =
                new UpdateYpObjectRequest.Builder<Spec, Status>()
                        .setAcl(acl)
                        .build();
        return ypRepository.updateObject(id, transaction, updateRequest);
    }
}
