package ru.yandex.infra.auth.yp;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import com.google.common.collect.Streams;
import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.auth.Metrics;
import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.TreeNode;
import ru.yandex.infra.auth.dto.ProjectInfo;
import ru.yandex.infra.controller.dto.NannyServiceMeta;
import ru.yandex.infra.controller.dto.ProjectMeta;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.dto.StageMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.NamespacedGaugeRegistry;
import ru.yandex.infra.controller.yp.YpObject;
import ru.yandex.infra.controller.yp.YpObjectSettings;
import ru.yandex.infra.controller.yp.YpObjectTransactionalRepository;
import ru.yandex.infra.controller.yp.YpObjectsCache;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.time.TimeUtils;
import ru.yandex.yp.client.api.DataModel;
import ru.yandex.yp.client.api.TNannyServiceSpec;
import ru.yandex.yp.client.api.TNannyServiceStatus;
import ru.yandex.yp.client.api.TProjectSpec;
import ru.yandex.yp.client.api.TProjectStatus;
import ru.yandex.yp.client.api.TStageSpec;
import ru.yandex.yp.client.api.TStageStatus;
import ru.yandex.yp.model.YpObjectType;

import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.META;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC_META;

public class YpObjectsTreeGetterImpl implements YpObjectsTreeGetter {
    private static final Logger LOG = LoggerFactory.getLogger(YpObjectsTreeGetterImpl.class);
    private static final String TMP_ACCOUNT = "tmp";

    private final YpObjectTransactionalRepository<StageMeta, TStageSpec, TStageStatus> ypStageClient;
    private final YpObjectsCache<StageMeta, TStageSpec, TStageStatus> stagesCache;
    private final YpObjectsCache<ProjectMeta, TProjectSpec, TProjectStatus> projectsCache;
    private final YpObjectsCache<SchemaMeta, DataModel.TGroupSpec, DataModel.TGroupStatus> groupsCache;
    private final YpObjectsCache<NannyServiceMeta, TNannyServiceSpec, TNannyServiceStatus> nannyServicesCache;
    private final Metrics metrics;
    private final long useUuidSinceTimestamp;
    private final boolean processNannyServicesWithoutProject;

    private volatile TreeNodeWithTimestamp lastLoadResults;
    private final AtomicBoolean loadInProgress = new AtomicBoolean(false);

    public YpObjectsTreeGetterImpl(YpObjectTransactionalRepository<StageMeta, TStageSpec, TStageStatus> ypStageClient,
            YpObjectTransactionalRepository<NannyServiceMeta, TNannyServiceSpec, TNannyServiceStatus> ypNannyServiceClient,
            YpObjectTransactionalRepository<ProjectMeta, TProjectSpec, TProjectStatus> ypProjectClient,
            YpObjectTransactionalRepository<SchemaMeta, DataModel.TGroupSpec, DataModel.TGroupStatus> ypGroupsClient,
            Metrics metrics,
            String useUuidSince,
            Map<YpObjectType, YpObjectSettings> ypObjectsCacheSettings,
            GaugeRegistry gaugeRegistry,
            boolean processNannyServicesWithoutProject) {
        this.ypStageClient = ypStageClient;
        this.metrics = metrics;
        this.processNannyServicesWithoutProject = processNannyServicesWithoutProject;

        stagesCache = new YpObjectsCache<>(ypStageClient,
                YpObjectSettings.getSettingsForType(ypObjectsCacheSettings, YpObjectType.STAGE),
                new NamespacedGaugeRegistry(gaugeRegistry, "stages"),
                META);
        nannyServicesCache = new YpObjectsCache<>(ypNannyServiceClient,
                YpObjectSettings.getSettingsForType(ypObjectsCacheSettings, YpObjectType.NANNY_SERVICE),
                new NamespacedGaugeRegistry(gaugeRegistry, "nanny_services"),
                META);
        projectsCache = new YpObjectsCache<>(ypProjectClient,
                YpObjectSettings.getSettingsForType(ypObjectsCacheSettings, YpObjectType.PROJECT),
                new NamespacedGaugeRegistry(gaugeRegistry, "projects"),
                SPEC_META);
        groupsCache = new YpObjectsCache<>(ypGroupsClient,
                YpObjectSettings.getSettingsForType(ypObjectsCacheSettings, YpObjectType.GROUP),
                new NamespacedGaugeRegistry(gaugeRegistry, "groups"),
                SPEC);

        final DateTime uuidSinceDateTime = new TimeUtils.DateTimeUtils().parse(useUuidSince, DateTimeZone.getDefault());
        this.useUuidSinceTimestamp = uuidSinceDateTime.getMillis() * 1000;
        LOG.info("Using uuid for idm.nodes.unique_id since {} ({})", uuidSinceDateTime, useUuidSinceTimestamp);
    }

    private CompletableFuture<Set<StageMeta>> loadStages(Long lastSelectTimestamp) {
        return stagesCache.selectObjects(Optional.of(lastSelectTimestamp))
                .thenApply(stages -> stages.values().stream()
                        .filter(Try::isSuccess)
                        .map(Try::get)
                        .filter(object -> validateObjectId(object.getMeta().getId()))
                        .map(YpObject::getMeta)
                        .collect(Collectors.toSet()));
    }

    private CompletableFuture<Set<NannyServiceMeta>> loadNannyServices(Long lastSelectTimestamp) {
        return nannyServicesCache.selectObjects(Optional.of(lastSelectTimestamp))
                .thenApply(objects -> objects.values().stream()
                        .filter(Try::isSuccess)
                        .map(Try::get)
                        .filter(object -> validateObjectId(object.getMeta().getId()))
                        .map(YpObject::getMeta)
                        .collect(Collectors.toSet()));
    }

    private CompletableFuture<Map<String, ProjectInfo>> loadProjects(Long lastSelectTimestamp) {
        return projectsCache.selectObjects(Optional.of(lastSelectTimestamp))
                .thenApply(projects -> projects.values().stream()
                .filter(Try::isSuccess)
                .map(Try::get)
                .filter(object -> validateObjectId(object.getMeta().getId()))
                .collect(Collectors.toMap(
                        project -> project.getMeta().getId(),
                        project -> new ProjectInfo(
                                project.getMeta(),
                                project.getSpec().getUserSpecificBoxTypesList(),
                                project.getMeta().getOwnerId(),
                                project.getSpec().getAccountId()))));
    }

    private CompletableFuture<Map<String, Set<String>>> loadGroups(Long lastSelectTimestamp) {
        return groupsCache.selectObjects(Optional.of(lastSelectTimestamp))
                .thenApply(groups -> groups.entrySet().stream()
                        .filter(entry -> entry.getValue().isSuccess())
                        .collect(Collectors.toMap(Map.Entry::getKey,
                                entry -> Set.copyOf(entry.getValue().get().getSpec().getMembersList()))));
    }

    private void collectMetrics(Map<String, ProjectInfo> projectIdToInfo, Set<StageMeta> stagesMeta) {
        long tmpProjectsCount = projectIdToInfo.values()
                .stream()
                .filter(projectInfo -> projectInfo.getAccountId().equals(TMP_ACCOUNT))
                .count();

        metrics.setTmpQuotaUsageCount((int)tmpProjectsCount);


        long stageQuotaMismatchCount = stagesMeta
                .stream()
                .filter(stageMeta -> {
                    if (StringUtils.isBlank(stageMeta.getAccountId())) {
                        return false;
                    }
                    var projectInfo = projectIdToInfo.get(stageMeta.getProjectId());
                    return projectInfo == null || !stageMeta.getAccountId().equals(projectInfo.getAccountId());
                })
                .count();

        metrics.setQuotaMismatchCount((int)stageQuotaMismatchCount);
    }

    public TreeNodeWithTimestamp getObjectsTree(boolean withMetrics) throws YpObjectsTreeGetterError {
        //Prevent loading of YP objects in parallel.
        //YpObjectsCache (stagesCache, projectsCache...) can be used from multiple threads,
        //but it will return exception for second request running in parallel.
        if (!loadInProgress.compareAndSet(false, true)) {
            //in progress in other thread
            if (lastLoadResults == null) {
                throw new RuntimeException("Initial snapshot of Projects/Stages tree was not loaded yet");
            }
            LOG.debug("Projects tree loading is in progress, using previous snapshot with YP timestamp {}",
                    lastLoadResults.getTimestamp());
            return lastLoadResults;
        }

        LOG.debug("Starting loading tree of projects from YP...");
        long startTimeMillis = System.currentTimeMillis();
        try {
            Long lastSelectTimestamp = ypStageClient.generateTimestamp().get();

            AtomicReference<Set<StageMeta>> stages = new AtomicReference<>();
            AtomicReference<Set<NannyServiceMeta>> nannyServices = new AtomicReference<>();
            AtomicReference<Map<String, ProjectInfo>> projects = new AtomicReference<>();
            AtomicReference<Map<String, Set<String>>> groups = new AtomicReference<>();
            CompletableFuture.allOf(
                    loadStages(lastSelectTimestamp).thenAccept(stages::set),
                    loadNannyServices(lastSelectTimestamp).thenAccept(nannyServices::set),
                    loadProjects(lastSelectTimestamp).thenAccept(projects::set),
                    loadGroups(lastSelectTimestamp).thenAccept(groups::set)
            ).get();

            Set<StageMeta> stagesMeta = stages.get();
            Set<NannyServiceMeta> nannyServicesMeta = nannyServices.get();
            Map<String, ProjectInfo> projectIdToInfo = projects.get();

            if (withMetrics) {
                collectMetrics(projectIdToInfo, stagesMeta);
            }

            TreeNode.Builder rootBuilder = new TreeNode.Builder().makeConvertibleToRole();

            addStageNodes(rootBuilder, projectIdToInfo, stagesMeta);
            addNannyServiceNodes(rootBuilder, projectIdToInfo, nannyServicesMeta);
            addChildlessProjectNodes(rootBuilder, projectIdToInfo, stagesMeta, nannyServicesMeta);

            lastLoadResults = new TreeNodeWithTimestamp(rootBuilder.build(), lastSelectTimestamp, groups.get());
            LOG.debug("Loaded projects tree with YP timestamp {} in {} ms",
                    lastSelectTimestamp, System.currentTimeMillis() - startTimeMillis);
            return lastLoadResults;
        } catch (InterruptedException | ExecutionException | RuntimeException e) {
            metrics.addYpError();
            throw new YpObjectsTreeGetterError("Exception while read tree of projects from YP", e);
        }
        finally {
            loadInProgress.set(false);
        }
    }

    private TreeNode.Builder createProjectNode(ProjectInfo projectInfo) {
        return new TreeNode.Builder()
                .withUniqueId(getUniqueIdForProjectOrStageIdmNode(projectInfo.getMeta()))
                .withName(projectInfo.getMeta().getId())
                .withOwner(projectInfo.getOwner())
                .makeConvertibleToRole();
    }

    private void addNannyServiceNodes(TreeNode.Builder rootBuilder, Map<String, ProjectInfo> projectIdToInfo, Set<NannyServiceMeta> nannyServicesMeta) {

        if (processNannyServicesWithoutProject) {
            rootBuilder.withChild(new TreeNode.Builder()
                    .withName(Role.NANNY_ROLES_PARENT_NODE)
                    .makeConvertibleToRole()
            );
        }

        nannyServicesMeta.forEach(nannyServiceMeta -> {
            String serviceId = nannyServiceMeta.getId();
            TreeNode.Builder serviceNode = new TreeNode.Builder()
                    .withUniqueId(nannyServiceMeta.getUuid())
                    .withName(serviceId)
                    .makeConvertibleToRole();

            TreeNode.Builder nannyNode = new TreeNode.Builder()
                    .withName(Role.NANNY_ROLES_PARENT_NODE)
                    .withChild(serviceNode)
                    .makeConvertibleToRole();

            final String projectId = nannyServiceMeta.getProjectId();
            if (StringUtils.isBlank(projectId)) {
                if (processNannyServicesWithoutProject) {
                    rootBuilder.withChild(nannyNode);
                }
                return;
            }

            ProjectInfo projectInfo = projectIdToInfo.get(projectId);
            if (projectInfo == null) {
                LOG.error("Project {} does not exist, failed to process nanny service {}!", projectId, serviceId);
                return;
            }

            TreeNode.Builder projectNode = createProjectNode(projectInfo).withChild(nannyNode);
            rootBuilder.withChild(projectNode);
        });
    }

    private void addChildlessProjectNodes(TreeNode.Builder rootBuilder,
                                          Map<String, ProjectInfo> projectIdToInfo,
                                          Set<StageMeta> stagesMeta,
                                          Set<NannyServiceMeta> nannyServicesMeta) {
        Set<String> addedProjectIds = Streams.concat(
                stagesMeta.stream().map(StageMeta::getProjectId),
                nannyServicesMeta.stream().map(NannyServiceMeta::getProjectId)
            ).collect(Collectors.toSet());

        Sets.difference(projectIdToInfo.keySet(), addedProjectIds)
                .forEach(projectId -> {
                    TreeNode.Builder projectNode = createProjectNode(projectIdToInfo.get(projectId));
                    rootBuilder.withChild(projectNode);
                });
    }

    private void addStageNodes(TreeNode.Builder rootBuilder, Map<String, ProjectInfo> projectIdToInfo, Set<StageMeta> stagesMeta) {
        stagesMeta.forEach(stageMeta -> {
            final String projectId = stageMeta.getProjectId();
            final String stageId = stageMeta.getId();

            ProjectInfo projectInfo = projectIdToInfo.get(projectId);
            if (projectInfo == null) {
                LOG.error("Project {} does not exist, failed to process stage {}!", projectId, stageId);
                return;
            }

            TreeNode.Builder stageNode = new TreeNode.Builder()
                    .withUniqueId(getUniqueIdForProjectOrStageIdmNode(stageMeta))
                    .withName(stageId)
                    .makeConvertibleToRole();

            projectInfo.getUserSpecificBoxes().forEach(specificBox -> {
                TreeNode.Builder boxNode = new TreeNode.Builder()
                        .withName(specificBox)
                        .withUniqueId(Role.getLeafUniqueId(stageNode.getUniqueId(), specificBox))
                        .makeConvertibleToRole();
                stageNode.withChild(boxNode);
            });

            TreeNode.Builder projectNode = createProjectNode(projectInfo).withChild(stageNode);
            rootBuilder.withChild(projectNode);
        });
    }

    @Override
    public String getUniqueIdForProjectOrStageIdmNode(SchemaMeta meta) {
        return meta.getCreationTime() >= useUuidSinceTimestamp ? meta.getUuid() : "";
    }

    private static boolean validateObjectId(String metaId) {
        return metaId.matches("[A-Za-z0-9\\-_]+");
    }
}
