package ru.yandex.infra.auth.yp;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.codahale.metrics.MetricRegistry;
import com.google.common.collect.ImmutableSet;
import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.auth.Metrics;
import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.controller.yp.YpTransactionClient;
import ru.yandex.yp.model.YpError;
import ru.yandex.yp.model.YpException;

import static java.lang.String.format;
import static java.lang.Thread.sleep;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.IDM_GROUP_PREFIX;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.doAddMembersToGroup;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.doRemoveMembersFromGroup;
import static ru.yandex.infra.controller.util.ExceptionUtils.tryExtractYpError;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.META;
import static ru.yandex.yp.model.YpErrorCodes.NO_SUCH_OBJECT;

public final class YpServiceImpl extends YpServiceReadOnlyImpl {
    private static final Logger LOG = LoggerFactory.getLogger(YpServiceImpl.class);
    private static final String ATTRIBUTE_PATH_DELIMITER = "/";
    private static final String MASTER_CLUSTER_NAME = "master";

    private final YpGroupsClient ypGroupsClientMaster;
    private final YpTransactionClient ypTransactionClientMaster;
    private final YpGroupsAsyncQueue groupsAsyncQueue;
    private final RolesInfo rolesInfo;
    private final Duration groupsRequestsRate;
    private final int groupsRequestsLimit;
    private final Metrics metrics;
    private final YpObjectPermissionsUpdater ypStagesAclUpdater;
    private final YpObjectPermissionsUpdater ypProjectsAclUpdater;
    private final YpObjectPermissionsUpdater ypNannyServiceAclUpdater;

    public YpServiceImpl(YpObjectsTreeGetter ypObjectsTreeGetter,
            YpObjectPermissionsUpdater ypStagesAclUpdater,
            YpObjectPermissionsUpdater ypProjectsAclUpdater,
            YpObjectPermissionsUpdater ypNannyServiceAclUpdater,
            YpClients master,
            Map<String, YpClients> slaves,
            RolesInfo rolesInfo, Config rootConfig,
            Duration groupsRequestsRate, int groupsRequestsLimit,
            MetricRegistry metricRegistry,
            Metrics metrics,
            String systemName) {
        super(ypObjectsTreeGetter, master, slaves, rootConfig, systemName, metrics);
        this.ypStagesAclUpdater = ypStagesAclUpdater;
        this.ypNannyServiceAclUpdater = ypNannyServiceAclUpdater;
        this.ypProjectsAclUpdater = ypProjectsAclUpdater;
        this.ypGroupsClientMaster = master.getGroupsClient();
        this.ypTransactionClientMaster = master.getYpTransactionClient();
        this.rolesInfo = rolesInfo;
        this.groupsRequestsRate = groupsRequestsRate;
        this.groupsRequestsLimit = groupsRequestsLimit;
        this.groupsAsyncQueue = new YpGroupsAsyncQueue(
                slaves,
                metricRegistry,
                metrics
        );

        this.metrics = metrics;
    }

    private static class RoleSubjectAction {
        final Role role;
        final String objectId;
        final String memberId;
        final String groupPrefix;
        final String groupId;
        final YpObjectPermissionsUpdater updater;
        final List<RolesInfo.RoleAce> aces;

        private RoleSubjectAction(Role role, String objectId,
                String memberId, String groupPrefix,
                YpObjectPermissionsUpdater updater,
                List<RolesInfo.RoleAce> aces) {
            this.role = role;
            this.objectId = objectId;
            this.memberId = memberId;
            this.groupId = getYpGroupName(role, groupPrefix);
            this.groupPrefix = groupPrefix;
            this.updater = updater;
            this.aces = aces;
        }

        static RoleSubjectAction create(RoleSubject roleSubject,
                YpServiceImpl ypService,
                RolesInfo rolesInfo,
                String groupPrefix) {
            Role role = roleSubject.getRole();
            YpObjectPermissionsUpdater updater = ypService.getAclUpdaterForRole(role);
            if (updater == null) {
                LOG.error("Invalid role format - too many levels: {}", role);
                return null;
            }

            String objectId;
            try {
                objectId = role.size() == 2
                        ? role.getLevelName(Role.PROJECT_ID_LEVEL_DEPTH_IN_ROLE).orElseThrow()
                        : role.isNannyRole() ? role.getNannyServiceId().orElseThrow() : role.getStageId().orElseThrow();
            } catch (NoSuchElementException e) {
                LOG.error("Error while parsing role: {}", role, e);
                return null;
            }

            List<RolesInfo.RoleAce> aces = tryConcatSpecificBox(rolesInfo.getRoleAces(role.getLeaf()), role);

            return new RoleSubjectAction(role,
                    objectId,
                    getLoginOrGroupWithPrefix(roleSubject, IDM_GROUP_PREFIX),
                    groupPrefix,
                    updater,
                    aces);
        }
    }

    private static String getLoginOrGroupWithPrefix(RoleSubject roleSubject, String prefix) {
        return roleSubject.isPersonal() ? roleSubject.getLogin() : prefix + roleSubject.getGroupId();
    }

    private static List<RolesInfo.RoleAce> tryConcatSpecificBox(List<RolesInfo.RoleAce> aces, Role role) {
        if (role.size() == 4 && !role.isNannyRole()) {
            return aces.stream()
                    .map(ace -> {
                        List<String> attributePaths = Collections.emptyList();
                        if (ace.getRoleAttributePaths().size() != 0) {
                            attributePaths = List.of(ace.getBaseAttributePath()
                                    .concat(ATTRIBUTE_PATH_DELIMITER)
                                    .concat(role.getLevelName(Role.BOX_ID_LEVEL_DEPTH_IN_ROLE).orElseThrow()));
                        }
                        // we concatenate only first attribute path
                        return new RolesInfo.RoleAce(ace.getRolePermissions(),
                                attributePaths,
                                ace.getBaseAttributePath());
                    })
                    .collect(Collectors.toList());
        }

        return aces;
    }

    private YpObjectPermissionsUpdater getAclUpdaterForRole(Role role) {
        if (role.size() == 2) {
            return ypProjectsAclUpdater;
        }
        if (role.isNannyRole()) {
            return ypNannyServiceAclUpdater;
        }
        return ypStagesAclUpdater;
    }

    @Override
    public void removeGroup(String groupId) {
        removeGroup(groupId, MASTER_CLUSTER_NAME, ypGroupsClientMaster);
        slaves.forEach((cluster, ypClients) -> removeGroup(groupId, cluster, ypClients.getGroupsClient()));
    }

    private CompletableFuture<?> removeGroup(String groupId, String cluster, YpGroupsClient client) {
        return client.removeGroup(groupId)
                .thenRun(() -> LOG.info("[{}]: YP group was successfully removed: {}", cluster, groupId))
                .exceptionally(exception -> {
                    Throwable cause = exception.getCause();
                    if (cause instanceof YpException && ((YpException)cause).getYpError().getCode() == NO_SUCH_OBJECT) {
                        LOG.debug("[{}] Can't remove yp group (does not exist): {}", cluster, groupId);
                    } else {
                        metrics.addYpError();
                        LOG.error("[{}]: YP error while removing group {}: {}", cluster, groupId, exception);
                    }
                    return null;
                });
    }

    @Override
    public CompletableFuture<?> removeRolesFromStageAcl(String stageId, Set<Role> removedRoles) {

        Role anyStageRole = removedRoles.stream().findAny().orElseThrow();
        //all roles have the same uniqueId (== Stage.uuid or empty string for old stages)
        String uuid = anyStageRole.getYpObjectUuid().orElse("");

        return master.getStageRepository()
                .getObject(stageId, META)
                .thenAccept(optionalObject -> {
                    //If stage is not found (was deleted), nothing to fix
                    if (optionalObject.isEmpty()) {
                        LOG.debug("Stage {} was removed. Skipping /meta/acl check", stageId);
                        return;
                    }

                    String stageUuid = ypObjectsTreeGetter.getUniqueIdForProjectOrStageIdmNode(optionalObject.get().getMeta());
                    //if stage with the same id but different uuid
                    if (!stageUuid.equals(uuid)) {
                        LOG.debug("Stage {} was recreated with another uuid. Skipping /meta/acl check for removed roles", stageId);
                        return;
                    }

                    Set<String> subjectIds = removedRoles.stream()
                            .map(this::getYpGroupName)
                            .filter(Objects::nonNull)
                            .collect(Collectors.toSet());

                    runWithRetries(() -> ypStagesAclUpdater.removeRoles(stageId, subjectIds),
                            String.format("Stage '%s' ACLs cleanup", stageId));
                })
                .exceptionally(exception -> {
                    LOG.error(String.format("[%s]: YP error while stage '%s' ACLs cleanup", MASTER_CLUSTER_NAME, stageId), exception);
                    return null;
                });
    }

    @Override
    public void addRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        RoleSubjectAction action = RoleSubjectAction.create(
                roleSubject,
                this,
                rolesInfo,
                systemName);
        if (action == null) {
            throw new YpServiceException("Empty action");
        }

        // Add member and acl in cross-location DC
        runWithRetries(() -> ypTransactionClientMaster.runWithTransaction(
                transaction -> doAddMembersToGroup(ypGroupsClientMaster, action.groupId, Set.of(action.memberId),
                        transaction, MASTER_CLUSTER_NAME, false)
                        .thenCompose(x -> {
                            if (!action.objectId.isEmpty()) {
                                return action.updater.addRole(
                                        action.objectId, action.groupId, action.aces, transaction);
                            }
                            return CompletableFuture.completedFuture(null);
                        })),
                format("adding role %s with member '%s' for object '%s'",
                        action.role.getExtendedDescription(), action.memberId, action.objectId));

        if (!action.objectId.isEmpty()) {
            // There is no need to store SUPER_USER role in local YP masters
            groupsAsyncQueue.addMembersToGroup(action.groupId, Set.of(action.memberId));
        }
    }

    @Override
    public void updateRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        RoleSubjectAction action = RoleSubjectAction.create(
                roleSubject,
                this,
                rolesInfo,
                systemName);
        if (action == null) {
            throw new YpServiceException("Empty action");
        }

        if (action.objectId.isEmpty()) {
            return;
        }

        runWithRetries(() -> action.updater.updateRole(action.objectId, action.groupId, action.aces),
                format("updating ACL for role %s for object '%s'", action.role, action.objectId));
    }

    @Override
    public void removeRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        final Role role = roleSubject.getRole();
        if (role.size() > Role.MAX_DEPLOY_ROLE_DEPTH) {
            LOG.warn("Role path {} is deeper than expected", role.getExtendedDescription());
            return;
        }
        String deployGroupId = getYpGroupName(role);
        String memberId = getLoginOrGroupWithPrefix(roleSubject, IDM_GROUP_PREFIX);

        removeMembersFromGroup(deployGroupId, ImmutableSet.of(memberId));
    }

    @Override
    public void addMembersToGroup(String groupId, Set<String> loginsToAdd) throws YpServiceException {
        runWithRetries(() -> ypTransactionClientMaster.runWithTransaction(
                transaction -> doAddMembersToGroup(ypGroupsClientMaster, groupId, loginsToAdd, transaction,
                        MASTER_CLUSTER_NAME, false)),
                format("adding members %s for group '%s'", loginsToAdd, groupId));

        groupsAsyncQueue.addMembersToGroup(groupId, loginsToAdd);
    }

    @Override
    protected void syncMembersInGroup(String groupId, Set<String> logins, String cluster) {
        groupsAsyncQueue.syncMembersInGroup(groupId, logins, cluster);
    }

    @Override
    public void removeMembersFromGroup(String groupId, Set<String> loginsToRemove) throws YpServiceException {
        runWithRetries(() -> ypTransactionClientMaster.runWithTransaction(
                transaction -> doRemoveMembersFromGroup(ypGroupsClientMaster, groupId, loginsToRemove, transaction,
                        MASTER_CLUSTER_NAME)),
                format("removing members %s from idm group '%s'", loginsToRemove, groupId));

        groupsAsyncQueue.removeMembersFromGroup(groupId, loginsToRemove);
    }

    private void runWithRetries(Supplier<CompletableFuture<?>> future, String actionDescription) throws YpServiceException {
        Exception exception = new Exception("");
        for (int tryCounter = 1; tryCounter <= groupsRequestsLimit; ++tryCounter) {
            try {
                if (tryCounter > 1) {
                    sleep(groupsRequestsRate.toMillis());
                }

                future.get().get();
                LOG.info("[{}]: Successful {}", MASTER_CLUSTER_NAME, actionDescription);
                return;
            } catch (InterruptedException | ExecutionException e) {
                if (e.getCause() instanceof MissedYpObjectException) {
                    String errorMessage = String.format("[%s]: Stopping retries (reason: %s) for action: %s",
                            MASTER_CLUSTER_NAME, e.getCause().getMessage(), actionDescription);
                    LOG.warn(errorMessage);
                    throw new YpServiceException(errorMessage, e.getCause());
                }
                metrics.addYpError();
                exception = e;
                LOG.warn("[{}]: Try {}. YP error while {}: {}",
                        MASTER_CLUSTER_NAME, tryCounter, actionDescription, e.getMessage());
            }
        }
        String shortExceptionMessage = tryGetShortYpErrorMessage(exception).orElse(exception.getMessage());

        LOG.error("[{}]: Max retries exceeded.", MASTER_CLUSTER_NAME, exception);
        throw new YpServiceException(shortExceptionMessage);
    }

    private Optional<String> tryGetShortYpErrorMessage(Throwable error) {
        Optional<YpError> ypErrorOpt = tryExtractYpError(error);
        if (ypErrorOpt.isPresent()) {
            YpError ypError = ypErrorOpt.get();
            while (!ypError.getInnerErrors().isEmpty()) {
                ypError = ypError.getInnerErrors().get(0);
            }
            return ypError.getMessage();
        }
        return Optional.empty();
    }
}
