package ru.yandex.infra.auth.yp;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.infra.auth.Metrics;
import ru.yandex.infra.auth.Role;
import ru.yandex.infra.auth.RoleSubject;
import ru.yandex.infra.auth.RolesInfo;
import ru.yandex.infra.auth.TreeNode;
import ru.yandex.infra.auth.yp.YpObjectsTreeGetter.TreeNodeWithTimestamp;
import ru.yandex.infra.controller.yp.LabelBasedRepository;
import ru.yandex.infra.controller.yp.YpObjectRepository;
import ru.yandex.infra.controller.yp.YpRequestWithPaging;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static ru.yandex.infra.auth.Role.ROLE_NAME_DELIMITER;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.IDM_GROUP_PREFIX;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.SYSTEM_IDM_LABEL_VALUE;
import static ru.yandex.infra.auth.yp.YpGroupsHelper.SYSTEM_LABEL_KEY;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.META;
import static ru.yandex.infra.controller.util.YpUtils.CommonSelectors.SPEC;
import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;

public class YpServiceReadOnlyImpl implements YpService {
    private static final Logger LOG = LoggerFactory.getLogger(YpServiceReadOnlyImpl.class);

    public static final String YP_GROUP_NAME_DELIMITER = ":";
    private static final String EMPTY_PREFIX = "";
    private static final String ROLES_CONFIG_KEY = "roles";
    private static final String DEFAULT_CHILD_NAME = "__default__";
    private static final String CHILDREN_CONFIG_PATH = "children.";
    // IDM waits at least 15 minutes before role removing
    private static final long GLOBAL_CLEANUP_TIMEOUT_MS = 30 * 60 * 1000; // 30 minutes

    protected final String systemName;
    private final Config rootConfig;
    protected final YpObjectsTreeGetter ypObjectsTreeGetter;
    private final YpGroupsClient ypGroupsClient;
    protected final YpClients master;
    protected final Map<String, YpClients> slaves;

    private long globalCleanupStartTimestamp = 0L;
    private final Metrics metrics;

    public YpServiceReadOnlyImpl(YpObjectsTreeGetter ypObjectsTreeGetter,
                                 YpClients master,
                                 Map<String, YpClients> slaves,
                                 Config rootConfig,
                                 String systemName,
                                 Metrics metrics) {

        if (systemName.contains(YP_GROUP_NAME_DELIMITER)) {
            throw new RuntimeException(String.format("SystemName '%s' should not contains symbol '%s', because it's used as ypGroupName separator",
                    systemName,
                    YP_GROUP_NAME_DELIMITER));
        }

        this.ypObjectsTreeGetter = ypObjectsTreeGetter;
        this.master = master;
        this.ypGroupsClient = master.getGroupsClient();
        this.slaves = slaves;
        this.rootConfig = rootConfig;
        this.systemName = systemName;
        this.metrics = metrics;
    }

    @Override
    public YpClients getMasterClusterClients() {
        return master;
    }

    @Override
    public Map<String, YpClients> getSlaveClusterClients() {
        return slaves;
    }

    @Override
    public String getSystemName() {
        return systemName;
    }

    @Override
    public Set<Role> getRoles() throws YpObjectsTreeGetterError {
        if (globalCleanupStartTimestamp != 0) {
            if (System.currentTimeMillis() - globalCleanupStartTimestamp < GLOBAL_CLEANUP_TIMEOUT_MS) {
                // Cleanup all roles excepting super user ones
                return new HashSet<>(Collections.singletonList(Role.superUser()));
            }
            LOG.warn("Global cleanup done -> start uploading data to IDM !!!");
            globalCleanupStartTimestamp = 0;
        }

        TreeNodeWithTimestamp rootWithSelectedTimestamp = ypObjectsTreeGetter.getObjectsTree(true);
        return convertTreeToRoles(rootWithSelectedTimestamp.getTreeNode());
    }

    @Override
    public Set<String> getGarbageGroups() {
        try {
            TreeNodeWithTimestamp rootWithSelectedTimestamp = ypObjectsTreeGetter.getObjectsTree(false);
            Set<Role> roles = convertTreeToRoles(rootWithSelectedTimestamp.getTreeNode());
            if (roles.size() == 0) {
                throw new RuntimeException("Can't generate Roles from YP Projects/Stages for unknown reason");
            }

            var labels = ImmutableMap.of(YpGroupsHelper.SYSTEM_LABEL_KEY, systemName);
            //Getting groups with the same timestamp!
            Map<String, Set<String>> groups = ypGroupsClient.getGroupsWithLabels(labels, rootWithSelectedTimestamp.getTimestamp()).get();

            Set<String> groupsToRemove = groups.keySet().stream()
                    .filter(groupName -> !roles.contains(YpServiceReadOnlyImpl.getRole(groupName, systemName)))
                    .collect(Collectors.toSet());
            LOG.info("Found {} unused of {} total YP groups to remove", groupsToRemove.size(), groups.size());
            return groupsToRemove;

        } catch (InterruptedException | ExecutionException | RuntimeException | YpObjectsTreeGetterError e) {
            metrics.addYpError();
            LOG.error("YP error while trying to get garbage groups", e);
            return Collections.emptySet();
        }
    }

    @Override
    public Set<RoleSubject> getRoleSubjects() {
        try {
            return getRoleSubjectsFromRoleMembers(getRoleMembers());
        } catch (Exception exception) {
            LOG.error("YP error while tried to get all role subjects", exception);
            metrics.addYpError();
            return emptySet();
        }
    }

    public static Set<RoleSubject> getRoleSubjectsFromRoleMembers(Map<Role, Set<String>> roleMembers) {
        return roleMembers.entrySet().stream()
                .flatMap(tuple -> tuple.getValue().stream().map(member -> member.startsWith(IDM_GROUP_PREFIX) ?
                        new RoleSubject(
                                EMPTY_PREFIX,
                                Long.parseLong(member.replace(IDM_GROUP_PREFIX, EMPTY_PREFIX)),
                                tuple.getKey()) :
                        new RoleSubject(member, 0L, tuple.getKey())))
                .collect(Collectors.toSet());
    }

    @Override
    public Map<Role, Set<String>> getRoleMembers() throws YpObjectsTreeGetterError {
        TreeNodeWithTimestamp rootWithSelectedTimestamp = ypObjectsTreeGetter.getObjectsTree(false);
        Set<Role> allActiveRoles = convertTreeToRoles(rootWithSelectedTimestamp.getTreeNode());
        Map<String, Set<String>> groupsWithMembers = rootWithSelectedTimestamp.getGroups();

        return allActiveRoles.stream()
                .map(role -> {
                    String ypGroupName = getYpGroupName(role);
                    if (ypGroupName == null) {
                        return null;
                    }
                    Set<String> members = groupsWithMembers.get(ypGroupName);
                    if (members == null) {
                        return null;
                    }
                    return Tuple2.tuple(role, members);
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(t -> t._1, t -> t._2));
    }

    @Override
    public Map<String, Set<String>> getGroupsWithPrefix(String prefix) {
        try {
            return ypGroupsClient.getGroupsWithPrefix(prefix).get();
        } catch (InterruptedException | ExecutionException | RuntimeException e) {
            metrics.addYpError();
            LOG.error(format("YP error while tried to get all groups with prefix %s", prefix), e);
            return emptyMap();
        }
    }

    @Override
    public Map<String, Set<String>> getGroupsWithLabels(Map<String, String> labels) {
        try {
            return ypGroupsClient.getGroupsWithLabels(labels).get();
        } catch (InterruptedException | ExecutionException | RuntimeException e) {
            metrics.addYpError();
            LOG.error(format("YP error while tried to get all groups with labels %s", labels), e);
            return emptyMap();
        }
    }

    @Override
    public void removeGroup(String groupId) {
        LOG.info("removeGroup: {}  - skipping (read only mode)", groupId);
    }

    @Override
    public CompletableFuture<?> removeRolesFromStageAcl(String stageId, Set<Role> removedRoles) {
        LOG.info("removeRolesFromStageAcl: stage '{}' - skipping (read only mode)", stageId);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public void addRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        LOG.info("addRoleSubject: {} - skipping (read only mode)", roleSubject);
    }

    @Override
    public void updateRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        LOG.info("updateRoleSubject: {} - skipping (read only mode)", roleSubject);
    }

    @Override
    public void removeRoleSubject(RoleSubject roleSubject) throws YpServiceException {
        LOG.info("removeRoleSubject: {} - skipping (read only mode)", roleSubject);
    }

    @Override
    public void addMembersToGroup(String groupId, Set<String> loginsToAdd) throws YpServiceException {
        LOG.info("addMembersToGroup: {} {} - skipping (read only mode)", groupId, loginsToAdd);
    }

    @Override
    public void removeMembersFromGroup(String groupId, Set<String> loginsToRemove) throws YpServiceException {
        LOG.info("removeMembersFromGroup: {} {} - skipping (read only mode)", groupId, loginsToRemove);
    }

    protected void syncMembersInGroup(String groupId, Set<String> logins, String cluster) {
        LOG.info("[{}] syncMembersInGroup: {} {}  - skipping (read only mode)", cluster, groupId, logins);
    }

    @Override
    public void syncGroupMembersToAllSlaveClusters(Map<String, String> labels) {
        LOG.info("syncGroupMembersToAllSlaveClusters with labels {}", labels);
        Map<String, Set<String>> membersPerGroupIdInMasterCluster;
        try {
            LOG.info("[master] Loading all groups with labels {}...", labels);
            membersPerGroupIdInMasterCluster = ypGroupsClient.getGroupsWithLabels(labels).get();
            LOG.info("[master] Loaded {} groups with labels {}", membersPerGroupIdInMasterCluster.size(), labels);
            if (membersPerGroupIdInMasterCluster.size() == 0) {
                metrics.addYpError();
                LOG.error("Loaded 0 groups from master, skipping groups sync between clusters");
                return;
            }
        } catch (InterruptedException | ExecutionException e) {
            metrics.addYpError();
            LOG.error("YP error while tried to get groups from master cluster", e);
            return;
        }

        if (SYSTEM_IDM_LABEL_VALUE.equals(labels.get(SYSTEM_LABEL_KEY))) {
            metrics.setYpIDMGroupsCount(membersPerGroupIdInMasterCluster.size());
        }

        for (var entry : slaves.entrySet()) {
            String cluster = entry.getKey();
            try {
                LOG.info("[{}] Loading all groups with labels {}...", cluster, labels);
                Map<String, Set<String>> membersInSlaveCluster = entry.getValue().getGroupsClient().getGroupsWithLabels(labels).get();
                LOG.info("[{}] Loaded {} groups with labels {}", cluster, membersInSlaveCluster.size(), labels);

                MapDifference<String, Set<String>> diff = Maps.difference(membersPerGroupIdInMasterCluster, membersInSlaveCluster);
                Map<String, MapDifference.ValueDifference<Set<String>>> groupsDiffering = diff.entriesDiffering();
                Map<String, Set<String>> groupsMissedOnSlave = diff.entriesOnlyOnLeft();
                Map<String, Set<String>> groupsRemovedFromMasterButPresentOnSlave = diff.entriesOnlyOnRight();
                LOG.info("[master -> {}] {} groups different; {} are missed on slave cluster; {} groups are missed on master", cluster,
                        groupsDiffering.size(), groupsMissedOnSlave.size(), groupsRemovedFromMasterButPresentOnSlave.size());

                //syncMembersInGroup - is read only in 'YpServiceReadOnlyImpl' and do real work in 'YpServiceImpl'
                groupsDiffering.forEach((groupId, membersDiff) -> syncMembersInGroup(groupId, membersDiff.leftValue(), cluster));
                groupsMissedOnSlave.forEach((groupId, membersFromMaster) -> syncMembersInGroup(groupId, membersFromMaster, cluster));
            } catch (InterruptedException | ExecutionException e) {
                metrics.addYpError();
                LOG.error("Failed to sync groups from master to " + cluster, e);
            }
        }
    }

    @Override
    public void setGlobalCleanupFlag() {
        LOG.warn("We are going to make global cleanup -> all roles will be revoked !!!");
        globalCleanupStartTimestamp = System.currentTimeMillis();
    }

    private Set<Role> convertTreeToRoles(TreeNode node) {
        return convertTreeToRoles(node, rootConfig, EMPTY_PREFIX);
    }

    static private Set<Role> convertTreeToRoles(TreeNode node, Config config, String prefix) {
        if (node == null) {
            return emptySet();
        }

        String newPrefix = prefix.isEmpty() ? node.getName() : prefix + ROLE_NAME_DELIMITER + node.getName();
        String fullPrefix = newPrefix.isEmpty() ? "" : newPrefix + ROLE_NAME_DELIMITER;
        Set<Role> roles = new HashSet<>();

        if (node.isConvertibleToRole()) {
            // Add non-leaf nodes
            config.getStringList(ROLES_CONFIG_KEY).stream()
                    .map(role -> new Role(fullPrefix, role, "", Role.getLeafUniqueId(node.getUniqueId(), role)))
                    .forEach(roles::add);

            roles.add(new Role(newPrefix, "", node.getOwner(), node.getUniqueId()));
        }

        roles.addAll(node.getChildren().stream()
                .flatMap(child -> convertTreeToRoles(
                        child,
                        config.hasPath(CHILDREN_CONFIG_PATH + child.getName()) ?
                                config.getConfig(CHILDREN_CONFIG_PATH + child.getName()) :
                                config.getConfig(CHILDREN_CONFIG_PATH + DEFAULT_CHILD_NAME),
                        newPrefix).stream())
                .collect(Collectors.toSet())
        );
        return roles;
    }

    protected String getYpGroupName(Role role) {
        return getYpGroupName(role, systemName);
    }

    public static String getYpGroupName(Role role, String groupPrefix) {
        if (StringUtils.isBlank(role.getLeaf())) {
            //Non-leaf roles (nodes with children) shouldn't have linked YP groups
            return null;
        }
        String groupName = role.isNannyRole() ?
                role.getNannyServiceId().orElseThrow() + ROLE_NAME_DELIMITER + role.getLeaf() :
                role.getLevelsJoinedWithDelimiter();
        if(!role.getUniqueId().isEmpty()) {
            String[] splitted = role.getUniqueId().split("\\" + ROLE_NAME_DELIMITER);
            final String uuid = splitted[splitted.length - 1];
            groupName += YP_GROUP_NAME_DELIMITER + uuid;
        }
        return groupPrefix + YP_GROUP_NAME_DELIMITER + groupName;
    }

    //This method is broken for all nanny service roles,
    //should not be used at all (currently used only in admin api)
    @Deprecated
    public static Role getRole(String ypGroupName, String groupPrefix) {
        //cutting "system" prefix
        int startIndex = ypGroupName.indexOf(YP_GROUP_NAME_DELIMITER);
        if (startIndex == -1 || !ypGroupName.startsWith(groupPrefix)) {
            return null;
        }

        //cutting "uniqueId" suffix
        int endIndex = ypGroupName.lastIndexOf(YP_GROUP_NAME_DELIMITER);

        String uuid;
        if (startIndex == endIndex) {
            endIndex = ypGroupName.length();
            uuid = "";
        }
        else {
            uuid = ypGroupName.substring(endIndex+1);
        }

        String role = ypGroupName.substring(startIndex+1, endIndex);
        int lastPoint = role.lastIndexOf(ROLE_NAME_DELIMITER);
        if (lastPoint == -1) {
            return new Role("", role, "", uuid);
        }

        String path = role.substring(0, lastPoint);
        String leaf = role.substring(lastPoint + 1);
        return new Role(
                path,
                leaf,
                "",
                uuid.isEmpty() ? "" : leaf + ROLE_NAME_DELIMITER + uuid
        );
    }

    private YpObjectRepository<?,?,?> getYpObjectRepository(RolesInfo.LevelName level) {
        switch (level) {
            case PROJECT: return master.getProjectRepository();
            case STAGE: return master.getStageRepository();
            case NANNY_SERVICE: return master.getNannyServiceRepository();
            default:
                throw new UnsupportedOperationException("Unsupported role level type: " + level.getName());
        }
    }

    @Override
    public Map<String, Set<String>> getRoleMembers(RolesInfo.LevelName level, String objectId) {
        try {
            return getYpObjectRepository(level)
                    .getObject(objectId, META)
                    .thenApply(optionalYpObject -> {
                        var futures = optionalYpObject.orElseThrow().getMeta().getAcl().getEntries()
                                .stream()
                                .flatMap(ace -> ace.getSubjectsList().stream())
                                .distinct()
                                .filter(subject -> subject.startsWith(systemName + YP_GROUP_NAME_DELIMITER) && getRole(subject, systemName) != null)
                                .map(groupId -> Tuple2.tuple(getRole(groupId, systemName).getLeaf(),
                                        master.getGroupRepository().getObject(groupId, SPEC)))
                                .collect(Collectors.toList());
                        return futures.stream().collect(Collectors.toMap(t -> t._1, t -> {
                            try {
                                return (Set<String>)t._2.get().map(ypObject -> new HashSet<>(ypObject.getSpec().getMembersList())).orElseThrow();
                            } catch (InterruptedException|ExecutionException e) {
                                throw new RuntimeException(e);
                            }
                        }));
                    }).get();
        } catch (InterruptedException | ExecutionException e) {
            metrics.addYpError();
            LOG.error("Failed to read {} '{}' from YP", level.getName(), objectId, e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getProjectIdFor(String nannyServiceId) {
        try {
            return master.getNannyServiceRepository()
                    .getObject(nannyServiceId, META)
                    .thenApply(object -> object.map(ypObject -> ypObject.getMeta().getProjectId())
                            .orElseThrow(() -> new MissedYpObjectException(nannyServiceId, YpObjectType.NANNY_SERVICE)))
                    .thenApply(projectId -> StringUtils.defaultIfEmpty(projectId, null))
                    .get();
        } catch (InterruptedException | ExecutionException e) {
            if (e.getCause() instanceof MissedYpObjectException) {
                throw (MissedYpObjectException)e.getCause();
            }
            metrics.addYpError();
            LOG.error("Failed to read nanny_service '{}' from YP", nannyServiceId, e);
            throw new RuntimeException(e);
        }
    }

    @Override
    public List<String> getNannyServices(String projectId) {
        YpSelectStatement.Builder builder = YpSelectStatement.builder(YpObjectType.NANNY_SERVICE, YpPayloadFormat.YSON)
                .addSelector("/meta/id")
                .setFilter("[/meta/project_id] = '" + projectId + "'");
        try {
            return YpRequestWithPaging.selectObjects(master.getNannyServiceRepository().getRawClient(),
                    LabelBasedRepository.REQUEST_IDS_PAGE_SIZE,
                    builder,
                    payloads -> payloadToYson(payloads.get(0)).stringValue()).get();
        } catch (InterruptedException | ExecutionException e) {
            metrics.addYpError();
            LOG.error("Failed to read nanny_services from YP", e);
            throw new RuntimeException(e);
        }
    }

}
