import collections
import os.path
from time import sleep
from Queue import Queue
from threading import Thread

import traceback

import six
import click
from sepelib.core import config

from awacsctl2.lib.cliutil import get_diff, indent
from awacsctl2.lib.compileutil import configure_inject
from awacsctl2.model import NamespaceHolder
from .client import AwacsClient


DEFAULT_CONFIG_PATH = '~/.awacsctl.cfg'
DEFAULT_CONFIG = {}


class InvalidConfig(Exception):
    pass


class Ctl(object):
    def __init__(self, config_path=DEFAULT_CONFIG_PATH):
        self.verbose = False
        config_path = os.path.expanduser(config_path)
        if os.path.exists(config_path):
            config.load(config_path)
        else:
            config.load()
        config.merge(DEFAULT_CONFIG)
        if not self.get_config('awacs_url'):
            raise InvalidConfig('Missing config option "awacs_url" in awacstl.cfg')

        if not self.get_config('awacs_token'):
            raise InvalidConfig('Missing config option "awacs_token" in awacsctl.cfg')

    def set_config(self, key, value):
        config.set_value(key, value)

    def get_config(self, key, default=None):
        return config.get_value(key, default=default)

    def get_awacs_client(self):
        url = self.get_config('awacs_url')
        token = self.get_config('awacs_token')
        return AwacsClient(base_url=url, token=token)


pass_ctl = click.make_pass_decorator(Ctl)

END = object()


def error(msg, exception=None):
    click.echo(click.style(msg, fg='red'))
    if exception is not None:
        traceback.print_exc(exception)


class LoggerThread(Thread):
    def __init__(self, q):
        self.q = q
        super(LoggerThread, self).__init__()

    def run(self):
        while 1:
            event = self.q.get()
            if event is END:
                return
            if event.is_success():
                color = 'green'
            elif event.is_failure():
                color = 'red'
            else:
                color = None
            click.echo(click.style(six.text_type(event), fg=color))


def get_namespace_dir(namespace_id):
    return os.path.join('.', namespace_id)


def list_namespace_dirs(namespaces_arg):
    namespace_dirs = collections.OrderedDict()

    if not namespaces_arg:
        namespace_dir = '.'
        namespace_id = os.path.basename(os.getcwd())
        namespace_dirs[namespace_id] = namespace_dir
    else:
        for namespace_id in namespaces_arg:
            namespace_dirs[namespace_id] = get_namespace_dir(namespace_id)

    return namespace_dirs


def do_push(namespace_id, namespace_dir, client, q):
    try:
        awacs_namespace = NamespaceHolder.from_awacs(client, namespace_id, q=q)
        fs_namespace = NamespaceHolder.from_fs(namespace_id, namespace_dir, q=q)
    except:
        return

    if awacs_namespace:
        missing_fs_balancers = set(awacs_namespace.balancers) - set(fs_namespace.balancers)
        if missing_fs_balancers:
            error('The following balancers are present in awacs '
                  'but missing from the namespace directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(missing_fs_balancers))))
            return
        missing_fs_upstreams = set(awacs_namespace.upstreams) - set(fs_namespace.upstreams)
        if missing_fs_upstreams:
            error('The following upstreams are present in awacs '
                  'but missing from the namespace directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(missing_fs_upstreams))))
            return
        missing_fs_backends = set(awacs_namespace.backends) - set(fs_namespace.backends)
        if missing_fs_backends:
            error('The following backends are present in awacs '
                  'but missing from the namespace directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(missing_fs_backends))))
            return

        not_matching_balancer_ids = set()
        fs_balancer_versions = awacs_namespace.balancer_versions
        awacs_balancer_versions = awacs_namespace.balancer_versions
        for balancer_id, awacs_version in awacs_balancer_versions.iteritems():
            if fs_balancer_versions[balancer_id] != awacs_version:
                not_matching_balancer_ids.add(balancer_id)
        if not_matching_balancer_ids:
            error('Current versions of the following balancers do not match '
                  'versions written in the directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(not_matching_balancer_ids))))
            return

        not_matching_upstream_ids = set()
        fs_upstream_versions = fs_namespace.upstream_versions
        awacs_upstream_versions = awacs_namespace.upstream_versions
        for upstream_id, awacs_version in awacs_upstream_versions.iteritems():
            if fs_upstream_versions[upstream_id] != awacs_version:
                not_matching_upstream_ids.add(upstream_id)
        if not_matching_upstream_ids:
            error('Current versions of the following upstreams do not match '
                  'versions written in the directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(not_matching_upstream_ids))))
            return

        not_matching_backend_ids = set()
        fs_backend_versions = awacs_namespace.backend_versions
        awacs_backend_versions = awacs_namespace.backend_versions
        for backend_id, awacs_version in awacs_backend_versions.iteritems():
            if fs_backend_versions[backend_id] != awacs_version:
                not_matching_backend_ids.add(backend_id)
        if not_matching_backend_ids:
            error('Current versions of the following backends do not match '
                  'versions written in the directory: "{}". '
                  'Please pull first.'.format('", "'.join(sorted(not_matching_backend_ids))))
            return

    updated_awacs_namespace = fs_namespace.to_awacs(client, q=q)  # note: probably partially updated!
    updated_awacs_namespace.write_versions_to_fs(namespace_dir)


@click.group()
@click.option('--config', nargs=2, multiple=True,
              metavar='KEY VALUE', help='Overrides a config key/value pair.')
@click.option('--verbose', '-v', is_flag=True,
              help='Enables verbose mode.')
@click.version_option('0.1')
@click.pass_context
def cli(ctx, config, verbose):
    """Welcome to awacsctl."""
    try:
        ctx.obj = Ctl()
    except Exception as e:
        raise InvalidConfig(e)
    ctx.obj.verbose = verbose
    for key, value in config:
        ctx.obj.set_config(key, value)


@cli.command()
@click.argument('namespace')
@pass_ctl
def init(ctl, namespace):
    """Initializes a namespace."""
    namespace_dir = os.path.join('.', namespace.strip('/'))
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        NamespaceHolder.create_layout(namespace_dir)
    except Exception as e:
        error('Exception raised when init namespace', e)
    finally:
        q.put(END)


@cli.command()
@click.argument('namespace')
@pass_ctl
def clone(ctl, namespace):
    """Clones a namespace."""
    namespace_id = namespace.strip('/')

    client = ctl.get_awacs_client()
    namespace_dir = os.path.join('.', namespace)
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        NamespaceHolder.create_layout(namespace_dir)
        namespace = NamespaceHolder.from_awacs(client, namespace_id, q=q)
        if namespace is None:
            error('Namespace {} does not exist'.format(namespace_id))
            return
        namespace.to_fs(namespace_dir, q)
    except Exception as e:
        error('Exception raised when clone namespace', e)
    finally:
        q.put(END)


@cli.command()
@click.argument('namespace')
@click.option('--outdir')
@click.option('--templates-dir')
@pass_ctl
def compile(ctl, namespace, outdir, templates_dir):
    namespace_id = namespace.strip('/')
    namespace_dir = os.path.join('.', namespace)
    if outdir is None:
        outdir = os.path.join(namespace_dir, 'bundle')

    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        configure_inject()
        namespace = NamespaceHolder.from_fs(namespace_id, namespace_dir, q)
        namespace.compile(ctl, outdir, templates_dir, q)
    except Exception as e:
        raise
        error(e.message)
    finally:
        q.put(END)


@cli.command()
@click.argument('namespaces', nargs=-1)
@click.option('--override', is_flag=True)
@pass_ctl
def pull(ctl, namespaces, override):
    namespaces = [n.strip('/') for n in namespaces]
    namespaces_to_pull = list_namespace_dirs(namespaces)
    client = ctl.get_awacs_client()
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        for namespace_id, namespace_dir in namespaces_to_pull.iteritems():
            namespace = NamespaceHolder.from_awacs(client, namespace_id, q=q)
            namespace.to_fs(namespace_dir, for_merge=not override, q=q)
    except Exception as e:
        error(e)
    finally:
        q.put(END)


@cli.command()
@click.argument('namespaces', nargs=-1)
@pass_ctl
def push(ctl, namespaces):
    namespaces = [n.strip('/') for n in namespaces]
    namespaces_to_push = list_namespace_dirs(namespaces)
    client = ctl.get_awacs_client()
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        for namespace_id, namespace_dir in namespaces_to_push.iteritems():
            do_push(namespace_id, namespace_dir, client, q)
    except Exception as e:
        raise
        error(e)
    finally:
        q.put(END)


def do_merge(from_dir, from_id, to_dir):
    from_namespace = NamespaceHolder.from_fs('xxx', from_dir)
    for attr in ('balancers', 'upstreams', 'backends'):
        from_entities = set(getattr(from_namespace, attr))
        for id_ in from_entities:
            with open(os.path.join(from_dir, attr, id_ + '.yml')) as f:
                from_yml = f.read()
            to_path = os.path.join(to_dir, attr, id_ + '.yml')
            suffix = ''
            if os.path.exists(to_path):
                with open(to_path) as f:
                    to_yml = f.read()
                if from_yml == to_yml:
                    continue
                else:
                    suffix = '.CONFLICT-with-{}'.format(from_id.replace('.', '-'))

            with open(to_path + suffix, 'w') as f:
                f.write(from_yml)


@cli.command()
@click.argument('namespaces', nargs=-1)
@click.option('--target-namespace')
@pass_ctl
def merge(ctl, namespaces, target_namespace):
    namespaces = [n.strip('/') for n in namespaces]
    target_namespace = target_namespace.strip('/')
    namespaces_to_merge = list_namespace_dirs(namespaces)
    target_namespace_dir = get_namespace_dir(target_namespace)
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        if os.path.exists(target_namespace_dir):
            error('{} already exists'.format(target_namespace_dir))
            return
        else:
            NamespaceHolder.create_layout(target_namespace_dir)
        for namespace_id, namespace_dir in sorted(namespaces_to_merge.items()):
            click.echo('Merging {}...'.format(namespace_id))
            do_merge(namespace_dir, namespace_id, target_namespace_dir)
        click.echo(click.style(
            'Merge is successful. '
            'Please do not forget to edit default values in {}.'.format(
                os.path.join(target_namespace_dir, 'namespace.yml')), 'green'))
    except Exception as e:
        error(e)
    finally:
        q.put(END)


def do_diff(from_namespace_id, to_namespace_id, full, q):
    from_namespace_dir = get_namespace_dir(from_namespace_id)
    to_namespace_dir = get_namespace_dir(to_namespace_id)
    from_namespace = NamespaceHolder.from_fs(from_namespace_id, from_namespace_dir, q=q)
    to_namespace = NamespaceHolder.from_fs(to_namespace_id, to_namespace_dir, q=q)

    sleep(.1)

    lines = [click.style('Difference between {} and {}'.format(from_namespace_dir, to_namespace_dir), fg='blue')]

    no_diff_prefix = '    * '
    diff_prefix = '    ! '

    for attr in ('balancers', 'upstreams', 'backends'):
        from_entities = set(getattr(from_namespace, attr))
        to_entities = set(getattr(to_namespace, attr))

        common_ids = sorted(from_entities & to_entities)
        if common_ids:
            lines.append('  Common {}:'.format(attr))
            for id_ in common_ids:
                with open(os.path.join(from_namespace_dir, attr, id_ + '.yml')) as f:
                    from_yml = f.read()
                with open(os.path.join(to_namespace_dir, attr, id_ + '.yml')) as f:
                    to_yml = f.read()
                if from_yml != to_yml:
                    line = diff_prefix + id_ + ' '
                    lines_to_append = [click.style(line, 'red')]
                    if full:
                        lines_to_append.append('')
                        lines_to_append.append(indent(get_diff(from_yml, to_yml), '        '))
                else:
                    line = no_diff_prefix + id_ + ' '
                    lines_to_append = [line]
                lines.extend(lines_to_append)

        from_extra_ids = sorted(from_entities - to_entities)
        if from_extra_ids:
            lines.append('  Extra {} in {}:'.format(attr, from_namespace_id))
            for id_ in from_extra_ids:
                lines.append(click.style(diff_prefix + id_, 'red'))

        to_extra_ids = sorted(to_entities - from_entities)
        if to_extra_ids:
            lines.append('  Extra {} in {}:'.format(attr, to_namespace_id))
            for id_ in to_extra_ids:
                lines.append(click.style(diff_prefix + id_, 'red'))

    click.echo('\n'.join(lines))


@cli.command()
@click.argument('from_namespace_id')
@click.argument('to_namespace_id')
@click.option('--full', is_flag=True)
@pass_ctl
def diff(ctl, from_namespace_id, to_namespace_id, full=False):
    from_namespace_id = from_namespace_id.strip('/')
    to_namespace_id = to_namespace_id.strip('/')
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        do_diff(from_namespace_id, to_namespace_id, full, q)
    except Exception as e:
        error(e)
    finally:
        q.put(END)


@cli.command()
@click.argument('namespace')
@click.option('--outdir')
@click.option('--templates-dir')
@pass_ctl
def digest(ctl, namespace, outdir, templates_dir):
    namespace_id = namespace.strip('/')
    namespace_dir = os.path.join('.', namespace)
    if outdir is None:
        outdir = os.path.join(namespace_dir, 'digest')
    q = Queue()
    lt = LoggerThread(q)
    lt.start()
    try:
        configure_inject()
        namespace = NamespaceHolder.from_fs(namespace_id, namespace_dir, q, ignore_acl=True)
        namespace.digest(ctl, outdir, templates_dir, q)
    except Exception:
        raise
    finally:
        q.put(END)
