from six.moves.urllib import parse as urlparse

import nanny_rpc_client
from infra.awacs.proto import api_pb2, api_stub


class AwacsClient(object):
    def __init__(self, base_url, token):
        rpc_url = urlparse.urljoin(base_url, 'api/')
        awacs_rpc = nanny_rpc_client.RequestsRpcClient(rpc_url=rpc_url, oauth_token=token)
        self.namespace_service_stub = api_stub.NamespaceServiceStub(awacs_rpc)
        self.balancer_service_stub = api_stub.BalancerServiceStub(awacs_rpc)
        self.upstream_service_stub = api_stub.UpstreamServiceStub(awacs_rpc)
        self.backend_service_stub = api_stub.BackendServiceStub(awacs_rpc)

    # ========== #
    # namespaces #
    # ========== #

    def create_namespace(self, id, category, abc_service_id, auth_pb):
        """
        :type id: str
        :type category: str
        :type abc_service_id: int
        :type auth_pb: awacs_api.model_pb2.Auth
        """
        req_pb = api_pb2.CreateNamespaceRequest()
        req_pb.meta.id = id
        req_pb.meta.category = category
        req_pb.meta.abc_service_id = abc_service_id
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.namespace_service_stub.create_namespace(req_pb)

    def update_namespace(self, id, category, abc_service_id, auth_pb):
        """
        :type id: str
        :type category: str
        :type abc_service_id: int | None
        :type auth_pb: awacs_api.model_pb2.Auth
        """
        req_pb = api_pb2.UpdateNamespaceRequest()
        req_pb.meta.id = id
        req_pb.meta.category = category
        if abc_service_id is not None:
            req_pb.meta.abc_service_id = abc_service_id
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.namespace_service_stub.update_namespace(req_pb)

    def remove_namespace(self, id):
        """
        :type id: str
        """
        raise NotImplementedError()
        # req_pb = api_pb2.RemoveNamespaceRequest(id=id)
        # return self.namespace_service_stub.remove_namespace(req_pb)

    def get_namespace(self, id):
        req = api_pb2.GetNamespaceRequest(id=id)
        return self.namespace_service_stub.get_namespace(req).namespace

    def list_namespaces(self):
        req = api_pb2.ListNamespacesRequest()
        return self.namespace_service_stub.list_namespaces(req).namespaces

    # balancers
    # =========

    def list_balancers(self, namespace_id):
        """
        :type namespace_id: str
        """
        req = api_pb2.ListBalancersRequest(namespace_id=namespace_id)
        return self.balancer_service_stub.list_balancers(req).balancers

    def get_balancer(self, namespace_id, id):
        """
        :type namespace_id: str
        :type id: str
        """
        req = api_pb2.GetBalancerRequest(id=id, namespace_id=namespace_id)
        return self.balancer_service_stub.get_balancer(req).balancer

    def create_balancer(self, namespace_id, id, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.BalancerSpec
        :type comment: str
        """
        req_pb = api_pb2.CreateBalancerRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.balancer_service_stub.create_balancer(req_pb)

    def update_balancer(self, namespace_id, id, version, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.BalancerSpec
        :type comment: str
        """
        req_pb = api_pb2.UpdateBalancerRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.version = version
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.balancer_service_stub.update_balancer(req_pb)

    def remove_balancer(self, namespace_id, id, version):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        """
        req_pb = api_pb2.RemoveBalancerRequest(id=id, namespace_id=namespace_id, version=version)
        return self.balancer_service_stub.remove_balancer(req_pb)

    # upstreams
    # =========

    def list_upstreams(self, namespace_id):
        """
        :type namespace_id: str
        """
        req = api_pb2.ListUpstreamsRequest(namespace_id=namespace_id)
        return self.upstream_service_stub.list_upstreams(req).upstreams

    def get_upstream(self, namespace_id, id):
        """
        :type namespace_id: str
        :type id: str
        """
        req = api_pb2.GetUpstreamRequest(id=id, namespace_id=namespace_id)
        return self.upstream_service_stub.get_upstream(req).upstream

    def create_upstream(self, namespace_id, id, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.UpstreamSpec
        :type comment: str
        """
        req_pb = api_pb2.CreateUpstreamRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.upstream_service_stub.create_upstream(req_pb)

    def update_upstream(self, namespace_id, id, version, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.UpstreamSpec
        :type comment: str
        """
        req_pb = api_pb2.UpdateUpstreamRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.version = version
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.upstream_service_stub.update_upstream(req_pb)

    def remove_upstream(self, namespace_id, id, version):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        """
        req_pb = api_pb2.RemoveUpstreamRequest(id=id, namespace_id=namespace_id, version=version)
        return self.upstream_service_stub.remove_upstream(req_pb)

    # backends
    # =========

    def list_backends(self, namespace_id, exclude_system=False):
        """
        :type namespace_id: str
        """
        req = api_pb2.ListBackendsRequest(namespace_id=namespace_id, query=api_pb2.ListBackendsQuery(exclude_system=exclude_system))
        return self.backend_service_stub.list_backends(req).backends

    def get_backend(self, namespace_id, id):
        """
        :type namespace_id: str
        :type id: str
        """
        req = api_pb2.GetBackendRequest(id=id, namespace_id=namespace_id)
        return self.backend_service_stub.get_backend(req).backend

    def create_backend(self, namespace_id, id, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.BackendSpec
        :type comment: str
        """
        req_pb = api_pb2.CreateBackendRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.backend_service_stub.create_backend(req_pb)

    def update_backend(self, namespace_id, id, version, auth_pb, spec_pb, comment=''):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        :type auth_pb: awacs_api.model_pb2.Auth
        :type spec_pb: awacs_api.model_pb2.BackendSpec
        :type comment: str
        """
        req_pb = api_pb2.UpdateBackendRequest(spec=spec_pb)
        req_pb.meta.id = id
        req_pb.meta.namespace_id = namespace_id
        req_pb.meta.version = version
        req_pb.meta.comment = comment
        req_pb.meta.auth.CopyFrom(auth_pb)
        return self.backend_service_stub.update_backend(req_pb)

    def remove_backend(self, namespace_id, id, version):
        """
        :type namespace_id: str
        :type id: str
        :type version: str
        """
        req_pb = api_pb2.RemoveBackendRequest(id=id, namespace_id=namespace_id, version=version)
        return self.backend_service_stub.remove_backend(req_pb)
