import six


class AwacsEvent(object):
    UPDATE = 0
    UPDATE_FAILURE = 1
    UPDATE_SUCCESS = 2
    CREATE = 3
    CREATE_FAILURE = 4
    CREATE_SUCCESS = 5
    READ = 6
    READ_FAILURE = 7
    READ_SUCCESS = 8

    NAMESPACE = 0
    BALANCER = 1
    BALANCER_LIST = 2
    UPSTREAM = 3
    UPSTREAM_LIST = 4
    BACKEND = 5
    BACKEND_LIST = 6

    def __init__(self, type, obj_type, obj_id=None, namespace_id=None, obj=None, exc=None):
        self.type = type
        self.obj_type = obj_type
        self.namespace_id = namespace_id
        self.obj_id = obj_id
        self.obj = obj
        self.exc = exc

    def is_success(self):
        return self.type in (self.CREATE_SUCCESS, self.UPDATE_SUCCESS, self.READ_SUCCESS)

    def is_failure(self):
        return self.type in (self.CREATE_FAILURE, self.UPDATE_FAILURE, self.READ_FAILURE)

    def __unicode__(self):
        parts = []

        if self.type == self.UPDATE:
            action = 'Updating'
        elif self.type == self.UPDATE_FAILURE:
            action = 'Failed to update'
        elif self.type == self.UPDATE_SUCCESS:
            action = 'Updated'
        elif self.type == self.CREATE:
            action = 'Creating'
        elif self.type == self.CREATE_FAILURE:
            action = 'Failed to create'
        elif self.type == self.CREATE_SUCCESS:
            action = 'Created'
        elif self.type == self.READ:
            action = 'Retrieving'
        elif self.type == self.READ_FAILURE:
            action = 'Failed to retrieve'
        elif self.type == self.READ_SUCCESS:
            action = 'Retrieved'
        else:
            raise RuntimeError()
        parts.append(action)

        if self.obj_type == self.NAMESPACE:
            subj = 'namespace "{}"'.format(self.obj_id)
        elif self.obj_type == self.BALANCER:
            subj = 'balancer "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.BALANCER_LIST:
            subj = 'list of "{}"\'s balancers'.format(self.namespace_id)
        elif self.obj_type == self.UPSTREAM:
            subj = 'upstream "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.UPSTREAM_LIST:
            subj = 'list of "{}"\'s upstreams'.format(self.namespace_id)
        elif self.obj_type == self.BACKEND:
            subj = 'backend "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.BACKEND_LIST:
            subj = 'list of "{}"\'s backends'.format(self.namespace_id)
        else:
            raise RuntimeError()
        parts.append(subj)

        if self.type in (self.CREATE, self.UPDATE, self.READ):
            parts[-1] += '...'
        elif self.type in (self.UPDATE_FAILURE, self.CREATE_FAILURE, self.READ_FAILURE) and self.exc:
            parts[-1] += ':\n'
            parts.append(six.text_type(self.exc))

        return ' '.join(parts)


class FsEvent(object):
    UPDATE = 0
    UPDATE_FAILURE = 1
    UPDATE_SUCCESS = 2
    UPDATE_SKIPPED = 3
    READ = 4
    READ_FAILURE = 5
    READ_SUCCESS = 6

    NAMESPACE = 0
    BALANCER = 1
    UPSTREAM = 2
    BACKEND = 3

    def __init__(self, type, obj_type, obj_id=None, obj_filepath=None, namespace_id=None, exc=None, comment=None):
        self.type = type
        self.obj_type = obj_type
        self.obj_id = obj_id
        self.obj_filepath = obj_filepath
        self.namespace_id = namespace_id
        self.exc = exc
        self.comment = comment

    def is_success(self):
        return self.type in (self.UPDATE_SUCCESS, self.READ_SUCCESS)

    def is_failure(self):
        return self.type in (self.UPDATE_FAILURE, self.READ_FAILURE)

    def __unicode__(self):
        parts = []

        if self.type == self.UPDATE:
            action = 'Updating'
        elif self.type == self.UPDATE_FAILURE:
            action = 'Failed to update'
        elif self.type == self.UPDATE_SUCCESS:
            action = 'Updated'
        elif self.type == self.UPDATE_SKIPPED:
            action = 'Did not update'
        elif self.type == self.READ:
            action = 'Reading'
        elif self.type == self.READ_FAILURE:
            action = 'Failed to read'
        elif self.type == self.READ_SUCCESS:
            action = 'Read'
        else:
            raise RuntimeError()
        parts.append(action)

        if self.obj_filepath:
            parts.append(self.obj_filepath)

        if self.obj_type == self.NAMESPACE:
            subj = 'namespace "{}"'.format(self.obj_id)
        elif self.obj_type == self.BALANCER:
            subj = 'balancer "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.UPSTREAM:
            subj = 'upstream "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.BACKEND:
            subj = 'backend "{}:{}"'.format(self.namespace_id, self.obj_id)
        else:
            raise RuntimeError()
        parts.append('({})'.format(subj))

        if self.type in (self.UPDATE, self.READ):
            parts[-1] += '...'
        elif self.type in (self.UPDATE_FAILURE, self.READ_FAILURE) and self.exc:
            parts[-1] += ':\n'
            parts.append(six.text_type(self.exc))
        elif self.type == self.UPDATE_SKIPPED and self.comment:
            parts[-1] += ':'
            parts.append(self.comment)

        return ' '.join(parts)


class CompileEvent(object):
    COMPILE = 0
    COMPILE_FAILURE = 1
    COMPILE_SUCCESS = 2
    PREPARE = 3
    PREPARE_FAILURE = 4
    PREPARE_SUCCESS = 5
    PARSE = 6
    PARSE_FAILURE = 7
    PARSE_SUCCESS = 8
    COLLECT_BACKENDS = 9
    COLLECT_BACKENDS_FAILURE = 10
    COLLECT_BACKENDS_SUCCESS = 11
    RESOLVE = 12
    RESOLVE_FAILURE = 13
    RESOLVE_SUCCESS = 14
    VALIDATE = 15
    VALIDATE_FAILURE = 16
    VALIDATE_SUCCESS = 17
    GENERATE = 18
    GENERATE_FAILURE = 19
    GENERATE_SUCCESS = 20
    DUMP = 21
    DUMP_FAILURE = 22
    DUMP_SUCCESS = 23

    NAMESPACE = 0
    BALANCER = 1
    UPSTREAM = 2
    BACKEND = 3

    def __init__(self, type, obj_type, obj_id=None, obj_filepath=None, namespace_id=None, exc=None, comment=None):
        self.type = type
        self.obj_type = obj_type
        self.obj_id = obj_id
        self.obj_filepath = obj_filepath
        self.namespace_id = namespace_id
        self.exc = exc
        self.comment = comment

    def is_success(self):
        return self.type in (
            self.COMPILE_SUCCESS, self.PREPARE_SUCCESS, self.PARSE_SUCCESS, self.COLLECT_BACKENDS_SUCCESS,
            self.RESOLVE_SUCCESS, self.VALIDATE_SUCCESS, self.GENERATE_SUCCESS, self.DUMP_SUCCESS,
        )

    def is_failure(self):
        return self.type in (
            self.COMPILE_FAILURE, self.PREPARE_FAILURE, self.PARSE_FAILURE, self.COLLECT_BACKENDS_FAILURE,
            self.RESOLVE_FAILURE, self.VALIDATE_FAILURE, self.GENERATE_FAILURE, self.DUMP_FAILURE,
        )

    def __unicode__(self):
        parts = []
        if self.type == self.COMPILE:
            action = 'Compiling'
        elif self.type == self.COMPILE_FAILURE:
            action = 'Failed to compile'
        elif self.type == self.COMPILE_SUCCESS:
            action = 'Compiled'
        elif self.type == self.PREPARE:
            action = 'Preparing'
        elif self.type == self.PREPARE_FAILURE:
            action = 'Failed to prepare'
        elif self.type == self.PREPARE_SUCCESS:
            action = 'Prepared'
        elif self.type == self.PARSE:
            action = 'Parsing'
        elif self.type == self.PARSE_FAILURE:
            action = 'Failed to parse'
        elif self.type == self.PARSE_SUCCESS:
            action = 'Parsed'
        elif self.type == self.COLLECT_BACKENDS:
            action = 'Collect backends'
        elif self.type == self.COLLECT_BACKENDS_FAILURE:
            action = 'Failed to collect backends'
        elif self.type == self.COLLECT_BACKENDS_SUCCESS:
            action = 'Backends collected'
        elif self.type == self.RESOLVE:
            action = 'Resolving'
        elif self.type == self.RESOLVE_FAILURE:
            action = 'Failed to resolve'
        elif self.type == self.RESOLVE_SUCCESS:
            action = 'Resolved'
        elif self.type == self.VALIDATE:
            action = 'Validating'
        elif self.type == self.VALIDATE_FAILURE:
            action = 'Failed to validate'
        elif self.type == self.VALIDATE_SUCCESS:
            action = 'Validated'
        elif self.type == self.GENERATE:
            action = 'Generating'
        elif self.type == self.GENERATE_FAILURE:
            action = 'Failed to generate'
        elif self.type == self.GENERATE_SUCCESS:
            action = 'Generated'
        elif self.type == self.DUMP:
            action = 'Dumping'
        elif self.type == self.DUMP_FAILURE:
            action = 'Failed to dump'
        elif self.type == self.DUMP_SUCCESS:
            action = 'Dumped'
        else:
            raise RuntimeError()
        parts.append(action)

        if self.obj_filepath:
            parts.append(self.obj_filepath)

        if self.obj_type == self.NAMESPACE:
            subj = 'namespace "{}"'.format(self.obj_id)
        elif self.obj_type == self.BALANCER:
            subj = 'balancer "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.UPSTREAM:
            subj = 'upstream "{}:{}"'.format(self.namespace_id, self.obj_id)
        elif self.obj_type == self.BACKEND:
            subj = 'backend "{}:{}"'.format(self.namespace_id, self.obj_id)
        else:
            raise RuntimeError()
        parts.append('({})'.format(subj))

        if self.type in (self.PREPARE, self.PARSE, self.RESOLVE, self.VALIDATE, self.GENERATE, self.DUMP):
            parts[-1] += '...'
        elif self.is_failure() and self.exc:
            parts[-1] += ':\n'
            parts.append(six.text_type(self.exc))

        return ' '.join(parts)
