import collections
import yaml


def stream_events(yml, loader_cls=yaml.Loader):
    """
    Parses a YAML stream and yields YAML events.

    :type yml: str or a file-like object with `read` method
    :type loader_cls: yaml.Loader | yaml.cyaml.CLoader
    :rtype: iterable[yaml.events.Event]
    """
    loader = loader_cls(yml)
    try:
        while loader.check_event():
            yield loader.get_event()
    finally:
        loader.dispose()


def compute_doc_boundaries(yml, assume_jinja2=False):
    """
    :type yml: str or a file-like object with `read` method
    :returns: a list of tuples (start line number, end line number)
    :rtype: list[tuple[int, int]]
    """
    rv = []
    curr_doc_start = None
    curr_doc_end = None
    if not yml.endswith('\n'):
        yml += '\n'
    try:
        for yaml_event in stream_events(yml):
            if isinstance(yaml_event, yaml.DocumentStartEvent):
                curr_doc_start = yaml_event.start_mark.line
            elif isinstance(yaml_event, yaml.DocumentEndEvent):
                assert curr_doc_start is not None
                curr_doc_end = yaml_event.start_mark.line
                rv.append((curr_doc_start, curr_doc_end))
                curr_doc_start = None
                curr_doc_end = None
    except yaml.YAMLError:
        if assume_jinja2 and len(rv) == 1 and curr_doc_start is not None:
            curr_doc_end = len(yml.splitlines())
            rv.append((curr_doc_start, curr_doc_end))
        else:
            raise
    return rv


def get_docs(yml, assume_jinja2=False):
    """
    :type yml: str
    :rtype: list[str]
    """
    docs = []
    boundaries = compute_doc_boundaries(yml, assume_jinja2=assume_jinja2)
    lines = yml.splitlines()
    for (start, end) in boundaries:
        doc = '\n'.join(lines[start:end])
        if not doc.endswith('\n'):
            doc += '\n'
        docs.append(doc)
    return docs


def represent_odict(dump, tag, mapping, flow_style=None):
    """Like BaseRepresenter.represent_mapping, but does not issue the sort().
    """
    value = []
    node = yaml.MappingNode(tag, value, flow_style=flow_style)
    if dump.alias_key is not None:
        dump.represented_objects[dump.alias_key] = node
    best_style = True
    if hasattr(mapping, 'items'):
        mapping = mapping.items()
    for item_key, item_value in mapping:
        node_key = dump.represent_data(item_key)
        node_value = dump.represent_data(item_value)
        if not (isinstance(node_key, yaml.ScalarNode) and not node_key.style):
            best_style = False
        if not (isinstance(node_value, yaml.ScalarNode) and not node_value.style):
            best_style = False
        value.append((node_key, node_value))
    if flow_style is None:
        if dump.default_flow_style is not None:
            node.flow_style = dump.default_flow_style
        else:
            node.flow_style = best_style
    return node


yaml.SafeDumper.add_representer(collections.OrderedDict,
                                lambda dumper, value: represent_odict(dumper, u'tag:yaml.org,2002:map', value))


def join_docs(yml_1, yml_2):
    if '\n' in yml_2:
        first_non_empty_line, _ = yml_2.lstrip().split('\n', 1)
    else:
        first_non_empty_line = yml_2

    if not yml_1.endswith('\n'):
        yml_1 += '\n'

    if first_non_empty_line.strip() == '---':
        rv = yml_1 + yml_2
    else:
        rv = '{}---\n{}'.format(yml_1, yml_2)
    if not rv.endswith('\n'):
        rv += '\n'
    return rv
