import collections

import requests
import yaml

from infra.awacs.proto import model_pb2
from awacsctl2.lib import yamlutil, jsonutil
from awacsctl2.lib.yamlutil import join_docs
from sepelib.core import config


AuthData = collections.namedtuple('AuthData', ['logins', 'groups'])
BalancerConfigTransportData = collections.namedtuple(
    'BalancerConfigTransportData', ['service_id', 'snapshot_priority', 'gridfs_url', 'instance_tags'])
BalancerValidatorSettingsData = collections.namedtuple(
    'BalancerValidatorSettingsData', ['common_services_balancer_mode_enabled'])
PortData = collections.namedtuple('PortData', ['policy', 'shift', 'override'])
WeightData = collections.namedtuple('WeightData', ['policy', 'override'])


class StaffGroupDoesNotExist(ValueError):
    pass


class AbcServiceDoesNotExist(ValueError):
    pass


def _staff_group_url_to_id(group_url):
    oauth_token = config.get_value('awacs_token')
    url = ('https://api.staff.yandex-team.ru/v3/groups?'
           'url={}&_one=1&_fields=url,id&is_deleted=false,true'.format(group_url))
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token),
    }
    resp = requests.get(url, headers=headers, timeout=5)
    resp.raise_for_status()
    data = resp.json()
    if 'id' in data:
        return str(data['id'])
    else:
        raise StaffGroupDoesNotExist('Staff group with url "{}" does not exist'.format(group_url))


def _staff_group_id_to_url(group_id):
    oauth_token = config.get_value('awacs_token')
    url = ('https://api.staff.yandex-team.ru/v3/groups?'
           'id={}&_one=1&_fields=url,id&is_deleted=false,true'.format(group_id))
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token),
    }
    resp = requests.get(url, headers=headers, timeout=5)
    resp.raise_for_status()
    data = resp.json()
    if 'url' in data:
        return str(data['url'])
    else:
        raise StaffGroupDoesNotExist('Staff group with id "{}" does not exist'.format(group_id))


def _abc_service_url_to_id(abc_service_url):
    """
    :type abc_service_url: str
    :rtype: int
    """
    oauth_token = config.get_value('awacs_token')
    url = 'https://abc-back.yandex-team.ru/api/v4/services/?slug={}&fields=id'.format(abc_service_url)
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token),
    }
    resp = requests.get(url, headers=headers, timeout=5)
    resp.raise_for_status()
    data = resp.json()
    if not data.get('results'):
        raise StaffGroupDoesNotExist('ABC service with slug "{}" does not exist'.format(abc_service_url))
    else:
        return data['results'][0]['id']


def _abc_service_id_to_url(abc_service_id):
    """
    :type abc_service_id: int
    :rtype: str
    """
    oauth_token = config.get_value('awacs_token')
    url = 'https://abc-back.yandex-team.ru/api/v4/services/?id={}&fields=slug'.format(abc_service_id)
    headers = {
        'Authorization': 'OAuth {}'.format(oauth_token),
    }
    resp = requests.get(url, headers=headers, timeout=5)
    resp.raise_for_status()
    data = resp.json()
    if not data.get('results'):
        raise StaffGroupDoesNotExist('ABC service with id "{}" does not exist'.format(abc_service_id))
    else:
        return str(data['results'][0]['slug'])


def _enum_value_to_str(number, enum_desc):
    enum_value = enum_desc.values_by_number.get(number, None)
    if enum_value is None:
        raise ValueError('enum type "{}" has no value with number {}'.format(enum_desc.full_name, number))
    return enum_value.name


def _parse_auth(d):
    auth = d.get('auth')
    if not auth:
        raise ValueError('must contain "auth" field')
    if not isinstance(auth, dict):
        raise ValueError('"auth" must be a dictionary')
    staff = auth.get('staff')
    if not staff:
        raise ValueError('"auth" must contain "staff" field')
    if not isinstance(staff, dict):
        raise ValueError('"auth.staff" must be a dictionary')
    owners = staff.get('owners')
    if not owners:
        raise ValueError('"auth.staff" must contain "owners" field')
    if not isinstance(owners, dict):
        raise ValueError('"auth.staff.owners" must be a dictionary')
    logins = owners.get('logins', [])
    groups = owners.get('groups', [])
    if not logins and not groups:
        raise ValueError('"auth.staff.owners" must contain at least "logins" or "group" field')
    if not isinstance(logins, list):
        raise ValueError('"auth.staff.owners.logins" must be a list')
    if not isinstance(groups, list):
        raise ValueError('"auth.staff.owners.groups" must be a list')
    return AuthData(logins=logins, groups=groups)


def _parse_labels(d):
    labels = d.get('labels', {})
    if not isinstance(labels, dict):
        raise ValueError('"labels" must be a dictionary')
    for k, v in labels.iteritems():
        if not isinstance(k, str) or not isinstance(v, str):
            raise ValueError('"labels" keys and values must be strings')
    return labels


def _resolve_group_urls(groups):
    group_ids = []
    for group in groups:
        try:
            group_id = _staff_group_url_to_id(group)
        except Exception as e:
            raise ValueError('Failed to resolve group "{}": {}'.format(group, e))
        else:
            group_ids.append(group_id)
    return group_ids


def _resolve_group_ids(group_ids):
    group_urls = []
    failed_groups = []
    for group_id in group_ids:
        try:
            group_id = _staff_group_id_to_url(group_id)
        except Exception as e:
            failed_groups.append((group_id, e))
        else:
            group_urls.append(group_id)
    return group_urls, failed_groups


def yml_to_upstream(yml, ignore_acl=False):
    """
    :type yml: str
    :rtype: tuple(model_pb2.Auth, model_pb2.UpstreamSpec)
    """
    docs = yamlutil.get_docs(yml)
    if len(docs) < 2:
        raise ValueError('Upstream YAML must contain exactly 2 documents (separated by "---"), got {}'.format(len(docs)))
    header_yml = docs[0]
    main_yml = docs[-1]

    header = yaml.load(header_yml, Loader=yaml.FullLoader)
    header = jsonutil.unflatten(header, separator='.')

    labels = _parse_labels(header)

    auth_pb = model_pb2.Auth()
    auth_pb.type = model_pb2.Auth.STAFF
    if not ignore_acl:
        auth_data = _parse_auth(header)
        auth_pb.staff.owners.logins.extend(auth_data.logins)
        auth_pb.staff.owners.group_ids.extend(_resolve_group_urls(auth_data.groups))

    spec_pb = model_pb2.UpstreamSpec()
    deleted = header.get('deleted')
    if deleted is not None:
        if not isinstance(deleted, bool):
            raise ValueError('"deleted" must be a boolean')
        spec_pb.deleted = deleted
    spec_pb.type = model_pb2.YANDEX_BALANCER
    spec_pb.labels.update(labels)
    spec_pb.yandex_balancer.yaml = main_yml

    config_mode = header.get('config_mode', 'full')
    if config_mode == 'full':
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.FULL_MODE
    elif config_mode == 'easy':
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.EASY_MODE
    elif config_mode == 'easy2':
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.EASY_MODE2
    elif config_mode == 'l7_fast':
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.L7_FAST_MODE
    elif config_mode == 'l7_fast_sitemap':
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.L7_FAST_SITEMAP_MODE
    else:
        raise ValueError('unknown config mode "{}"'.format(config_mode))

    return auth_pb, spec_pb


def upstream_to_yml(auth_pb, spec_pb):
    """
    :type auth_pb: model_pb2.Auth
    :type spec_pb: model_pb2.UpstreamSpec
    :rtype: str
    """
    header, failed_groups = _auth_pb_to_dict(auth_pb)
    header['labels'] = collections.OrderedDict(sorted(spec_pb.labels.items()))
    if spec_pb.deleted:
        header['deleted'] = spec_pb.deleted
    if spec_pb.yandex_balancer.mode == model_pb2.YandexBalancerUpstreamSpec.FULL_MODE:
        pass
    elif spec_pb.yandex_balancer.mode == model_pb2.YandexBalancerUpstreamSpec.EASY_MODE:
        header['config_mode'] = 'easy'
    elif spec_pb.yandex_balancer.mode == model_pb2.YandexBalancerUpstreamSpec.EASY_MODE2:
        header['config_mode'] = 'easy2'
    elif spec_pb.yandex_balancer.mode == model_pb2.YandexBalancerUpstreamSpec.L7_FAST_MODE:
        header['config_mode'] = 'l7_fast'
    elif spec_pb.yandex_balancer.mode == model_pb2.YandexBalancerUpstreamSpec.L7_FAST_SITEMAP_MODE:
        header['config_mode'] = 'l7_fast_sitemap'
    else:
        raise ValueError('unknown config mode')

    header_yml = yaml.safe_dump(header, default_flow_style=False)
    main_yml = spec_pb.yandex_balancer.yaml
    return join_docs(header_yml, main_yml), failed_groups


def _parse_config_transport(d):
    config_transport = d.get('config_transport')
    if not config_transport:
        raise ValueError('must contain "config_transport" field')
    if not isinstance(config_transport, dict):
        raise ValueError('"config_transport" must be a dictionary')
    nanny_static_file = config_transport.get('nanny_static_file')
    if not nanny_static_file:
        raise ValueError('"config_transport" must contain "nanny_static_file" field')
    if not isinstance(nanny_static_file, dict):
        raise ValueError('"config_transport.nanny_static_file" must be a dictionary')

    service_id = nanny_static_file.get('service_id')
    if not service_id:
        raise ValueError('"config_transport.nanny_static_file" must contain "service_id" field')
    snapshot_priority = nanny_static_file.get('snapshot_priority')
    allowed_priorities = ('NORMAL', 'NONE', 'CRITICAL')
    if snapshot_priority and snapshot_priority not in allowed_priorities:
        raise ValueError('"config_transport.nanny_static_file.snapshot_priority" '
                         'must be one of the following: {}'.format(', '.join(allowed_priorities)))
    gridfs_url = nanny_static_file.get('gridfs_url')
    instance_tags = nanny_static_file.get('instance_tags')
    return BalancerConfigTransportData(
        service_id=service_id,
        snapshot_priority=snapshot_priority,
        gridfs_url=gridfs_url,
        instance_tags=instance_tags
    )


def _parse_validator_settings(d):
    validator_settings = d.get('validator_settings')
    if not validator_settings:
        return
    if not isinstance(validator_settings, dict):
        raise ValueError('"validator_settings" must be a dictionary')
    common_services_balancer_mode_enabled = validator_settings.get('common_services_balancer_mode_enabled')
    if (common_services_balancer_mode_enabled is not None and
            not isinstance(common_services_balancer_mode_enabled, bool)):
        raise ValueError('"validator_settings.common_services_balancer_mode_enabled" must be boolean')
    return BalancerValidatorSettingsData(
        common_services_balancer_mode_enabled=common_services_balancer_mode_enabled
    )


def yml_to_balancer(yml, ignore_acl=False):
    """
    :type yml: str
    :rtype: tuple(model_pb2.Auth, model_pb2.BalancerSpec)
    """
    docs = yamlutil.get_docs(yml, assume_jinja2=True)
    if len(docs) != 2:
        raise ValueError('Balancer YAML must contain exactly 2 documents (separated by "---")')
    header_yml = docs[0]
    main_yml = docs[1]

    header = yaml.load(header_yml, Loader=yaml.FullLoader)
    header = jsonutil.unflatten(header, separator='.')

    config_transport = _parse_config_transport(header)
    validator_settings = _parse_validator_settings(header)

    auth_pb = model_pb2.Auth()
    auth_pb.type = model_pb2.Auth.STAFF
    if not ignore_acl:
        auth_data = _parse_auth(header)
        auth_pb.staff.owners.logins.extend(auth_data.logins)
        auth_pb.staff.owners.group_ids.extend(_resolve_group_urls(auth_data.groups))

    spec_pb = model_pb2.BalancerSpec()
    spec_pb.type = model_pb2.YANDEX_BALANCER
    spec_pb.yandex_balancer.yaml = main_yml
    if 'template_engine' in header:
        spec_pb.yandex_balancer.template_engine = model_pb2.TemplateEngineType.Value(header['template_engine'])
    if 'mode' in header:
        spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerSpec.ConfigMode.Value(header['mode'])

    spec_pb.config_transport.type = model_pb2.NANNY_STATIC_FILE
    spec_pb.config_transport.nanny_static_file.service_id = config_transport.service_id
    if config_transport.gridfs_url is not None:
        spec_pb.config_transport.nanny_static_file.gridfs_url = config_transport.gridfs_url
    if config_transport.snapshot_priority is not None:
        snapshot_priority = getattr(model_pb2.BalancerNannyStaticFileTransportSpec, config_transport.snapshot_priority)
        spec_pb.config_transport.nanny_static_file.snapshot_priority = snapshot_priority
    if config_transport.instance_tags is not None:
        for tag in config_transport.instance_tags:
            setattr(spec_pb.config_transport.nanny_static_file.instance_tags, tag, config_transport.instance_tags[tag])

    if validator_settings and validator_settings.common_services_balancer_mode_enabled is not None:
        spec_pb.validator_settings.common_services_balancer_mode_enabled = \
            validator_settings.common_services_balancer_mode_enabled

    return auth_pb, spec_pb


def _auth_pb_to_dict(auth_pb):
    """
    :type auth_pb: model_pb2.Auth
    """
    groups, failed_groups = _resolve_group_ids(auth_pb.staff.owners.group_ids)
    return collections.OrderedDict([
        ('auth', collections.OrderedDict({
            'staff': collections.OrderedDict({
                'owners': collections.OrderedDict([
                    ('logins', list(auth_pb.staff.owners.logins)),
                    ('groups', groups),
                ]),
            }),
        }))
    ]), failed_groups


def balancer_to_yml(auth_pb, spec_pb):
    """
    :type auth_pb: model_pb2.Auth
    :type spec_pb: model_pb2.BalancerSpec
    :type category: str
    :rtype: str
    """
    nanny_static_file = collections.OrderedDict([
        ('service_id', spec_pb.config_transport.nanny_static_file.service_id),
    ])
    if spec_pb.config_transport.nanny_static_file.gridfs_url:
        nanny_static_file['gridfs_url'] = spec_pb.config_transport.nanny_static_file.gridfs_url
    if spec_pb.config_transport.nanny_static_file.HasField('instance_tags'):
        nanny_static_file['instance_tags'] = {}
        for field_desc, field_value in spec_pb.config_transport.nanny_static_file.instance_tags.ListFields():
            nanny_static_file['instance_tags'][field_desc.name] = field_value
    snapshot_priority = spec_pb.config_transport.nanny_static_file.snapshot_priority
    if snapshot_priority != 0:
        name = model_pb2.BalancerNannyStaticFileTransportSpec.SnapshotPriority.Name(snapshot_priority)
        nanny_static_file['snapshot_priority'] = name

    header, failed_groups = _auth_pb_to_dict(auth_pb)
    header['config_transport'] = collections.OrderedDict([
        ('nanny_static_file', nanny_static_file)
    ])

    if spec_pb.validator_settings.common_services_balancer_mode_enabled:
        header['validator_settings'] = collections.OrderedDict([
            ('common_services_balancer_mode_enabled', spec_pb.validator_settings.common_services_balancer_mode_enabled)
        ])

    template_engine = spec_pb.yandex_balancer.template_engine
    if template_engine != 0:
        header['template_engine'] = model_pb2.TemplateEngineType.Name(template_engine)

    mode = spec_pb.yandex_balancer.mode
    if mode != 0:
        header['mode'] = model_pb2.YandexBalancerSpec.ConfigMode.Name(mode)

    header_yml = yaml.safe_dump(header, default_flow_style=False)
    main_yml = spec_pb.yandex_balancer.yaml
    return join_docs(header_yml, main_yml), failed_groups


def namespace_to_yml(auth_pb, category, abc_service_id=None):
    """
    :type auth_pb: awacs.proto.model_pb2.Auth
    :type category: str
    :type abc_service_id: str
    :rtype: str
    """
    header, failed_groups = _auth_pb_to_dict(auth_pb)
    header['category'] = category
    if abc_service_id:
        header['abc_service'] = _abc_service_id_to_url(abc_service_id)
    return yaml.safe_dump(header, default_flow_style=False), failed_groups


def yml_to_namespace(yml, ignore_acl=False):
    """
    :type yml: str
    :rtype: (awacs.proto.model_pb2.Auth, str, str)
    """
    data = yaml.load(yml, Loader=yaml.FullLoader)
    data = jsonutil.unflatten(data, separator='.')

    category = data.get('category')
    abc_service_id = data.get('abc_service')
    if not ignore_acl and abc_service_id:
        abc_service_id = _abc_service_url_to_id(abc_service_id)

    auth_pb = model_pb2.Auth()
    auth_pb.type = model_pb2.Auth.STAFF
    if not ignore_acl:
        auth_data = _parse_auth(data)
        auth_pb.staff.owners.logins.extend(auth_data.logins)
        auth_pb.staff.owners.group_ids.extend(_resolve_group_urls(auth_data.groups))

    return auth_pb, category, abc_service_id


def _parse_port(d):
    """
    :type d: dict
    :rtype: PortData
    """
    port = d.get('port')
    if not port:
        return None
    if not isinstance(port, dict):
        raise ValueError('"port" must be a dictionary')
    if len(port) > 1 or port.keys()[0] not in ('shift', 'override'):
        raise ValueError('"port" dictionary must contain exactly one field, either "shift" or "override"')
    if 'override' in port:
        if not isinstance(port['override'], int):
            raise ValueError('"port.override" must be an int')
        return PortData(policy='OVERRIDE', override=port['override'], shift=None)
    if 'shift' in port:
        if not isinstance(port['shift'], int):
            raise ValueError('"port.shift" must be an int')
        return PortData(policy='SHIFT', override=None, shift=port['shift'])


def _parse_weight(d):
    """
    :type d: dict
    :rtype: Weight
    """
    weight = d.get('weight')
    if not weight:
        return None
    if not isinstance(weight, dict):
        raise ValueError('"weight" must be a dictionary')
    if len(weight) > 1 or weight.keys()[0] != 'override':
        raise ValueError('"weight" dictionary must contain exactly one field "override"')
    if 'override' in weight:
        if not isinstance(weight['override'], int):
            raise ValueError('"weight.override" must be an int')
        return WeightData(policy='OVERRIDE', override=weight['override'])


def _parse_nanny_snapshot(d, pb):
    """
    :type d: dict
    """
    service_id = d.get('service_id')
    if not service_id:
        raise ValueError('must contain "service_id" field')
    pb.service_id = service_id

    snapshot_id = d.get('snapshot_id')
    if snapshot_id is not None:
        pb.snapshot_id = snapshot_id

    use_mtn = d.get('use_mtn')
    if use_mtn is not None:
        if not isinstance(use_mtn, bool):
            raise ValueError('"use_mtn" must be a boolean')
        pb.use_mtn.value = use_mtn

    port_data = _parse_port(d)
    if port_data:
        pb.port.policy = model_pb2.Port.Policy.Value(port_data.policy)
        if pb.port.policy == pb.port.OVERRIDE:
            pb.port.override = port_data.override
        if pb.port.policy == pb.port.SHIFT:
            pb.port.shift = port_data.shift

    weight_data = _parse_weight(d)
    if weight_data:
        pb.weight.policy = model_pb2.Weight.Policy.Value(weight_data.policy)
        if pb.weight.policy == pb.weight.OVERRIDE:
            pb.weight.override = weight_data.override


def _parse_gencfg_group(d, pb):
    """
    :type d: dict
    """
    name = d.get('name')
    if not name:
        raise ValueError('must contain "name" field')
    pb.name = name

    version = d.get('version')
    if not version:
        raise ValueError('must contain "version" field')
    pb.version = version

    use_mtn = d.get('use_mtn')
    if use_mtn is not None:
        if not isinstance(use_mtn, bool):
            raise ValueError('"use_mtn" must be a boolean')
        pb.use_mtn.value = use_mtn

    port_data = _parse_port(d)
    if port_data:
        pb.port.policy = model_pb2.Port.Policy.Value(port_data.policy)
        if pb.port.policy == pb.port.OVERRIDE:
            pb.port.override = port_data.override
        if pb.port.policy == pb.port.SHIFT:
            pb.port.shift = port_data.shift

    weight_data = _parse_weight(d)
    if weight_data:
        pb.weight.policy = model_pb2.Weight.Policy.Value(weight_data.policy)
        if pb.weight.policy == pb.weight.OVERRIDE:
            pb.weight.override = weight_data.override


def _parse_yp_endpoint_set(d, pb):
    """
    :type d: dict
    """
    cluster = d.get('cluster')
    if not cluster:
        raise ValueError('must contain "cluster" field')
    pb.cluster = cluster

    endpoint_set_id = d.get('endpoint_set_id')
    if endpoint_set_id is not None:
        pb.endpoint_set_id = endpoint_set_id

    port_data = _parse_port(d)
    if port_data:
        pb.port.policy = model_pb2.Port.Policy.Value(port_data.policy)
        if pb.port.policy == pb.port.OVERRIDE:
            pb.port.override = port_data.override
        if pb.port.policy == pb.port.SHIFT:
            pb.port.shift = port_data.shift

    weight_data = _parse_weight(d)
    if weight_data:
        pb.weight.policy = model_pb2.Weight.Policy.Value(weight_data.policy)
        if pb.weight.policy == pb.weight.OVERRIDE:
            pb.weight.override = weight_data.override


def yml_to_backend(yml, ignore_acl=False):
    """
    :type yml: str
    :rtype: tuple(model_pb2.Auth, model_pb2.BackendSpec)
    """
    data = yaml.load(yml, Loader=yaml.FullLoader)
    data = jsonutil.unflatten(data, separator='.')

    labels = _parse_labels(data)
    port_data = _parse_port(data)

    auth_pb = model_pb2.Auth()
    auth_pb.type = model_pb2.Auth.STAFF
    if not ignore_acl:
        auth_data = _parse_auth(data)
        auth_pb.staff.owners.logins.extend(auth_data.logins)
        auth_pb.staff.owners.group_ids.extend(_resolve_group_urls(auth_data.groups))

    spec_pb = model_pb2.BackendSpec()
    deleted = data.get('deleted')
    if deleted is not None:
        if not isinstance(deleted, bool):
            raise ValueError('"deleted" must be a boolean')
        spec_pb.deleted = deleted

    spec_pb.labels.update(labels)
    selector_pb = spec_pb.selector
    if port_data:
        selector_pb.port.policy = model_pb2.Port.Policy.Value(port_data.policy)
        if selector_pb.port.policy == selector_pb.port.OVERRIDE:
            selector_pb.port.override = port_data.override
        if selector_pb.port.policy == selector_pb.port.SHIFT:
            selector_pb.port.shift = port_data.shift

    use_mtn = data.get('use_mtn')
    if use_mtn is not None:
        if not isinstance(use_mtn, bool):
            raise ValueError('"use_mtn" must be a boolean')
        selector_pb.use_mtn = use_mtn

    nanny_snapshots = data.get('nanny_snapshots', [])
    gencfg_groups = data.get('gencfg_groups', [])
    yp_endpoint_sets = data.get('yp_endpoint_sets', [])
    manual = data.get('manual', False)

    n = len(filter(bool, (nanny_snapshots, gencfg_groups, yp_endpoint_sets, manual)))

    if n > 1:
        raise ValueError('at most one of "nanny_snapshots", "gencfg_groups", '
                         '"yp_endpoint_sets" and "manual" can be set')
    elif n == 0:
        raise ValueError('either "nanny_snapshots", "gencfg_groups", '
                         '"yp_endpoint_sets" or "manual" must be set')

    if nanny_snapshots:
        selector_pb.type = selector_pb.NANNY_SNAPSHOTS
        for s in nanny_snapshots:
            s = jsonutil.unflatten(s, separator='.')
            _parse_nanny_snapshot(s, selector_pb.nanny_snapshots.add())

    if gencfg_groups:
        selector_pb.type = selector_pb.GENCFG_GROUPS
        for g in gencfg_groups:
            g = jsonutil.unflatten(g, separator='.')
            _parse_gencfg_group(g, selector_pb.gencfg_groups.add())

    if yp_endpoint_sets:
        selector_pb.type = selector_pb.YP_ENDPOINT_SETS
        for y in yp_endpoint_sets:
            y = jsonutil.unflatten(y, separator='.')
            _parse_yp_endpoint_set(y, selector_pb.yp_endpoint_sets.add())

    if manual:
        selector_pb.type = selector_pb.MANUAL

    return auth_pb, spec_pb


def backend_to_yml(auth_pb, spec_pb):
    """
    :type auth_pb: model_pb2.Auth
    :type spec_pb: model_pb2.BackendSpec
    :rtype: str
    """
    data, failed_groups = _auth_pb_to_dict(auth_pb)
    data['labels'] = collections.OrderedDict(sorted(spec_pb.labels.items()))
    selector_pb = spec_pb.selector
    if selector_pb.port.policy == selector_pb.port.OVERRIDE:
        data['port.override'] = selector_pb.port.override
    if selector_pb.port.policy == selector_pb.port.SHIFT:
        data['port.shift'] = selector_pb.port.shift
    if selector_pb.use_mtn:
        data['use_mtn'] = selector_pb.use_mtn

    if selector_pb.type == selector_pb.NANNY_SNAPSHOTS:
        nanny_snapshots = []
        for snapshot_pb in selector_pb.nanny_snapshots:
            nanny_snapshot = collections.OrderedDict({'service_id': snapshot_pb.service_id})
            if snapshot_pb.snapshot_id:
                nanny_snapshot['snapshot_id'] = snapshot_pb.snapshot_id
            if snapshot_pb.port.policy == snapshot_pb.port.OVERRIDE:
                nanny_snapshot['port.override'] = snapshot_pb.port.override
            if snapshot_pb.port.policy == snapshot_pb.port.SHIFT:
                nanny_snapshot['port.shift'] = snapshot_pb.port.shift
            if snapshot_pb.weight.policy == snapshot_pb.weight.OVERRIDE:
                nanny_snapshot['weight.override'] = snapshot_pb.weight.override
            if snapshot_pb.HasField('use_mtn'):
                nanny_snapshot['use_mtn'] = snapshot_pb.use_mtn.value
            nanny_snapshots.append(nanny_snapshot)
        data['nanny_snapshots'] = nanny_snapshots
    elif selector_pb.type == selector_pb.GENCFG_GROUPS:
        gencfg_groups = []
        for gencfg_group_pb in selector_pb.gencfg_groups:
            gencfg_group = collections.OrderedDict([
                ('name', gencfg_group_pb.name),
                ('version', gencfg_group_pb.version),
            ])
            if gencfg_group_pb.port.policy == gencfg_group_pb.port.OVERRIDE:
                gencfg_group['port.override'] = gencfg_group_pb.port.override
            if gencfg_group_pb.port.policy == gencfg_group_pb.port.SHIFT:
                gencfg_group['port.shift'] = gencfg_group_pb.port.shift
            if gencfg_group_pb.weight.policy == gencfg_group_pb.weight.OVERRIDE:
                gencfg_group['weight.override'] = gencfg_group_pb.weight.override
            if gencfg_group_pb.HasField('use_mtn'):
                gencfg_group['use_mtn'] = gencfg_group_pb.use_mtn.value
            gencfg_groups.append(gencfg_group)
        data['gencfg_groups'] = gencfg_groups
    elif selector_pb.type in (selector_pb.YP_ENDPOINT_SETS, selector_pb.YP_ENDPOINT_SETS_SD):
        yp_endpoint_sets = []
        for yp_endpoint_set_pb in selector_pb.yp_endpoint_sets:
            yp_endpoint_set = collections.OrderedDict([
                ('cluster', yp_endpoint_set_pb.cluster),
                ('endpoint_set_id', yp_endpoint_set_pb.endpoint_set_id),
            ])
            if yp_endpoint_set_pb.port.policy == yp_endpoint_set_pb.port.OVERRIDE:
                yp_endpoint_set['port.override'] = yp_endpoint_set_pb.port.override
            if yp_endpoint_set_pb.port.policy == yp_endpoint_set_pb.port.SHIFT:
                yp_endpoint_set['port.shift'] = yp_endpoint_set_pb.port.shift
            if yp_endpoint_set_pb.weight.policy == yp_endpoint_set_pb.weight.OVERRIDE:
                yp_endpoint_set['weight.override'] = yp_endpoint_set_pb.weight.override
            yp_endpoint_sets.append(yp_endpoint_set)
        data['yp_endpoint_sets'] = yp_endpoint_sets
    elif selector_pb.type == selector_pb.MANUAL:
        data['manual'] = True
    elif selector_pb.type == selector_pb.BALANCERS:
        balancers = []
        for balancer_pb in selector_pb.balancers:
            balancers.append(balancer_pb.id)
        data['balancers'] = balancers
    else:
        t = _enum_value_to_str(selector_pb.type, selector_pb.DESCRIPTOR.fields_by_name['type'].enum_type)
        raise ValueError('Unknown selector type: {}'.format(t))
    if spec_pb.deleted:
        data['deleted'] = spec_pb.deleted
    return yaml.safe_dump(data, default_flow_style=False), failed_groups
