import json
import time
import collections
import os.path
import string
import random
from Queue import Queue
from operator import itemgetter

import yaml
from nanny_rpc_client.exceptions import NotFoundError

from awacsctl2 import marshaller
from awacsctl2.lib.cliutil import format_tree
from awacsctl2.lib.pbutil import clone_pb
from awacsctl2.lib.compileutil import get_nannyclient
from awacsctl2.events import AwacsEvent, FsEvent, CompileEvent

from awacs import yamlparser
from awacs.wrappers.base import Holder
from infra.awacs.proto import modules_pb2, model_pb2
from awacs.model.balancer.vector import BalancerVersion, UpstreamVersion, BackendVersion, EndpointSetVersion
from awacs.model.balancer import generator


def get_current_runtime_attrs_id(service_id):
    nanny_client = get_nannyclient()
    return nanny_client.get_service_instances_section(service_id)[0]


class ConflictError(Exception):
    pass


class NamespaceError(Exception):
    pass


class NamespaceNotFound(Exception):
    pass


class BaseEntity(object):
    def __init__(self, id, version, auth_pb, spec_pb):
        self.id = id
        self.version = version
        self.auth_pb = auth_pb
        self.spec_pb = spec_pb

    @classmethod
    def from_pb(cls, pb):
        return cls(id=(pb.meta.namespace_id, pb.meta.id), version=pb.meta.version,
                   auth_pb=pb.meta.auth, spec_pb=pb.spec)

    def to_yml(self):
        raise NotImplementedError

    def from_yml(self, id, version, yml):
        raise NotImplementedError

    def get_version(self, ctime):
        raise NotImplementedError

    def __str__(self):
        return '\n'.join(('/'.join(self.id), self.version, str(self.auth_pb), str(self.spec_pb)))

    def __eq__(self, other):
        return (
            self.id == other.id and
            self.version == other.version and
            self.auth_pb == other.auth_pb and
            self.spec_pb == other.spec_pb
        )


class Namespace(object):
    def __init__(self, id, auth_pb, category, abc_service_id=None):
        self.id = id
        self.auth_pb = auth_pb
        self.category = category
        self.abc_service_id = abc_service_id

    def to_yml(self):
        """
        :rtype: str
        """
        return marshaller.namespace_to_yml(self.auth_pb, self.category, self.abc_service_id)

    @classmethod
    def from_yml(cls, id, yml, ignore_acl=False):
        """
        :type id: str
        :type yml: str
        :rtype: str
        """
        auth_pb, category, abc_service_id = marshaller.yml_to_namespace(yml, ignore_acl=ignore_acl)
        return cls(id=id, auth_pb=auth_pb, category=category, abc_service_id=abc_service_id)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Namespace
        :rtype: Namespace
        """
        return cls(
            id=pb.meta.id,
            auth_pb=pb.meta.auth,
            category=pb.meta.category,
            abc_service_id=pb.meta.abc_service_id
        )

    def __str__(self):
        return '\n'.join((self.id, self.category, str(self.auth_pb)))

    def __eq__(self, other):
        return (
            self.id == other.id and
            self.auth_pb == other.auth_pb and
            self.category == other.category
        )


class Balancer(BaseEntity):
    def to_yml(self):
        return marshaller.balancer_to_yml(self.auth_pb, self.spec_pb)

    @classmethod
    def from_yml(cls, id, version, yml, ignore_acl=False):
        """
        :type id: (str, str)
        :type version: str
        :type yml: str
        :rtype: Balancer
        """
        assert isinstance(id, tuple) and len(id) == 2
        auth_pb, spec_pb = marshaller.yml_to_balancer(yml, ignore_acl=ignore_acl)
        return cls(id=id, version=version, auth_pb=auth_pb, spec_pb=spec_pb)

    @classmethod
    def from_pb(cls, pb):
        """
        :type pb: awacs.proto.model_pb2.Balancer
        :rtype: Balancer
        """
        return cls(
            id=(pb.meta.namespace_id, pb.meta.id),
            version=pb.meta.version,
            auth_pb=pb.meta.auth,
            spec_pb=pb.spec
        )

    def get_version(self, ctime):
        """
        :param int ctime: time in microseconds
        :rtype: BalancerVersion
        """
        return BalancerVersion(ctime, self.id, self.version)

    def __eq__(self, other):
        if not isinstance(other, Balancer):
            return False

        # normalize specs
        spec_pb = clone_pb(self.spec_pb)
        other_spec_pb = clone_pb(other.spec_pb)
        for pb in (spec_pb, other_spec_pb):
            if pb.yandex_balancer.HasField('config'):
                pb.yandex_balancer.ClearField('config')
            pb.validator_settings.SetInParent()

        return (
            self.id == other.id and
            self.version == other.version and
            self.auth_pb == other.auth_pb and
            spec_pb == other_spec_pb
        )


class Upstream(BaseEntity):
    def to_yml(self):
        return marshaller.upstream_to_yml(self.auth_pb, self.spec_pb)

    @classmethod
    def from_yml(cls, id, version, yml, ignore_acl=False):
        """
        :type id: (str, str)
        :type version: str
        :type yml: str
        :rtype: Upstream
        """
        assert isinstance(id, tuple) and len(id) == 2
        auth_pb, spec_pb = marshaller.yml_to_upstream(yml, ignore_acl=ignore_acl)
        return cls(id=id, version=version, auth_pb=auth_pb, spec_pb=spec_pb)

    def get_version(self, ctime):
        """
        :param int ctime: time in microseconds
        :rtype: UpstreamVersion
        """
        return UpstreamVersion(ctime, self.id, self.version, self.spec_pb.deleted)

    def __eq__(self, other):
        if not isinstance(other, Upstream):
            return False

        if self.spec_pb.yandex_balancer.HasField('config'):
            spec_pb = clone_pb(self.spec_pb)
            spec_pb.yandex_balancer.ClearField('config')
        else:
            spec_pb = self.spec_pb

        if other.spec_pb.yandex_balancer.HasField('config'):
            other_spec_pb = clone_pb(other.spec_pb)
            other_spec_pb.yandex_balancer.ClearField('config')
        else:
            other_spec_pb = other.spec_pb

        return (
            self.id == other.id and
            self.version == other.version and
            self.auth_pb == other.auth_pb and
            spec_pb == other_spec_pb
        )


class Backend(BaseEntity):
    def to_yml(self):
        return marshaller.backend_to_yml(self.auth_pb, self.spec_pb)

    @classmethod
    def from_yml(cls, id, version, yml, ignore_acl=False):
        """
        :type id: (str, str)
        :type version: str
        :type yml: str
        :rtype: Backend
        """
        assert isinstance(id, tuple) and len(id) == 2
        auth_pb, spec_pb = marshaller.yml_to_backend(yml, ignore_acl=ignore_acl)
        return cls(id=id, version=version, auth_pb=auth_pb, spec_pb=spec_pb)

    def get_version(self, ctime):
        """
        :param int ctime: time in microseconds
        :rtype: BackendVersion
        """
        return BackendVersion(ctime, self.id, self.version, self.spec_pb.deleted)


def get_default_namespace_content():
    return '''auth:
  staff:
    owners:
      logins:
      - {}
      groups: []
category: common
# abc_service: FILLME'''.format(os.getenv('LOGNAME'))


def get_default_awacs_versions_content():
    return '''versions:
    balancers: {}
    upstreams: {}
    backends: {}'''


class NamespaceHolder(object):
    def __init__(self, id, namespace, balancers, upstreams, backends):
        self.id = id
        self.namespace = namespace
        self.balancers = collections.OrderedDict(sorted(balancers.items()))
        self.upstreams = collections.OrderedDict(sorted(upstreams.items()))
        self.backends = collections.OrderedDict(sorted(backends.items()))

    def to_tree_str(self):
        tree = (
            self.id, [
                ('balancers', [(balancer_id, []) for balancer_id in sorted(self.balancers)]),
                ('upstreams', [(upstream_id, []) for upstream_id in sorted(self.upstreams)]),
                ('backends', [(backend_id, []) for backend_id in sorted(self.backends)]),
            ]
        )
        return format_tree(
            tree, format_node=itemgetter(0), get_children=itemgetter(1))

    @classmethod
    def create_layout(cls, dir):
        if os.path.exists(dir):
            raise ConflictError('{} already exists'.format(dir))
        else:
            os.makedirs(dir)
        for path in ('balancers', 'upstreams', 'backends', '.awacs'):
            fullpath = os.path.join(dir, path)
            if os.path.exists(fullpath):
                raise ConflictError('{} is not empty, {} already exists'.format(dir, fullpath))
            os.makedirs(fullpath)
        with open(os.path.join(dir, 'namespace.yml'), 'w') as f:
            f.write(get_default_namespace_content())
        with open(os.path.join(dir, '.awacs', 'versions'), 'w') as f:
            f.write(get_default_awacs_versions_content())

    @classmethod
    def from_awacs(cls, client, namespace_id, q=None):
        if q is None:
            q = Queue()
        ev = AwacsEvent

        q.put(ev(type=ev.READ, obj_type=ev.NAMESPACE, obj_id=namespace_id))
        try:
            try:
                namespace_pb = client.get_namespace(namespace_id)
            except NotFoundError:
                return None
            namespace = Namespace.from_pb(namespace_pb)

            balancers = {}
            q.put(ev(type=ev.READ, obj_type=ev.BALANCER_LIST, namespace_id=namespace_id))
            try:
                balancer_pbs = client.list_balancers(namespace_id)
            except Exception as e:
                q.put(ev(type=ev.READ_FAILURE, obj_type=ev.BALANCER_LIST, namespace_id=namespace_id, exc=e))
                raise
            else:
                q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.BALANCER_LIST, namespace_id=namespace_id))

            for balancer_pb in balancer_pbs:
                balancer = Balancer.from_pb(balancer_pb)
                full_balancer_id = balancer.id
                assert full_balancer_id[0] == namespace_id
                balancers[full_balancer_id[1]] = balancer

            upstreams = {}
            q.put(ev(type=ev.READ, obj_type=ev.UPSTREAM_LIST, namespace_id=namespace_id))
            try:
                upstream_pbs = client.list_upstreams(namespace_id)
            except Exception as e:
                q.put(ev(type=ev.READ_FAILURE, obj_type=ev.UPSTREAM_LIST, namespace_id=namespace_id, exc=e))
                raise
            else:
                q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.UPSTREAM_LIST, namespace_id=namespace_id))
            for upstream_pb in upstream_pbs:
                upstream = Upstream.from_pb(upstream_pb)
                full_upstream_id = upstream.id
                assert full_upstream_id[0] == namespace_id
                upstreams[full_upstream_id[1]] = upstream

            backends = {}
            q.put(ev(type=ev.READ, obj_type=ev.BACKEND_LIST, namespace_id=namespace_id))
            try:
                backend_pbs = client.list_backends(namespace_id, exclude_system=True)
            except Exception as e:
                q.put(ev(type=ev.READ_FAILURE, obj_type=ev.BACKEND_LIST, namespace_id=namespace_id, exc=e))
                raise
            else:
                q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.BACKEND_LIST, namespace_id=namespace_id))
            for backend_pb in backend_pbs:
                backend = Backend.from_pb(backend_pb)
                full_backend_id = backend.id
                assert full_backend_id[0] == namespace_id
                backends[full_backend_id[1]] = backend
        except Exception as e:
            q.put(ev(type=ev.READ_FAILURE, obj_type=ev.NAMESPACE, obj_id=namespace_id, exc=e))
            raise
        else:
            q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.NAMESPACE, obj_id=namespace_id))
            return NamespaceHolder(namespace_id, namespace, balancers, upstreams, backends)

    def to_awacs(self, client, comment='changed using awacsctl', q=None):
        if q is None:
            q = Queue()
        ev = AwacsEvent

        q.put(ev(type=ev.UPDATE, obj_type=ev.NAMESPACE, obj_id=self.id))
        updated_namespace = None
        updated_balancers = {}
        updated_upstreams = {}
        updated_backends = {}
        try:
            try:
                namespace_pb = client.update_namespace(
                    id=self.namespace.id,
                    category=self.namespace.category,
                    abc_service_id=self.namespace.abc_service_id,
                    auth_pb=self.namespace.auth_pb
                ).namespace
            except NotFoundError:
                if not self.namespace.abc_service_id:
                    raise ValueError('"abc_service" must be set')
                namespace_pb = client.create_namespace(
                    id=self.namespace.id,
                    category=self.namespace.category,
                    abc_service_id=self.namespace.abc_service_id,
                    auth_pb=self.namespace.auth_pb
                ).namespace

            updated_namespace = Namespace.from_pb(namespace_pb)

            for balancer in self.balancers.itervalues():
                is_balancer_new = balancer.version is None
                q.put(ev(type=ev.UPDATE if not is_balancer_new else ev.CREATE,
                         obj_type=ev.BALANCER, obj_id=balancer.id[1],
                         namespace_id=self.id, obj=balancer))
                try:
                    if balancer.version is not None:
                        resp_pb = client.update_balancer(
                            namespace_id=balancer.id[0],
                            id=balancer.id[1],
                            version=balancer.version,
                            auth_pb=balancer.auth_pb,
                            spec_pb=balancer.spec_pb,
                            comment=comment
                        )
                    else:
                        resp_pb = client.create_balancer(
                            namespace_id=balancer.id[0],
                            id=balancer.id[1],
                            auth_pb=balancer.auth_pb,
                            spec_pb=balancer.spec_pb,
                            comment=comment
                        )
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE if not is_balancer_new else ev.CREATE_FAILURE,
                             obj_type=ev.BALANCER, obj_id=balancer.id[1],
                             namespace_id=self.id, obj=balancer, exc=e))
                    raise
                else:
                    balancer = Balancer.from_pb(resp_pb.balancer)
                    q.put(ev(type=ev.UPDATE_SUCCESS if not is_balancer_new else ev.CREATE_SUCCESS,
                             obj_type=ev.BALANCER, obj_id=balancer.id[1],
                             namespace_id=self.id, obj=balancer))
                    updated_balancers[balancer.id] = balancer

            for upstream in self.upstreams.itervalues():
                is_upstream_new = upstream.version is None
                q.put(ev(type=ev.UPDATE if not is_upstream_new else ev.CREATE,
                         obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                         namespace_id=self.id, obj=upstream))
                try:
                    if upstream.version is not None:
                        resp_pb = client.update_upstream(
                            namespace_id=upstream.id[0],
                            id=upstream.id[1],
                            version=upstream.version,
                            auth_pb=upstream.auth_pb,
                            spec_pb=upstream.spec_pb,
                            comment=comment
                        )
                    else:
                        resp_pb = client.create_upstream(
                            namespace_id=upstream.id[0],
                            id=upstream.id[1],
                            auth_pb=upstream.auth_pb,
                            spec_pb=upstream.spec_pb,
                            comment=comment
                        )
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE if not is_upstream_new else ev.CREATE_FAILURE,
                             obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                             namespace_id=self.id, obj=upstream, exc=e))
                    raise
                else:
                    upstream = Upstream.from_pb(resp_pb.upstream)
                    q.put(ev(type=ev.UPDATE_SUCCESS if not is_upstream_new else ev.CREATE_SUCCESS,
                             obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                             namespace_id=self.id, obj=upstream))
                    updated_upstreams[upstream.id] = upstream

            for backend in self.backends.itervalues():
                is_backend_new = backend.version is None
                q.put(ev(type=ev.UPDATE if not is_backend_new else ev.CREATE,
                         obj_type=ev.BACKEND, obj_id=backend.id[1],
                         namespace_id=self.id, obj=backend))
                try:
                    if backend.version is not None:
                        resp_pb = client.update_backend(
                            namespace_id=backend.id[0],
                            id=backend.id[1],
                            version=backend.version,
                            auth_pb=backend.auth_pb,
                            spec_pb=backend.spec_pb,
                            comment=comment
                        )
                    else:
                        resp_pb = client.create_backend(
                            namespace_id=backend.id[0],
                            id=backend.id[1],
                            auth_pb=backend.auth_pb,
                            spec_pb=backend.spec_pb,
                            comment=comment
                        )
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE if not is_backend_new else ev.CREATE_FAILURE,
                             obj_type=ev.BACKEND, obj_id=backend.id[1],
                             namespace_id=self.id, obj=backend, exc=e))
                    raise
                else:
                    backend = Backend.from_pb(resp_pb.backend)
                    q.put(ev(type=ev.UPDATE_SUCCESS if not is_backend_new else ev.CREATE_SUCCESS,
                             obj_type=ev.BACKEND, obj_id=backend.id[1],
                             namespace_id=self.id, obj=backend))
                    updated_backends[backend.id] = backend
        except Exception as e:
            q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, exc=e))
            # let's return partially updated namespace
            return NamespaceHolder(
                id=self.id,
                namespace=updated_namespace,
                balancers=dict(self.balancers, **updated_balancers),
                upstreams=dict(self.upstreams, **updated_upstreams),
                backends=dict(self.backends, **updated_backends)
            )
        else:
            q.put(ev(type=ev.UPDATE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id))
            return NamespaceHolder(
                id=self.id,
                namespace=updated_namespace,
                balancers=updated_balancers,
                upstreams=updated_upstreams,
                backends=updated_backends
            )

    @classmethod
    def _read_yaml_files(cls, dir):
        for name, filepath in cls._list_yaml_files(dir):
            with open(filepath) as f:
                yield name, filepath, f.read()

    @staticmethod
    def _list_yaml_files(dir):
        for filename in os.listdir(dir):
            filepath = os.path.join(dir, filename)
            if not os.path.isfile(filepath):
                continue
            name, ext = os.path.splitext(filename)
            if ext not in ('.yml', '.yaml'):
                continue
            yield name, filepath

    @classmethod
    def check_conflict_presence(cls, dir):
        conflict_files = []
        for nested_dir in ('balancers', 'upstreams', 'backends'):
            for filename in os.listdir(os.path.join(dir, nested_dir)):
                filepath = os.path.join(dir, nested_dir, filename)
                name, ext = os.path.splitext(filename)
                if ext.startswith('.CONFLICT'):
                    conflict_files.append(filepath)
        if conflict_files:
            raise ConflictError('Please resolve the following conflicts:\n  {}'.format('\n  '.join(conflict_files)))

    @classmethod
    def read_backends_from_fs(cls, namespace_id, dir, q=None):
        ev = FsEvent
        if q is None:
            q = Queue()

        versions_filepath = os.path.join(dir, '.awacs', 'versions')
        if not os.path.exists(versions_filepath):
            return None

        q.put(ev(type=ev.READ, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir))

        try:
            with open(versions_filepath) as f:
                versions = yaml.load(f, Loader=yaml.FullLoader)['versions']
            cls.check_conflict_presence(dir)

            backends = {}
            backends_dir = os.path.join(dir, 'backends')
            for name, _ in cls._list_yaml_files(backends_dir):
                y = '''{auth: {staff: {owners: {logins: [nanny-robot]}}}, nanny_snapshots: [{service_id: does-not-matter}]}'''
                backend = Backend.from_yml(
                    id=(namespace_id, name), version=versions['backends'].get(name), yml=y)
                backends[name] = backend
        except Exception as e:
            q.put(ev(type=ev.READ_FAILURE, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir, exc=e))
            raise
        else:
            q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir))
            return backends

    @classmethod
    def from_fs(cls, namespace_id, dir, q=None, ignore_acl=False):
        ev = FsEvent
        if q is None:
            q = Queue()

        versions_filepath = os.path.join(dir, '.awacs', 'versions')
        if not os.path.exists(versions_filepath):
            return None

        q.put(ev(type=ev.READ, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir))

        try:
            with open(versions_filepath) as f:
                versions = yaml.load(f, Loader=yaml.FullLoader)['versions']
            cls.check_conflict_presence(dir)

            namespace_yml_path = os.path.join(dir, 'namespace.yml')
            with open(namespace_yml_path) as f:
                # what if path does not exists?
                namespace_yml = f.read()

            namespace = Namespace.from_yml(id=namespace_id, yml=namespace_yml, ignore_acl=ignore_acl)

            balancers = {}
            balancers_dir = os.path.join(dir, 'balancers')
            for name, path, content in cls._read_yaml_files(balancers_dir):
                balancer = Balancer.from_yml(
                    id=(namespace_id, name), version=versions['balancers'].get(name), yml=content,
                    ignore_acl=ignore_acl)
                balancers[name] = balancer

            upstreams = {}
            upstreams_dir = os.path.join(dir, 'upstreams')
            for name, path, content in cls._read_yaml_files(upstreams_dir):
                upstream = Upstream.from_yml(
                    id=(namespace_id, name), version=versions['upstreams'].get(name), yml=content,
                    ignore_acl=ignore_acl)
                upstreams[name] = upstream

            backends = {}
            backends_dir = os.path.join(dir, 'backends')
            for name, path, content in cls._read_yaml_files(backends_dir):
                backend = Backend.from_yml(
                    id=(namespace_id, name), version=versions['backends'].get(name), yml=content, ignore_acl=ignore_acl)
                backends[name] = backend

            namespace_holder = cls(namespace_id, namespace, balancers, upstreams, backends)
        except Exception as e:
            q.put(ev(type=ev.READ_FAILURE, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir, exc=e))
            raise
        else:
            q.put(ev(type=ev.READ_SUCCESS, obj_type=ev.NAMESPACE, obj_id=namespace_id, obj_filepath=dir))
            return namespace_holder

    @property
    def balancer_versions(self):
        balancer_versions = collections.OrderedDict()
        for balancer in self.balancers.itervalues():
            balancer_versions[balancer.id[1]] = balancer.version
        return balancer_versions

    @property
    def upstream_versions(self):
        upstream_versions = collections.OrderedDict()
        for upstream in self.upstreams.itervalues():
            upstream_versions[upstream.id[1]] = upstream.version
        return upstream_versions

    @property
    def backend_versions(self):
        backend_versions = collections.OrderedDict()
        for backend in self.backends.itervalues():
            backend_versions[backend.id[1]] = backend.version
        return backend_versions

    def write_versions_to_fs(self, dir):
        versions = collections.OrderedDict([
            ('balancers', self.balancer_versions),
            ('upstreams', self.upstream_versions),
            ('backends', self.backend_versions),
        ])
        awacs_dir = os.path.join(dir, '.awacs')
        with open(os.path.join(awacs_dir, 'versions'), 'w') as f:
            f.write(yaml.safe_dump({'versions': versions}, default_flow_style=False))

    def rename_deleted_objects_in_fs(self, dir):
        balancers_dir = os.path.join(dir, 'balancers')
        for name, path, _ in self._read_yaml_files(balancers_dir):
            if name not in self.balancers:
                os.rename(path, path + '.DELETED')

        upstreams_dir = os.path.join(dir, 'upstreams')
        for name, path, _ in self._read_yaml_files(upstreams_dir):
            if name not in self.upstreams:
                os.rename(path, path + '.DELETED')

        backends_dir = os.path.join(dir, 'backends')
        for name, path, _ in self._read_yaml_files(backends_dir):
            if name not in self.backends:
                os.rename(path, path + '.DELETED')

    def to_fs(self, dir, q=None, for_merge=False):
        ev = FsEvent
        if q is None:
            q = Queue()
        q.put(ev(type=ev.UPDATE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir))

        try:

            try:
                curr_fs_namespace = NamespaceHolder.from_fs(self.id, dir)
            except Exception as e:
                q.put(ev(type=ev.READ_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=e))
                curr_fs_namespace = None
            yml, failed_ns_groups = self.namespace.to_yml()
            with open(os.path.join(dir, 'namespace.yml'), 'w') as f:
                f.write(yml)

            balancers_dir = os.path.join(dir, 'balancers')
            failed_balancer_groups = {}
            for balancer in self.balancers.itervalues():
                filename = '{}.yml'.format(balancer.id[1])
                filepath = os.path.join(balancers_dir, filename)
                if curr_fs_namespace and balancer.id in curr_fs_namespace.balancers:
                    if curr_fs_namespace.balancers[balancer.id] == balancer:
                        # do not update YAML if nothing's changed
                        q.put(ev(type=ev.UPDATE_SKIPPED, obj_type=ev.BALANCER, obj_id=balancer.id[1],
                                 obj_filepath=filepath, namespace_id=self.id, comment='nothing has changed'))
                        continue
                    elif for_merge:  # and curr_fs_namespace.balancers[balancer.id].version != balancer.version:
                        filepath += '.CONFLICT'
                try:
                    yml, failed_balancer_groups[balancer.id[1]] = balancer.to_yml()
                    with open(filepath, 'w') as f:
                        f.write(yml)
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.BALANCER, obj_id=balancer.id[1],
                             obj_filepath=filepath, namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.UPDATE_SUCCESS, obj_type=ev.BALANCER, obj_id=balancer.id[1],
                             obj_filepath=filepath, namespace_id=self.id))

            upstreams_dir = os.path.join(dir, 'upstreams')
            failed_upstream_groups = {}
            for upstream in self.upstreams.itervalues():
                filename = '{}.yml'.format(upstream.id[1])
                filepath = os.path.join(upstreams_dir, filename)
                if curr_fs_namespace and upstream.id in curr_fs_namespace.upstreams:
                    if curr_fs_namespace.upstreams[upstream.id] == upstream:
                        # do not update YAML if nothing's changed
                        q.put(ev(type=ev.UPDATE_SKIPPED, obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                                 obj_filepath=filepath, namespace_id=self.id, comment='nothing has changed'))
                        continue
                    elif for_merge:
                        filepath += '.CONFLICT'
                try:
                    yml, failed_upstream_groups[upstream.id[1]] = upstream.to_yml()
                    with open(filepath, 'w') as f:
                        f.write(yml)
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                             obj_filepath=filepath, namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.UPDATE_SUCCESS, obj_type=ev.UPSTREAM, obj_id=upstream.id[1],
                             obj_filepath=filepath, namespace_id=self.id))

            backends_dir = os.path.join(dir, 'backends')
            failed_backend_groups = {}
            for backend in self.backends.itervalues():
                filename = '{}.yml'.format(backend.id[1])
                filepath = os.path.join(backends_dir, filename)
                if curr_fs_namespace and backend.id in curr_fs_namespace.backends:
                    if curr_fs_namespace.backends[backend.id] == backend:
                        # do not update YAML if nothing's changed
                        q.put(ev(type=ev.UPDATE_SKIPPED, obj_type=ev.BACKEND, obj_id=backend.id[1],
                                 obj_filepath=filepath, namespace_id=self.id, comment='nothing has changed'))
                        continue
                    elif for_merge:
                        filepath += '.CONFLICT'
                try:
                    yml, failed_backend_groups[backend.id[1]] = backend.to_yml()
                    with open(filepath, 'w') as f:
                        f.write(yml)
                except Exception as e:
                    q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.BACKEND, obj_id=backend.id[1],
                             obj_filepath=filepath, namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.UPDATE_SUCCESS, obj_type=ev.BACKEND, obj_id=backend.id[1],
                             obj_filepath=filepath, namespace_id=self.id))

            self.write_versions_to_fs(dir)
            self.rename_deleted_objects_in_fs(dir)
            if failed_ns_groups:
                q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=ValueError('Failed to update groups {}'.format(failed_ns_groups))))
            if failed_balancer_groups:
                q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=ValueError('Failed to update groups {}'.format(failed_balancer_groups))))
            if failed_upstream_groups:
                q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=ValueError('Failed to update groups {}'.format(failed_upstream_groups))))
            if failed_backend_groups:
                q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=ValueError('Failed to update groups {}'.format(failed_backend_groups))))
        except Exception as e:
            q.put(ev(type=ev.UPDATE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir, exc=e))
        else:
            q.put(ev(type=ev.UPDATE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=dir))

    def compile(self, ctl, out_dir, templates_dir=None, q=None):
        ev = CompileEvent
        if q is None:
            q = Queue()
        q.put(ev(type=ev.COMPILE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
        try:
            try:
                q.put(ev(type=ev.PREPARE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
                if not os.path.exists(out_dir):
                    os.makedirs(out_dir)
            except Exception as e:
                q.put(ev(type=ev.PREPARE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir, exc=e))
                raise
            else:
                q.put(ev(type=ev.PREPARE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))

            ctime = int(time.time() * 1000000)  # time in microseconds
            upstream_spec_pbs = dict()
            for upstream in self.upstreams.itervalues():
                try:
                    q.put(ev(type=ev.PARSE, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id))
                    upstream_pb = yamlparser.parse(modules_pb2.Holder, upstream.spec_pb.yandex_balancer.yaml)
                    upstream.spec_pb.yandex_balancer.config.CopyFrom(upstream_pb)
                    upstream_spec_pbs[upstream.get_version(ctime)] = upstream.spec_pb
                except Exception as e:
                    q.put(ev(type=ev.PARSE_FAILURE, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.PARSE_SUCCESS, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id))

            try:
                q.put(ev(type=ev.COLLECT_BACKENDS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))

                all_full_backend_ids = set()
                for upstream_spec_pb in upstream_spec_pbs.itervalues():
                    config = Holder(upstream_spec_pb.yandex_balancer.config)
                    full_backend_ids = generator.get_would_be_injected_full_backend_ids(self.id, config)
                    all_full_backend_ids.update(full_backend_ids)

                temp_upstream_spec_pbs = {upstream_version.upstream_id: upstream_pb for upstream_version, upstream_pb
                                          in upstream_spec_pbs.iteritems()}

                for balancer in self.balancers.itervalues():
                    balancer_pb = generator.get_yandex_config_pb(self.id, balancer.spec_pb, temp_upstream_spec_pbs)
                    config = Holder(balancer_pb)
                    full_backend_ids = generator.get_would_be_injected_full_backend_ids(self.id, config)
                    all_full_backend_ids.update(full_backend_ids)

                namespace_ids = {namespace_id for namespace_id, _ in all_full_backend_ids}
                namespaces = dict()
                for namespace_id in namespace_ids:
                    if namespace_id == self.id:
                        namespaces[namespace_id] = self
                    else:
                        if not templates_dir:
                            raise NamespaceNotFound('Namespace "{}" not found'.format(namespace_id))
                        ns = NamespaceHolder.from_fs(
                            namespace_id, os.path.join(templates_dir, namespace_id), q)
                        if ns is None:
                            raise NamespaceNotFound('Namespace "{}" not found'.format(namespace_id))
                        namespaces[namespace_id] = ns
                namespace_backends = collections.defaultdict(list)
                for namespace_id, backend_id in all_full_backend_ids:
                    namespace_backends[namespace_id].append(namespaces[namespace_id].backends[backend_id])
            except Exception as e:
                q.put(ev(type=ev.COLLECT_BACKENDS_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id,
                         obj_filepath=out_dir, exc=e))
                raise
            else:
                q.put(ev(type=ev.COLLECT_BACKENDS_SUCCESS, obj_type=ev.NAMESPACE,
                         obj_id=self.id, obj_filepath=out_dir))

            backend_spec_pbs = {}
            endpoint_set_spec_pbs = {}
            for namespace_id, backends in namespace_backends.iteritems():
                for backend in backends:
                    try:
                        q.put(ev(type=ev.RESOLVE, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0]))
                        version = BackendVersion(ctime, backend.id,
                                                 backend.version, backend.spec_pb.deleted)
                        backend_spec_pbs[version] = backend.spec_pb

                        version = EndpointSetVersion(ctime, backend.id,
                                                     backend.version, backend.spec_pb.deleted)
                        for snapshot_pb in backend.spec_pb.selector.nanny_snapshots:
                            if not snapshot_pb.snapshot_id:
                                snapshot_pb.snapshot_id = get_current_runtime_attrs_id(snapshot_pb.service_id)
                        endpoint_set_spec_pbs[version] = generator.map_backend_to_endpoint_set(backend.spec_pb)
                    except Exception as e:
                        q.put(ev(type=ev.RESOLVE_FAILURE, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0], exc=e))
                        raise
                    else:
                        q.put(ev(type=ev.RESOLVE_SUCCESS, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0]))

            backends_json_path = os.path.join(out_dir, 'backends.json')
            try:
                q.put(ev(type=ev.DUMP, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=backends_json_path))
                with open(backends_json_path, 'w') as f:
                    json.dump(generator.generate_backends_json(endpoint_set_spec_pbs, self.id), f, indent=4)
            except Exception as e:
                q.put(ev(type=ev.DUMP_FAILURE, obj_type=ev.NAMESPACE,
                         obj_id=self.id, obj_filepath=backends_json_path, exc=e))
                raise
            else:
                q.put(ev(type=ev.DUMP_SUCCESS, obj_type=ev.NAMESPACE,
                         obj_id=self.id, obj_filepath=backends_json_path))

            for balancer in self.balancers.itervalues():
                try:
                    q.put(ev(type=ev.VALIDATE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))
                    validation_result = generator.validate_config(
                        self.id, balancer.get_version(ctime), balancer.spec_pb,
                        upstream_spec_pbs, backend_spec_pbs, endpoint_set_spec_pbs,
                    )
                    balancer_holder = validation_result.balancer
                except Exception as e:
                    q.put(ev(type=ev.VALIDATE_FAILURE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.VALIDATE_SUCCESS, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))

                try:
                    q.put(ev(type=ev.GENERATE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))
                    config = (balancer_holder.module or balancer_holder.chain).to_config()
                    lua = config.to_top_level_lua()
                except Exception as e:
                    q.put(ev(type=ev.GENERATE_FAILURE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.GENERATE_SUCCESS, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))

                lua_path = os.path.join(out_dir, balancer.id[1] + '.lua')
                try:
                    q.put(ev(type=ev.DUMP, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1],
                             obj_filepath=lua_path, namespace_id=self.id))
                    with open(lua_path, 'w') as f:
                        f.write(lua)
                except Exception as e:
                    q.put(ev(type=ev.DUMP_FAILURE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], obj_filepath=lua_path, namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.DUMP_SUCCESS, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], obj_filepath=lua_path, namespace_id=self.id))

        except Exception as e:
            q.put(ev(type=ev.COMPILE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir, exc=e))
            raise
        else:
            q.put(ev(type=ev.COMPILE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))

    @staticmethod
    def _get_random_endpoint_set(backend_spec_pb):
        def random_ipv4():
            return '.'.join(str(random.randint(0, 255)) for _ in range(4))

        def random_ipv6():
            return ':'.join(
                ''.join(random.choice(string.hexdigits).lower() for _ in range(4))
                for _ in range(8)
            )

        spec_pb = model_pb2.EndpointSetSpec()
        spec_pb.deleted = backend_spec_pb.deleted
        spec_pb.instances.add(
            host=hex(random.randint(1, 1000000)) + '.yandex.ru',
            port=random.randint(80, 1024),
            ipv4_addr=random_ipv4(),
            ipv6_addr=random_ipv6(),
            weight=1,
        )
        return spec_pb

    def digest(self, ctl, out_dir, templates_dir=None, q=None):
        ev = CompileEvent
        if q is None:
            q = Queue()
        q.put(ev(type=ev.COMPILE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
        try:
            try:
                q.put(ev(type=ev.PREPARE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
                if not os.path.exists(out_dir):
                    os.makedirs(out_dir)
            except Exception as e:
                q.put(ev(type=ev.PREPARE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir, exc=e))
                raise
            else:
                q.put(ev(type=ev.PREPARE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
            ctime = int(time.time() * 1000000)  # time in microseconds
            upstream_spec_pbs = dict()
            for upstream in self.upstreams.itervalues():
                try:
                    q.put(ev(type=ev.PARSE, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id))
                    upstream_pb = yamlparser.parse(modules_pb2.Holder, upstream.spec_pb.yandex_balancer.yaml)
                    upstream.spec_pb.yandex_balancer.config.CopyFrom(upstream_pb)
                    upstream_spec_pbs[upstream.get_version(ctime)] = upstream.spec_pb
                except Exception as e:
                    q.put(ev(type=ev.PARSE_FAILURE, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.PARSE_SUCCESS, obj_type=ev.UPSTREAM,
                             obj_id=upstream.id[1], namespace_id=self.id))
            try:
                q.put(ev(type=ev.COLLECT_BACKENDS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
                all_full_backend_ids = set()
                for upstream_spec_pb in upstream_spec_pbs.itervalues():
                    config = Holder(upstream_spec_pb.yandex_balancer.config)
                    full_backend_ids = generator.get_would_be_injected_full_backend_ids(self.id, config)
                    all_full_backend_ids.update(full_backend_ids)
                temp_upstream_spec_pbs = {upstream_version.upstream_id: upstream_pb for upstream_version, upstream_pb
                                          in upstream_spec_pbs.iteritems()}
                for balancer in self.balancers.itervalues():
                    balancer_pb = generator.get_yandex_config_pb(self.id, balancer.spec_pb, temp_upstream_spec_pbs)
                    config = Holder(balancer_pb)
                    full_backend_ids = generator.get_would_be_injected_full_backend_ids(self.id, config)
                    all_full_backend_ids.update(full_backend_ids)
                namespace_ids = {namespace_id for namespace_id, _ in all_full_backend_ids}
                namespaces = dict()
                for namespace_id in namespace_ids:
                    if namespace_id == self.id:
                        namespaces[namespace_id] = self.backends
                    else:
                        if not templates_dir:
                            raise NamespaceNotFound('Namespace "{}" not found'.format(namespace_id))
                        namespaces[namespace_id] = NamespaceHolder.read_backends_from_fs(
                            namespace_id, os.path.join(templates_dir, namespace_id), q)

                namespace_backends = collections.defaultdict(list)
                for namespace_id, backend_id in all_full_backend_ids:
                    namespace_backends[namespace_id].append(namespaces[namespace_id][backend_id])
            except Exception as e:
                q.put(ev(type=ev.COLLECT_BACKENDS_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id,
                         obj_filepath=out_dir, exc=e))
                raise
            else:
                q.put(ev(type=ev.COLLECT_BACKENDS_SUCCESS, obj_type=ev.NAMESPACE,
                         obj_id=self.id, obj_filepath=out_dir))
            backend_spec_pbs = {}
            endpoint_set_spec_pbs = {}
            for namespace_id, backends in namespace_backends.iteritems():
                for backend in backends:
                    try:
                        q.put(ev(type=ev.RESOLVE, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0]))
                        version = BackendVersion(ctime, backend.id,
                                                 backend.version, backend.spec_pb.deleted)
                        backend_spec_pbs[version] = backend.spec_pb
                        version = EndpointSetVersion(ctime, backend.id,
                                                     backend.version, backend.spec_pb.deleted)
                        endpoint_set_spec_pbs[version] = self._get_random_endpoint_set(backend.spec_pb)
                    except Exception as e:
                        q.put(ev(type=ev.RESOLVE_FAILURE, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0], exc=e))
                        raise
                    else:
                        q.put(ev(type=ev.RESOLVE_SUCCESS, obj_type=ev.BACKEND,
                                 obj_id=backend.id[1], namespace_id=backend.id[0]))
            for balancer in self.balancers.itervalues():
                try:
                    q.put(ev(type=ev.VALIDATE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))
                    validation_result = generator.validate_config(
                        self.id, balancer.get_version(ctime), balancer.spec_pb,
                        upstream_spec_pbs, backend_spec_pbs, endpoint_set_spec_pbs,
                        _digest=True
                    )
                    balancer_holder = validation_result.balancer
                    result_path = os.path.join(out_dir, ':'.join(balancer.id) + '.pb')
                    with open(result_path, 'wb') as f:
                        f.write(balancer_holder.pb.SerializeToString())
                except Exception as e:
                    q.put(ev(type=ev.VALIDATE_FAILURE, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id, exc=e))
                    raise
                else:
                    q.put(ev(type=ev.VALIDATE_SUCCESS, obj_type=ev.BALANCER,
                             obj_id=balancer.id[1], namespace_id=self.id))
        except Exception as e:
            q.put(ev(type=ev.COMPILE_FAILURE, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir, exc=e))
            raise
        else:
            q.put(ev(type=ev.COMPILE_SUCCESS, obj_type=ev.NAMESPACE, obj_id=self.id, obj_filepath=out_dir))
