package main

import (
	"errors"
	"fmt"
	"log"
	"os"
	"strings"

	"a.yandex-team.ru/infra/awacs/awacslet/internal/container"
	"a.yandex-team.ru/library/go/core/buildinfo"
)

var commands = []string{"version", "start", "prepare", "notify", "status", "stop", "reopenlog", "get_workers_count", "check_fq_enabled"}
var commandsListStr = fmt.Sprintf("\"%s\"", strings.Join(commands, "\", \""))
var supportedDaemons = []string{"balancer", "push-client"}
var supportedDaemonsStr = fmt.Sprintf("\"%s\"", strings.Join(supportedDaemons, "\", \""))

func notify(c *container.Container, args []string) error {
	if len(args) < 2 {
		return errors.New("\"notify\" requires a \"notify_action\" argument and a list of changes")
	}
	if args[0] != "notify_action" {
		return fmt.Errorf("\"notify\" must be succeeded by \"notify_action\" argument, not %q", args[0])
	}
	changes := args[1:]
	if len(changes) > 1 {
		return errors.New("too many changes")
	}
	for i, change := range changes {
		if change != "!config.lua" && change != "+config.lua" {
			return fmt.Errorf("changes[%d]: unexpected change %q", i, change)
		}
	}
	return c.ReloadConfig()
}

func start(c *container.Container, args []string) error {
	if len(args) != 1 {
		return errors.New("\"start\" requires an argument")
	}
	switch args[0] {
	case "balancer":
		if err := c.StartBalancer(); err != nil {
			return fmt.Errorf("balancer start failed: %v", err)
		}
	case "push-client":
		if err := c.StartPushClient(); err != nil {
			return fmt.Errorf("push-client start failed: %v", err)
		}
	default:
		return fmt.Errorf("\"start\" argument must be one of the following: %s", supportedDaemons)
	}
	return nil
}

func reopenlog(c *container.Container, args []string) error {
	if len(args) != 1 {
		return errors.New("\"reopenlog\" requires an argument")
	}
	switch args[0] {
	case "balancer":
		if err := c.ReopenBalancerLog(); err != nil {
			return fmt.Errorf("balancer reopenlog failed: %v", err)
		}
	case "push-client":
		if err := c.ReopenPushClientLog(); err != nil {
			return fmt.Errorf("push-client reopenlog failed: %v", err)
		}
	default:
		return fmt.Errorf("\"reopenlog\" argument must be one of the following: %s", supportedDaemons)
	}
	return nil
}

func main() {
	if len(os.Args) < 2 {
		log.Fatalf("expected command: %s", commandsListStr)
	}

	if os.Args[1] == "version" {
		fmt.Println(buildinfo.Info.ProgramVersion)
		os.Exit(0)
	}

	c, err := container.New()
	if err != nil {
		log.Fatalf("failed to initialize: %v", err)
	}

	switch os.Args[1] {
	case "start":
		if err := start(c, os.Args[2:]); err != nil {
			log.Fatalf("start failed: %v", err)
		}
	case "prepare":
		if err := c.Prepare(); err != nil {
			log.Fatalf("prepare failed: %v", err)
		}
	case "notify":
		if err := notify(c, os.Args[2:]); err != nil {
			log.Fatalf("notify failed: %v", err)
		}
	case "status":
		if err := c.Status(); err != nil {
			log.Fatalf("status failed: %v", err)
		}
	case "stop":
		if err := c.Stop(); err != nil {
			log.Fatalf("stop failed: %v", err)
		}
	case "reopenlog":
		if err := reopenlog(c, os.Args[2:]); err != nil {
			log.Fatalf("reopenlog failed: %v", err)
		}
	case "get_workers_count":
		fmt.Print(c.GetWorkersCount())
	case "check_fq_enabled":
		fmt.Printf("%t\n", c.CheckFqNetSchedulerEnabled())
	default:
		log.Fatalf("command must be one of the following: %s", commandsListStr)
	}
}
