package certs

import (
	"archive/tar"
	"bytes"
	"compress/gzip"
	"errors"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"path/filepath"
	"regexp"
	"strings"
)

const targzPubKeyNameRe string = `^allCAs-(.+)\.pem$`

type Cert struct {
	name       string
	pubKeyPem  []byte
	privKeyPem []byte
	tickets    [][]byte
}

func writeFileIfNotExists(path string, data []byte, perm os.FileMode) error {
	f, err := os.OpenFile(path, os.O_WRONLY|os.O_CREATE|os.O_EXCL, perm)
	if err != nil {
		return err
	}
	_, err = f.Write(data)
	if err1 := f.Close(); err == nil {
		err = err1
	}
	return err
}

func isFileAlreadyWritten(path string, data []byte) bool {
	writtenData, err := ioutil.ReadFile(path)
	if err != nil {
		return false
	}
	return bytes.Equal(data, writtenData)
}

func writeFileIfNotExistsAndNotAlreadyWritten(path string, data []byte, perm os.FileMode) error {
	if isFileAlreadyWritten(path, data) {
		return nil
	}
	return writeFileIfNotExists(path, data, perm)
}

func (c Cert) Dump(pubDirPath, privDirPath string) error {
	pubKeyPemFilePath := filepath.Join(pubDirPath, "allCAs-"+c.name+".pem")
	if err := writeFileIfNotExistsAndNotAlreadyWritten(pubKeyPemFilePath, c.pubKeyPem, 0755); err != nil {
		return fmt.Errorf("failed to write %s: %w", pubKeyPemFilePath, err)
	}

	privKeyPemFilePath := filepath.Join(privDirPath, c.name+".pem")
	if err := writeFileIfNotExistsAndNotAlreadyWritten(privKeyPemFilePath, c.privKeyPem, 0755); err != nil {
		return fmt.Errorf("failed to write %s: %w", privKeyPemFilePath, err)
	}

	ticketFileNames := [3]string{
		"1st." + c.name + ".key",
		"2nd." + c.name + ".key",
		"3rd." + c.name + ".key",
	}
	for i, ticketFileName := range ticketFileNames {
		ticketFilePath := filepath.Join(privDirPath, ticketFileName)
		if err := writeFileIfNotExistsAndNotAlreadyWritten(ticketFilePath, c.tickets[i], 0755); err != nil {
			return fmt.Errorf("failed to write %s: %w", ticketFilePath, err)
		}
	}
	return nil
}

func New(name string, pubKeyPem, privKeyPem []byte, tickets [][]byte) (*Cert, error) {
	if len(tickets) != 3 {
		return nil, errors.New("tickets must contain exactly 3 elements")
	}
	return &Cert{
		name:       name,
		pubKeyPem:  pubKeyPem,
		privKeyPem: privKeyPem,
		tickets:    tickets,
	}, nil
}

func LoadFromTarGz(r io.Reader) (*Cert, error) {
	gr, err := gzip.NewReader(r)
	if err != nil {
		return nil, fmt.Errorf("failed to ungzip: %w", err)
	}
	tr := tar.NewReader(gr)
	files := make(map[string][]byte)
	for {
		f, err := tr.Next()
		if err == io.EOF {
			break
		}
		if err != nil {
			return nil, fmt.Errorf("failed to untar: %w", err)
		}

		mode := f.FileInfo().Mode()
		switch {
		case mode.IsRegular():
			data, err := ioutil.ReadAll(tr)
			if err != nil {
				return nil, fmt.Errorf("failed to untar %s: %w", f.Name, err)
			}
			files[filepath.Clean(f.Name)] = data
		case mode.IsDir():
			continue
		default:
			return nil, fmt.Errorf("failed to untar %s: unsupported file type: %v", f.Name, mode)
		}
	}
	re := regexp.MustCompile(targzPubKeyNameRe)
	var name string
	for path := range files {
		match := re.FindStringSubmatch(path)
		if match != nil {
			name = match[1]
			break
		}
	}
	if name == "" {
		return nil, fmt.Errorf("could not find %s in targz", targzPubKeyNameRe)
	}

	cert := Cert{name: name}

	var ok bool
	pubKeyPemPath := "allCAs-" + name + ".pem"
	cert.pubKeyPem, ok = files[pubKeyPemPath]
	if !ok {
		return nil, fmt.Errorf("could not find %s in targz", pubKeyPemPath)
	}
	delete(files, pubKeyPemPath)

	privKeyPemPath := filepath.Join("priv", name+".pem")
	cert.privKeyPem, ok = files[privKeyPemPath]
	if !ok {
		return nil, fmt.Errorf("could not find %s in targz", pubKeyPemPath)
	}
	delete(files, privKeyPemPath)

	firstTicketPath := filepath.Join("priv", "1st."+name+".key")
	secondTicketPath := filepath.Join("priv", "2nd."+name+".key")
	thirdTicketPath := filepath.Join("priv", "3rd."+name+".key")
	for _, ticketPath := range []string{firstTicketPath, secondTicketPath, thirdTicketPath} {
		data, ok := files[ticketPath]
		if !ok {
			return nil, fmt.Errorf("could not find %s in targz", firstTicketPath)
		}
		delete(files, ticketPath)
		cert.tickets = append(cert.tickets, data)
	}

	extraPaths := make([]string, len(files))
	i := 0
	for path := range files {
		extraPaths[i] = path
		i++
	}
	if len(extraPaths) > 0 {
		return nil, fmt.Errorf("unexpected extra files in targz: %s", strings.Join(extraPaths, ", "))
	}
	return &cert, nil
}
