package certs_test

import (
	"crypto/rsa"
	"crypto/x509"
	"encoding/pem"
	"fmt"
	"io/ioutil"
	"os"
	"path/filepath"
	"strings"
	"testing"

	"a.yandex-team.ru/infra/awacs/awacslet/internal/certs"
)

func readAllCAs() (*x509.CertPool, error) {
	roots := x509.NewCertPool()
	rootsPemFilePath := "testdata/allCAs.pem"
	rootsPemData, err := ioutil.ReadFile(rootsPemFilePath)
	if err != nil {
		return nil, fmt.Errorf("failed to read %s: %v", rootsPemFilePath, err)
	}
	ok := roots.AppendCertsFromPEM(rootsPemData)
	if !ok {
		return nil, fmt.Errorf("failed to parse %s: %v", rootsPemFilePath, err)
	}
	return roots, nil
}

func parseCertificate(pemFilePath string) (*x509.Certificate, error) {
	pemData, err := ioutil.ReadFile(pemFilePath)
	if err != nil {
		return nil, fmt.Errorf("failed to read %s: %v", pemFilePath, err)
	}
	block, _ := pem.Decode(pemData)
	if block == nil {
		return nil, fmt.Errorf("failed to decode %s", pemFilePath)
	}
	cert, err := x509.ParseCertificate(block.Bytes)
	if err != nil {
		return nil, fmt.Errorf("failed to parse cert from %s: %v", pemFilePath, err)
	}
	return cert, nil
}

func parsePrivateKey(pemFilePath string) (*rsa.PrivateKey, error) {
	pemData, err := ioutil.ReadFile(pemFilePath)
	if err != nil {
		return nil, fmt.Errorf("failed to read %s: %v", pemFilePath, err)
	}
	block, _ := pem.Decode(pemData)
	if block == nil {
		return nil, fmt.Errorf("failed to decode %s", pemFilePath)
	}
	cert, err := x509.ParsePKCS8PrivateKey(block.Bytes)
	if err != nil {
		return nil, fmt.Errorf("failed to parse cert from %s: %v", pemFilePath, err)
	}
	return cert.(*rsa.PrivateKey), nil
}

func prepareTempDirs() (string, string) {
	tempDirPath, _ := ioutil.TempDir("/tmp/", "awacslet-certs-test")

	pubDirPath := tempDirPath
	privDirPath := filepath.Join(tempDirPath, "priv")

	_ = os.Mkdir(privDirPath, 0755)
	return pubDirPath, privDirPath
}

func TestLoadAndDump(t *testing.T) {
	f, _ := os.Open("testdata/secrets.tgz")
	defer f.Close()

	c, err := certs.LoadFromTarGz(f)
	if err != nil {
		t.Fatalf("LoadFromTarGz failed: %v", err)
	}

	pubDirPath, privDirPath := prepareTempDirs()
	defer os.RemoveAll(pubDirPath)
	defer os.RemoveAll(privDirPath)

	err = c.Dump(pubDirPath, privDirPath)
	if err != nil {
		t.Fatalf("Dump failed: %v", err)
	}

	roots, err := readAllCAs()
	if err != nil {
		t.Fatalf("readAllCAs failed: %v", err)
	}

	pemFilePath := filepath.Join(pubDirPath, "allCAs-romanovich-awacslet-test.pem")
	cert, err := parseCertificate(pemFilePath)
	if err != nil {
		t.Fatalf("parseCertificate failed: %v", err)
	}
	opts := x509.VerifyOptions{
		DNSName: "romanovich-awacslet-test.yandex.net",
		Roots:   roots,
	}
	if _, err := cert.Verify(opts); err != nil {
		t.Fatalf("failed to verify %s: %v", pemFilePath, err)
	}

	pemFilePath = filepath.Join(pubDirPath, "priv", "romanovich-awacslet-test.pem")
	key, err := parsePrivateKey(pemFilePath)
	if err != nil {
		t.Fatalf("parsePrivateKey failed: %v", err)
	}
	err = key.Validate()
	if err != nil {
		t.Fatalf("failed to validate private key: %v", err)
	}

	ticketFileNames := [3]string{
		"1st.romanovich-awacslet-test.key",
		"2nd.romanovich-awacslet-test.key",
		"3rd.romanovich-awacslet-test.key",
	}
	for _, ticketFileName := range ticketFileNames {
		ticketFilePath := filepath.Join(pubDirPath, "priv", ticketFileName)
		if _, err := os.Stat(ticketFilePath); os.IsNotExist(err) {
			t.Fatalf("%s does not exist: %v", ticketFilePath, err)
		}
	}
}

func TestConflict(t *testing.T) {
	c1, err := certs.New(
		"test",
		[]byte{'a'},
		[]byte{'a'},
		[][]byte{[]byte{'1'}, []byte{'2'}, []byte{'3'}},
	)
	if err != nil {
		t.Fatalf("failed to create cert: %v", err)
	}

	c2, err := certs.New(
		"test",
		[]byte{'b'},
		[]byte{'b'},
		[][]byte{[]byte{'4'}, []byte{'5'}, []byte{'6'}},
	)
	if err != nil {
		t.Fatalf("failed to create cert: %v", err)
	}

	pubDirPath, privDirPath := prepareTempDirs()
	defer os.RemoveAll(pubDirPath)
	defer os.RemoveAll(privDirPath)

	err = c1.Dump(pubDirPath, privDirPath)
	if err != nil {
		t.Fatalf("Dump failed: %v", err)
	}

	err = c2.Dump(pubDirPath, privDirPath)
	if err == nil {
		t.Fatalf("Dump not failed: %v", err)
	}

	if !strings.HasSuffix(err.Error(), "file exists") {
		t.Fatalf("Dump failed not as expected: %v", err)
	}
}
