package container

import (
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"strings"
)

func stashAwacsEnvVars(path string) error {
	var b strings.Builder
	for _, e := range os.Environ() {
		if strings.HasPrefix(e, "AWACS_") {
			b.WriteString(e)
			b.WriteString(string("\n"))
		}
	}
	f, err := os.OpenFile(path, os.O_CREATE|os.O_WRONLY|os.O_TRUNC, 0666)
	if err != nil {
		return fmt.Errorf("failed to open %s: %w", path, err)
	}
	defer f.Close()
	_, err = io.WriteString(f, b.String())
	if err != nil {
		return fmt.Errorf("failed to write %s: %w", path, err)
	}
	return nil
}

func unstashAwacsEnvVars(path string) ([]string, error) {
	f, err := os.Open(path)
	if err != nil {
		return nil, fmt.Errorf("failed to open %s: %w", path, err)
	}
	defer f.Close()

	content, err := ioutil.ReadAll(f)
	if err != nil {
		return nil, fmt.Errorf("failed to read %s: %w", path, err)
	}
	if len(content) == 0 {
		return nil, nil
	}
	return strings.Split(string(content), "\n"), nil
}

func getEnvVar(key string) (string, error) {
	rv, ok := os.LookupEnv(key)
	if !ok {
		return "", fmt.Errorf("env var %s is not set", key)
	}
	return rv, nil
}

func getEnvVarOrDefault(key, defaultValue string) string {
	value, ok := os.LookupEnv(key)
	if ok {
		return value
	} else {
		return defaultValue
	}
}

func setEnvVar(key, value string) error {
	return os.Setenv(key, value)
}
