package container

import (
	"bufio"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"math"
	"net/http"
	"net/url"
	"os"
	"os/exec"
	"path"
	"path/filepath"
	"sort"
	"strconv"
	"strings"
	"syscall"
	"time"

	"a.yandex-team.ru/infra/awacs/awacslet/internal/certs"
	porto_api "a.yandex-team.ru/infra/porto/api_go"
)

const EnvPrefixString = "ENV_"

type Container struct {
	bsconfigDir                     string // $BSCONFIG_IDIR
	bsconfigPort                    int    // $BSCONFIG_IPORT
	shmDir                          string // "/dev/shm/"
	secretsDir                      string // "/dev/shm/balancer/"
	logsDir                         string // "/place/db/www/logs/"
	dc                              string // "sas", "man", "vla", "iva" or "myt"
	workers                         int
	gracefulShutdownTimeout         string // "55s"
	gracefulShutdownCooldownTimeout string // "30s"
	workersProviderPath             string // "./awacslet_get_workers_provider.lua"
	sdCacheDir                      string // "./sd_cache or /awacs/sd_cache
	useSudoForShawshank             bool
	fqInterfaces                    []string //  "tun0 ip6tnl0"
}

type KeyValue struct {
	key   string
	value string
}

func New() (*Container, error) {
	err := prepareConfLocal()
	if err != nil {
		return nil, fmt.Errorf("failed to parse Conf.local: %w", err)
	}

	portoConn, err := connectToPorto()
	if err != nil {
		return nil, fmt.Errorf("failed to connect to porto: %w", err)
	}

	shmDir := getEnvVarOrDefault("AWACSLET_DEV_SHM_DIR_PATH", "/dev/shm/")
	logsDir := getEnvVarOrDefault("AWACSLET_LOGS_DIR_PATH", "/place/db/www/logs/")
	workersProviderPath := getEnvVarOrDefault(
		"AWACSLET_WORKERS_PROVIDER_PATH", "./awacslet_get_workers_provider.lua")
	/*
		Graceful shutdown timeouts are described in https://st.yandex-team.ru/AWACS-324.
		Each keepalived pings its real servers every 10 seconds.
		So we spend 35 secs in cooldown mode doing two things:
		* serving rs_weight=0 to all keepalived checks;
		* closing TCP sessions for all incoming HTTP requests.
		After that we instruct the system to stop accepting all new connections and continue finishing
		existing ones for 50 seconds.
		Please note that after 90 seconds since instancectl initiated .Stop() call, it will send us TERM signal.
		In 30 seconds after that it will try to KILL us.
		These timeouts are confured in Nanny service (Instance spec → Container pre stop handler).
	*/
	gracefulShutdownCooldownTimeout := getEnvVarOrDefault("AWACSLET_GRACEFUL_SHUTDOWN_COOLDOWN_TIMEOUT", "35s")
	gracefulShutdownTimeout := getEnvVarOrDefault("AWACSLET_GRACEFUL_SHUTDOWN_TIMEOUT", "50s")

	useSudoForShawshankStr := getEnvVarOrDefault("AWACSLET_USE_SUDO_FOR_SHAWSHANK", "1")
	useSudoForShawshank, err := strconv.Atoi(useSudoForShawshankStr)
	if err != nil {
		return nil, fmt.Errorf("failed to read \"AWACSLET_USE_SUDO_FOR_SHAWSHANK\" env var: %w", err)
	}
	if useSudoForShawshank != 0 && useSudoForShawshank != 1 {
		return nil, fmt.Errorf("failed to read \"AWACSLET_USE_SUDO_FOR_SHAWSHANK\" env var: must be 0 or 1")
	}

	fqInterfacesStr := getEnvVarOrDefault("AWACSLET_FQ_INTERFACES", "")
	// strings.FieldsFunc used here for splitting because of strings.Split returns empty elements
	fqInterfaces := strings.FieldsFunc(strings.TrimSpace(fqInterfacesStr), func(c rune) bool { return c == ' ' })

	bsconfigPortStr, err := getEnvVar("BSCONFIG_IPORT")
	if err != nil {
		return nil, fmt.Errorf("failed to read \"BSCONFIG_IPORT\" env var: %w", err)
	}

	bsconfigPort, err := strconv.Atoi(bsconfigPortStr)
	if err != nil {
		return nil, fmt.Errorf("failed to read \"BSCONFIG_IPORT\" env var: %w", err)
	}

	bsconfigDir, err := getEnvVar("BSCONFIG_IDIR")
	if err != nil {
		return nil, fmt.Errorf("failed to read \"BSCONFIG_IDIR\" env var: %w", err)
	}

	dc, err := getEnvVar("a_dc")
	if err != nil {
		return nil, fmt.Errorf("failed to read \"a_dc\" env var: %w", err)
	}

	workers, err := getWorkersCount(portoConn)
	if err != nil {
		return nil, fmt.Errorf("failed to get workers count: %w", err)
	}

	var sdCacheDir string
	awacsDir := getEnvVarOrDefault("AWACSLET_AWACS_DIR_PATH", "/awacs")
	if isDirAndWritable(awacsDir) {
		log.Printf("detected writable %s dir", awacsDir)
		sdCacheDir = path.Join(awacsDir, "sd_cache")
	} else {
		sdCacheDir = "./sd_cache"
	}

	return &Container{
		bsconfigDir,
		bsconfigPort,
		shmDir,
		path.Join(shmDir, "balancer"),
		logsDir,
		dc,
		workers,
		gracefulShutdownTimeout,
		gracefulShutdownCooldownTimeout,
		workersProviderPath,
		sdCacheDir,
		useSudoForShawshank == 1,
		fqInterfaces,
	}, nil
}

func prepareConfLocal() error {
	// In case when instancectl is used, environment variables could
	// be set from Conf.local file. This is an emulation for backward
	// compatibility.
	// Format:
	// ENV_<key>=<value>\n
	fd, err := os.Open("Conf.local")
	if err == nil {
		defer fd.Close()

		scanner := bufio.NewScanner(fd)
		for scanner.Scan() {
			for scanner.Scan() {
				line := strings.Split(scanner.Text(), "=")
				if len(line) == 2 && strings.HasPrefix(line[0], EnvPrefixString) {
					if err = setEnvVar(line[0][4:], line[1]); err != nil {
						return err
					}
				}
			}
		}

		err = scanner.Err()
	} else if os.IsNotExist(err) {
		return nil
	}

	return err
}

func connectToPorto() (porto_api.PortoAPI, error) {
	conn, err := porto_api.Dial()
	if err != nil {
		return nil, fmt.Errorf("failed to dial Porto: %w", err)
	}
	return conn, nil
}

func getPortoContainerName(portoConn porto_api.PortoAPI) (string, error) {
	absName, err := portoConn.GetProperty("self", "absolute_name")
	if err != nil {
		return "", fmt.Errorf("could not request \"absolute_path\" for current porto container: %w", err)
	}

	splitName := strings.Split(absName, "/")

	if len(splitName) < 3 {
		return "", fmt.Errorf("container path should have more than two levels: %s", absName)
	}

	rootName := strings.Join(splitName[:3], "/")

	return rootName, nil
}

func getWorkersCount(portoConn porto_api.PortoAPI) (int, error) {
	cpuLimitStr, err := getEnvVar("CPU_LIMIT")
	if err != nil {
		// Fallback for non-YP containers
		portoContainerName, err := getPortoContainerName(portoConn)
		if err != nil {
			return 0, fmt.Errorf("failed to get root container name for current porto container: %w", err)
		}

		if cpuLimitStr, err = portoConn.GetProperty(portoContainerName, "cpu_limit"); err != nil {
			return 0, fmt.Errorf("could not request \"cpu_limit\" for porto container [%s]: %w", portoContainerName, err)
		}
	}
	if !strings.HasSuffix(cpuLimitStr, "c") {
		return 0, errors.New("CPU_LIMIT env var value does not have suffix \"c\"")
	}
	cpuLimit, err := strconv.ParseFloat(strings.TrimSuffix(cpuLimitStr, "c"), 32)
	if err != nil {
		return 0, fmt.Errorf("failed to parse env var CPU_LIMIT or cpu_limit porto container property: %w", err)
	}
	cpuLimit = math.Ceil(cpuLimit)
	return int(cpuLimit), nil
}

func (c *Container) prepareCerts() error {
	pubDirPath := c.secretsDir
	privDirPath := filepath.Join(c.secretsDir, "priv")
	if err := os.MkdirAll(c.secretsDir, 0777); err != nil {
		return fmt.Errorf("failed to make dir %s: %w", c.secretsDir, err)
	}
	if err := os.MkdirAll(privDirPath, 0700); err != nil {
		return fmt.Errorf("failed to make dir %s: %w", privDirPath, err)
	}
	if err := os.Chmod(privDirPath, 0700); err != nil {
		return fmt.Errorf("failed to chmod 0700 dir %s: %w", privDirPath, err)
	}

	log.Printf("discovering certs")
	certTgzPaths, err := c.discoverCertTgzs()
	if err != nil {
		return fmt.Errorf("failed to discover cert tgzs: %w", err)
	}
	sort.Strings(certTgzPaths)
	log.Printf("discovered %d possible cert tgzs: %+q", len(certTgzPaths), certTgzPaths)

	for _, path := range certTgzPaths {
		log.Printf("unpacking %s", path)
		f, err := os.Open(path)
		if err != nil {
			return fmt.Errorf("failed to open %s: %w", path, err)
		}
		c, err := certs.LoadFromTarGz(f)
		_ = f.Close()
		if err != nil {
			return fmt.Errorf("failed to load cert %s: %w", path, err)
		}
		if err := c.Dump(pubDirPath, privDirPath); err != nil {
			return fmt.Errorf("failed to dump cert %s: %w", path, err)
		}
		log.Printf("unpacked %s", path)
	}
	return nil
}

func (c *Container) discoverCertTgzs() ([]string, error) {
	var tgzPaths []string
	files, err := ioutil.ReadDir(c.bsconfigDir)
	if err != nil {
		return nil, fmt.Errorf("failed to read dir %s: %w", c.bsconfigDir, err)
	}
	for _, fi := range files {
		if fi.IsDir() && strings.HasPrefix(fi.Name(), "secrets") {
			tgzPath := filepath.Join(c.bsconfigDir, fi.Name(), "secrets.tgz")
			_, err := os.Stat(tgzPath)
			if os.IsNotExist(err) {
				continue
			} else if err != nil {
				return nil, fmt.Errorf("failed to stat %s: %w", tgzPath, err)
			}
			tgzPaths = append(tgzPaths, tgzPath)
		}
	}
	return tgzPaths, nil
}

func (c *Container) runShawshank(shawshankPath string) error {
	var p string
	var args []string
	if c.useSudoForShawshank {
		p = "sudo"
		args = append(args, shawshankPath)
	} else {
		p = shawshankPath
	}

	cmd := exec.Command(p, args...)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	cmd.Dir = c.bsconfigDir
	cmd.Env = os.Environ()
	if err := cmd.Start(); err != nil {
		return fmt.Errorf("failed to start %s %v: %w", p, args, err)
	}
	if err := cmd.Wait(); err != nil {
		return fmt.Errorf("failed to execute %s %v: %w", p, args, err)
	}
	return nil
}

func (c *Container) processRequirements(reqs []string) error {
	for _, req := range reqs {
		switch req {
		case "shawshank":
			shawshankPath, err := exec.LookPath("shawshank")
			if err != nil {
				return fmt.Errorf("failed to locate shawshank binary: %w", err)
			}
			if err := c.runShawshank(shawshankPath); err != nil {
				return fmt.Errorf("failed to run shawshank: %w", err)
			}
		default:
			return fmt.Errorf("unknown requirement \"%s\"", req)
		}
	}
	return nil
}

func (c *Container) Prepare() error {
	if len(c.fqInterfaces) > 0 {
		err := c.InstallFqNetScheduler()
		if err != nil {
			return fmt.Errorf("failed to install fq net scheduler: %w", err)
		}
		log.Printf("installed fq qdisc for interfaces %s", strings.Join(c.fqInterfaces, ", "))
	}

	src := filepath.Join(c.bsconfigDir, "config.lua")
	dst := filepath.Join(c.bsconfigDir, "balancer.cfg")
	if err := copyFile(src, dst); err != nil {
		return fmt.Errorf("failed to copyFile %s to %s: %w", src, dst, err)
	}

	if err := os.MkdirAll(c.secretsDir, 0755); err != nil {
		return fmt.Errorf("failed to create %s: %w", c.secretsDir, err)
	}
	log.Printf("created %s", c.secretsDir)

	logsBalancerDir := filepath.Join(c.logsDir, "balancer")
	if err := os.MkdirAll(logsBalancerDir, 0755); err != nil {
		return fmt.Errorf("failed to create %s: %w", logsBalancerDir, err)
	}
	log.Printf("created %s", logsBalancerDir)

	if err := c.prepareCerts(); err != nil {
		return fmt.Errorf("failed to prepare certs: %w", err)
	}
	log.Printf("prepared certs")

	envStashPath := filepath.Join(c.shmDir, "env_save")
	err := stashAwacsEnvVars(envStashPath)
	log.Printf("stashed awacs env vars to %s", envStashPath)
	if err != nil {
		return fmt.Errorf("failed to stash env vars %s: %w", envStashPath, err)
	}

	containerSpecPath := filepath.Join(c.bsconfigDir, "awacs-balancer-container-spec.pb.json")
	if _, err := os.Stat(containerSpecPath); err != nil {
		if os.IsNotExist(err) {
			return nil
		}
		return fmt.Errorf("failed to stat %s: %w", containerSpecPath, err)
	}
	log.Printf("found container spec %s", containerSpecPath)

	containerSpec, err := ReadBalancerContainerSpec(containerSpecPath)
	if err != nil {
		return fmt.Errorf("failed to read %s: %w", containerSpecPath, err)
	}
	var reqNames []string
	for _, req := range containerSpec.Requirements {
		reqNames = append(reqNames, req.Name)
	}
	return c.processRequirements(reqNames)
}

func (c *Container) testBalancerConfig(cfgPath string, env []string) (bool, error) {
	balancerPath, err := exec.LookPath("./balancer")
	if err != nil {
		return false, fmt.Errorf("failed to find ./balancer: %w", err)
	}

	args := []string{"-K", cfgPath}
	for _, arg := range c.getBalancerConfigArgs() {
		args = append(args, fmt.Sprintf("-V%s=%s", arg.key, arg.value))
	}
	log.Printf("starting %s %+q", balancerPath, args)
	cmd := exec.Command(balancerPath, args...)
	cmd.Stdout = os.Stdout
	cmd.Stderr = os.Stderr
	cmd.Dir = c.bsconfigDir
	cmd.Env = append(env, os.Environ()...)

	if err := cmd.Start(); err != nil {
		return false, fmt.Errorf("failed to start balancer: %w", err)
	}

	if err := cmd.Wait(); err != nil {
		if exiterr, ok := err.(*exec.ExitError); ok {
			if status, ok := exiterr.Sys().(syscall.WaitStatus); ok {
				return status.ExitStatus() == 0, nil
			} else {
				return false, err
			}
		} else {
			return false, fmt.Errorf("failed to run %s: %w", balancerPath, err)
		}
	}
	return true, nil
}

func (c *Container) getBalancerConfigArgs() []KeyValue {
	return []KeyValue{
		{"DC", c.dc},
		{"port", strconv.Itoa(c.bsconfigPort)},
		{"get_workers_provider", c.workersProviderPath},
		{"sd_cache_dir", c.sdCacheDir},
	}
}

func (c *Container) callBalancer(path string, args []KeyValue) error {
	URL := fmt.Sprintf("http://localhost:%d/%s", c.bsconfigPort, path)
	req, err := http.NewRequest("GET", URL, nil)
	if err != nil {
		return fmt.Errorf("failed to create admin request: %w", err)
	}

	var buf strings.Builder
	for _, arg := range args {
		if buf.Len() > 0 {
			buf.WriteByte('&')
		}
		buf.WriteString(url.QueryEscape(arg.key))
		buf.WriteByte('=')
		buf.WriteString(url.QueryEscape(arg.value))
	}
	req.URL.RawQuery = buf.String()
	log.Println("calling " + req.URL.String())

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return err
	}
	defer resp.Body.Close()
	body, _ := ioutil.ReadAll(resp.Body)
	if resp.StatusCode != 200 {
		return fmt.Errorf("call failed (status code: %d, body: %s)", resp.StatusCode, string(body))
	}
	return nil
}

func (c *Container) StartBalancer() error {
	envStashPath := filepath.Join(c.shmDir, "env_save")
	err := stashAwacsEnvVars(envStashPath)
	log.Printf("stashed awacs env vars to %s", envStashPath)
	if err != nil {
		return fmt.Errorf("failed to stash env vars %s: %w", envStashPath, err)
	}
	ok, err := c.testBalancerConfig("./config.lua", os.Environ())
	if err != nil {
		return fmt.Errorf("failed to test balancer config: %w", err)
	}
	if !ok {
		return errors.New("balancer config test failed")
	}

	balancerPath, err := exec.LookPath("./balancer")
	if err != nil {
		return fmt.Errorf("failed to find ./balancer: %w", err)
	}
	args := []string{
		balancerPath,
		"balancer.cfg",
		"-L",
	}
	for _, arg := range c.getBalancerConfigArgs() {
		args = append(args, fmt.Sprintf("-V%s=%s", arg.key, arg.value))
	}
	if err := syscall.Exec(balancerPath, args, os.Environ()); err != nil {
		return fmt.Errorf("failed to exec %v: %w", args, err)
	}
	return nil
}

func (c *Container) ReloadConfig() error {
	envStashPath := filepath.Join(c.shmDir, "env_save")
	env, err := unstashAwacsEnvVars(envStashPath)
	if err != nil {
		return fmt.Errorf("failed to load env vars from %s: %w", envStashPath, err)
	}
	log.Printf("unstashed env vars from %s\n", envStashPath)

	src := filepath.Join(c.bsconfigDir, "config.lua")
	dst := filepath.Join(c.bsconfigDir, "balancer.cfg.new")
	if err := copyFile(src, dst); err != nil {
		return fmt.Errorf("failed to copy %s to %s: %w", src, dst, err)
	}
	log.Printf("copied %s to %s\n", src, dst)

	fmt.Println("testing balancer config ./balancer.cfg.new")
	ok, err := c.testBalancerConfig("./balancer.cfg.new", env)
	if err != nil {
		return fmt.Errorf("failed to test balancer config: %w", err)
	}
	if !ok {
		return errors.New("balancer config test failed")
	}

	log.Println("balancer config test succeeded")
	src = filepath.Join(c.bsconfigDir, "balancer.cfg")
	dst = filepath.Join(c.bsconfigDir, "balancer.cfg.old")
	if err := copyFile(src, dst); err != nil {
		return fmt.Errorf("failed to copy %s to %s: %w", src, dst, err)
	}
	log.Printf("copied %s to %s\n", src, dst)

	src = filepath.Join(c.bsconfigDir, "balancer.cfg.new")
	dst = filepath.Join(c.bsconfigDir, "balancer.cfg")
	if err := os.Rename(src, dst); err != nil {
		return fmt.Errorf("failed to move %s to %s: %w", src, dst, err)
	}
	log.Printf("renamed %s to %s\n", src, dst)

	params := []KeyValue{
		{"action", "reload_config"},
		{"new_config_path", "balancer.cfg"},
		{"skip_block_requests", "1"},
	}
	for _, arg := range c.getBalancerConfigArgs() {
		params = append(params, KeyValue{
			fmt.Sprintf("V_%s", arg.key), arg.value,
		})
	}
	log.Printf("calling admin, params: %+q\n", params)
	if err := c.callBalancer("admin", params); err != nil {
		return fmt.Errorf("failed to call admin: %w", err)
	}

	return nil
}

func (c *Container) Status() error {
	params := []KeyValue{
		{"action", "version"},
	}
	log.Printf("calling admin, params: %+q\n", params)
	return c.callBalancer("admin", params)
}

func (c *Container) Stop() error {
	params := []KeyValue{
		{"action", "graceful_shutdown"},
		{"timeout", c.gracefulShutdownTimeout},
		{"cooldown", c.gracefulShutdownCooldownTimeout},
		{"peek_timeout", "4s"},
	}
	log.Printf("calling admin, params: %+q\n", params)
	return c.callBalancer("admin", params)
}

func (c *Container) ReopenBalancerLog() error {
	params := []KeyValue{
		{"action", "reopenlog"},
	}
	log.Printf("calling admin, params: %+q\n", params)
	if err := c.callBalancer("admin", params); err != nil {
		return err
	}
	// Sleeping 10 seconds, see RUNTIMECLOUD-6820 for details
	log.Println("sleeping 10 seconds")
	time.Sleep(10 * time.Second)
	return nil
}

func (c *Container) GetWorkersCount() int {
	return c.workers
}

func (c *Container) StartPushClient() error {
	pushClientPath, err := exec.LookPath("./push-client")
	if err != nil {
		return fmt.Errorf("failed to find ./push-client: %w", err)
	}
	args := []string{
		pushClientPath,
		"-c",
		"./push-client_real.conf",
		"-f",
	}
	if err := syscall.Exec(pushClientPath, args, os.Environ()); err != nil {
		return fmt.Errorf("failed to exec %v: %w", args, err)
	}
	return nil
}

func (c *Container) ReopenPushClientLog() error {
	pidStr, err := ioutil.ReadFile("./pids/push-client")
	if err != nil {
		return fmt.Errorf("failed to read ./pids/push-client: %v", err)
	}
	pid, err := strconv.Atoi(string(pidStr))
	if err != nil {
		return fmt.Errorf("failed to parse pid from ./pids/push-client: %v", err)
	}
	// https://st.yandex-team.ru/LBOPS-5773#5f3b88cbba6b9e23a8765b84
	err = syscall.Kill(pid, syscall.SIGHUP)
	if err != nil {
		return fmt.Errorf("failed to send SIGHUP to %d: %v", pid, err)
	}
	return nil
}

func (c Container) InstallFqNetScheduler() error {
	for _, iface := range c.fqInterfaces {
		txCount, err := countTxQueuesByInterface(iface)
		if err != nil {
			return fmt.Errorf("failed to count tx queues for %s: %w", iface, err)
		}
		err = installFqNetScheduler(iface, txCount)
		if err != nil {
			return fmt.Errorf("failed to install fq net scheduler for %s: %w", iface, err)
		}
	}
	return nil
}

func (c *Container) CheckFqNetSchedulerEnabled() bool {
	if len(c.fqInterfaces) == 0 {
		return false
	}

	for _, iface := range c.fqInterfaces {
		if !checkFqNetSchedulerEnabled(iface) {
			return false
		}
	}

	return true
}
