// +build linux

package container

import (
	"fmt"
	"github.com/vishvananda/netlink"
	"golang.org/x/sys/unix"
	"io"
	"io/ioutil"
	"os"
	"strings"
)

func isDirAndWritable(path string) bool {
	fi, err := os.Stat(path)
	return err == nil && fi.IsDir() && unix.Access(path, unix.W_OK) == nil
}

func copyFile(src, dst string) error {
	sourceFileStat, err := os.Stat(src)
	if err != nil {
		return err
	}

	if !sourceFileStat.Mode().IsRegular() {
		return fmt.Errorf("%s is not a regular file", src)
	}

	source, err := os.Open(src)
	if err != nil {
		return err
	}
	defer source.Close()

	destination, err := os.Create(dst)
	if err != nil {
		return err
	}
	defer destination.Close()
	_, err = io.Copy(destination, source)
	return err
}

func countTxQueuesByInterface(iface string) (uint16, error) {
	txCount := uint16(0)
	if x, err := ioutil.ReadDir(fmt.Sprintf("/sys/class/net/%s/queues/", iface)); err == nil {
		for _, i := range x {
			if strings.HasPrefix(i.Name(), "tx-") {
				txCount += 1
			}
		}
	} else {
		return 0, fmt.Errorf("could not get information about interface %s", iface)
	}
	return txCount, nil
}

type Mq struct {
	netlink.QdiscAttrs
}

func (qdisc *Mq) Attrs() *netlink.QdiscAttrs {
	return &qdisc.QdiscAttrs
}
func (qdisc *Mq) Type() string {
	return "mq"
}

func installFqNetScheduler(iface string, count uint16) error {
	if count < 1 {
		return fmt.Errorf("incorrect tx queues count %d", count)
	} else if count == 1 {
		// Set fq as root rule for interface with one queue.
		link, err := netlink.LinkByName(iface)
		if err != nil {
			return err
		}
		return netlink.QdiscReplace(&netlink.Fq{
			QdiscAttrs: netlink.QdiscAttrs{
				LinkIndex: link.Attrs().Index,
				Parent:    netlink.HANDLE_ROOT,
			},
		})
	} else {
		// Set mq as root and leaves as fq for multi-queue interfaces.
		link, err := netlink.LinkByName(iface)
		if err != nil {
			return err
		}
		err = netlink.QdiscReplace(&Mq{
			QdiscAttrs: netlink.QdiscAttrs{
				LinkIndex: link.Attrs().Index,
				Handle:    netlink.MakeHandle(1, 0),
				Parent:    netlink.HANDLE_ROOT,
			},
		})
		if err != nil {
			return err
		}
		for i := uint16(1); i <= count; i++ {
			err := netlink.QdiscReplace(&netlink.Fq{
				QdiscAttrs: netlink.QdiscAttrs{
					LinkIndex: link.Attrs().Index,
					Handle:    netlink.MakeHandle(1, i),
					Parent:    netlink.MakeHandle(1, 0),
				},
			})
			if err != nil {
				return err
			}
		}

		return nil
	}
}

func checkFqNetSchedulerEnabled(iface string) bool {
	link, err := netlink.LinkByName(iface)
	if err != nil {
		return false
	}

	qdisc, err := netlink.QdiscList(link)
	if err != nil {
		return false
	}

	if len(qdisc) == 0 {
		return false
	}

	// Check is fq root rule for interface with one queue.
	if len(qdisc) == 1 && qdisc[0].Type() == "fq" {
		return true
	}

	// Check is mq root and leaves are fq for multi-queue interfaces.
	if len(qdisc) > 1 && qdisc[0].Type() == "mq" {
		for i := 1; i < len(qdisc); i++ {
			if qdisc[i].Type() != "fq" {
				return false
			}
		}
		return true
	}

	return false
}
