package awacs_test

import (
	"context"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"os"
	"reflect"
	"testing"
	"time"

	"github.com/golang/protobuf/proto"
	"github.com/stretchr/testify/require"
	"google.golang.org/genproto/protobuf/field_mask"

	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	pb "a.yandex-team.ru/infra/awacs/proto"
	"a.yandex-team.ru/library/go/test/requirepb"
)

type testClient struct {
	awacs.Client
}

func newClient(t *testing.T) *testClient {
	t.Helper()

	token := os.Getenv("AWACS_TOKEN")
	if token == "" {
		t.Skip("AWACS_TOKEN is required as this client interacts with (test) awacs API.")
	}

	c := awacs.NewClient(awacs.WithToken(token), awacs.WithTestAPI(), awacs.WithDebug())

	return &testClient{Client: c}
}

func TestClient_ListNamespaces(t *testing.T) {
	c := newClient(t)

	ctx := context.Background()
	rsp, err := c.ListNamespaces(ctx, &pb.ListNamespacesRequest{
		FieldMask: &field_mask.FieldMask{Paths: []string{"meta"}},
	}, nil)
	require.NoError(t, err)
	require.NotEmpty(t, rsp.GetNamespaces())
}

func TestClient_error(t *testing.T) {
	server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		w.WriteHeader(http.StatusInternalServerError)
		_, _ = w.Write([]byte("some error"))
	}))
	defer server.Close()

	ctx := context.Background()
	url := server.URL + "/ListNamespaces/"
	c := awacs.NewClient(awacs.WithDebug(), awacs.WithAPIURL(url))
	_, err := c.ListNamespaces(ctx, &pb.ListNamespacesRequest{}, nil)
	require.Error(t, err)
}

func TestClient(t *testing.T) {
	for _, tc := range []struct {
		endpoint string
		req      proto.Message
		rsp      proto.Message
	}{
		{
			endpoint: "ListNamespaces",
			req: &pb.ListNamespacesRequest{
				FieldMask: &field_mask.FieldMask{Paths: []string{"meta"}},
			},
			rsp: &pb.ListNamespacesResponse{
				Namespaces: []*pb.Namespace{
					{
						Meta: &pb.NamespaceMeta{
							Id:       "test-namespace",
							Category: "test-namespace",
							Auth: &pb.Auth{
								Type: pb.Auth_STAFF,
								Staff: &pb.StaffAuth{
									Owners: &pb.StaffUsersGroup{Logins: []string{"verytable"}},
								},
							},
							AbcServiceId: 470,
						},
					},
				},
				Total: 1,
			},
		},
		{
			endpoint: "CreateNamespace",
			req: &pb.CreateNamespaceRequest{
				Meta: &pb.NamespaceMeta{
					Id:       "test-namespace",
					Category: "test-namespace",
					Auth: &pb.Auth{
						Type: pb.Auth_STAFF,
						Staff: &pb.StaffAuth{
							Owners: &pb.StaffUsersGroup{Logins: []string{"verytable"}},
						},
					},
					AbcServiceId: 470,
				},
			},
			rsp: &pb.CreateNamespaceResponse{
				Namespace: &pb.Namespace{
					Meta: &pb.NamespaceMeta{
						Id:       "test-namespace",
						Category: "test-namespace",
						Auth: &pb.Auth{
							Type: pb.Auth_STAFF,
							Staff: &pb.StaffAuth{
								Owners: &pb.StaffUsersGroup{Logins: []string{"verytable"}},
							},
						},
						AbcServiceId: 470,
					},
				},
			},
		},
		{
			endpoint: "CreateBalancer",
			req: &pb.CreateBalancerRequest{
				Meta: &pb.BalancerMeta{
					NamespaceId: "test-namespace",
					Id:          "test-balancer",
				},
			},
			rsp: &pb.Status{
				Status:  "BAD",
				Message: "Something went band",
				Reason:  "Very bad!",
				Code:    503,
			},
		},
	} {
		t.Run(tc.endpoint, func(t *testing.T) {
			server := httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				body, err := ioutil.ReadAll(r.Body)
				require.NoError(t, err)

				in := reflect.New(reflect.TypeOf(tc.req).Elem()).Interface().(proto.Message)
				err = proto.Unmarshal(body, in)
				require.NoError(t, err)
				requirepb.Equal(t, tc.req, in)

				if status, ok := (tc.rsp).(*pb.Status); ok {
					w.WriteHeader(int(status.Code))
				}
				out, err := proto.Marshal(tc.rsp)
				require.NoError(t, err)
				_, _ = w.Write(out)
			}))
			defer server.Close()

			ctx, cancel := context.WithTimeout(context.Background(), time.Second)
			defer cancel()

			url := fmt.Sprintf("%s/%s/", server.URL, tc.endpoint)
			c := awacs.NewClient(awacs.WithDebug(), awacs.WithAPIURL(url))
			out := reflect.ValueOf(c).MethodByName(tc.endpoint).Call(
				[]reflect.Value{reflect.ValueOf(ctx), reflect.ValueOf(tc.req)},
			)

			if status, ok := (tc.rsp).(*pb.Status); ok {
				// test failed calls
				err := out[1].Interface().(error)
				var apierr *awacs.APIError
				require.True(t, errors.As(err, &apierr))
				require.Equal(t, apierr.Message, status.Message)
				require.Equal(t, apierr.Reason, status.Reason)
				require.Equal(t, apierr.Status, status.Status)
				require.Equal(t, apierr.Code, status.Code)
			} else {
				// test successful calls
				var err error
				require.Equal(t, err, out[1].Interface())
				requirepb.Equal(t, tc.rsp, out[0].Interface())
			}
		})
	}
}
