package awacs

import (
	"context"
	"fmt"
	"path"

	"github.com/go-resty/resty/v2"
	"github.com/golang/protobuf/proto"
	"google.golang.org/grpc"

	awacspb "a.yandex-team.ru/infra/awacs/proto"
	"a.yandex-team.ru/library/go/core/xerrors"
)

// conn is an implementation of grpc.ClientConnInterface that supports only unary RPCs.
type conn struct {
	url string
	c   *resty.Client
}

func newConn(url string, c *resty.Client) *conn {
	return &conn{
		url: url,
		c:   c,
	}
}

type APIError struct {
	Status  string
	Message string
	Reason  string
	Code    int32
}

func (ae *APIError) Error() string {
	return fmt.Sprintf("bad response status: %d (%s); %s; %s", ae.Code, ae.Status, ae.Reason, ae.Message)
}

func (ae *APIError) IsNotFound() bool {
	return ae != nil && ae.Code == 404
}

func IgnoreNotFound(err error) error {
	if ae, ok := err.(*APIError); ok && ae.IsNotFound() {
		return nil
	}
	return err
}

// Invoke serializes args into wire format and sends it as a POST request body
// to the http API endpoint inferred from the method.
// Response is deserialized into reply.
func (c *conn) Invoke(ctx context.Context, method string, args interface{}, reply interface{}, opts ...grpc.CallOption) error {
	in, ok := args.(proto.Message)
	if !ok {
		return xerrors.Errorf("invalid argument type: %T", args)
	}

	out, ok := reply.(proto.Message)
	if !ok {
		return xerrors.Errorf("invalid reply type: %T", reply)
	}

	data, err := proto.Marshal(in)
	if err != nil {
		return err
	}

	rsp, err := c.c.R().
		SetContext(ctx).
		SetBody(data).
		Post(c.url + path.Base(method) + "/")
	if err != nil {
		return err
	}

	if rsp.StatusCode() < 200 || rsp.StatusCode() >= 300 {
		status := &awacspb.Status{}
		if proto.Unmarshal(rsp.Body(), status) != nil {
			// Failed to unmarshal into a structured error, return an unstructured one
			return xerrors.Errorf("bad response status: %d; response: %s", rsp.StatusCode(), rsp.Body())
		}
		return &APIError{
			Status:  status.Status,
			Message: status.Message,
			Reason:  status.Reason,
			Code:    status.Code,
		}
	}

	return proto.Unmarshal(rsp.Body(), out)
}

func (c *conn) NewStream(ctx context.Context, desc *grpc.StreamDesc, method string, opts ...grpc.CallOption) (grpc.ClientStream, error) {
	panic("implement me")
}
