package client

import (
	"a.yandex-team.ru/infra/awacs/clients/go/awacs"
	awacspb "a.yandex-team.ru/infra/awacs/proto"
	"context"
	"errors"
)

type Client struct {
	c awacs.Client
}

func NewAwacsClient(apiUrl, oauthToken string) *Client {
	return &Client{
		c: awacs.NewClient(
			awacs.WithToken(oauthToken),
			awacs.WithAPIURL(apiUrl),
		),
	}
}

func IsNotFound(err error) bool {
	apierr := &awacs.APIError{}
	return errors.As(err, &apierr) && apierr.Code == 404
}

func (c *Client) GetDnsRecord(ctx context.Context, namespaceId, dnsRecordId string) (*awacspb.DnsRecord, error) {
	req := &awacspb.GetDnsRecordRequest{NamespaceId: namespaceId, Id: dnsRecordId}
	rsp, err := c.c.GetDnsRecord(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetDnsRecord(), nil
}

func (c *Client) CreateDnsRecord(ctx context.Context, namespaceId, dnsRecordId string, spec *awacspb.DnsRecordSpec) (*awacspb.DnsRecord, error) {
	meta := &awacspb.DnsRecordMeta{
		Id:          dnsRecordId,
		NamespaceId: namespaceId,
		Auth: &awacspb.Auth{
			Type: awacspb.Auth_STAFF,
		},
	}
	req := &awacspb.CreateDnsRecordRequest{
		Meta: meta,
		Req: &awacspb.CreateDnsRecordRequest_Spec{
			Spec: spec,
		},
	}
	rsp, err := c.c.CreateDnsRecord(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetDnsRecord(), nil
}

func (c *Client) UpdateDnsRecord(ctx context.Context, namespaceId, dnsRecordId, version, comment string, spec *awacspb.DnsRecordSpec) (*awacspb.DnsRecord, error) {
	meta := &awacspb.DnsRecordMeta{
		Id:          dnsRecordId,
		NamespaceId: namespaceId,
		Version:     version,
		Comment:     comment,
	}
	req := &awacspb.UpdateDnsRecordRequest{
		Meta: meta,
		Spec: spec,
	}
	rsp, err := c.c.UpdateDnsRecord(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetDnsRecord(), nil
}

func (c *Client) GetBackend(ctx context.Context, namespaceId, backendId string) (*awacspb.Backend, error) {
	req := &awacspb.GetBackendRequest{NamespaceId: namespaceId, Id: backendId}
	rsp, err := c.c.GetBackend(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetBackend(), nil
}

func (c *Client) CreateBackend(ctx context.Context, namespaceId, backendId string, spec *awacspb.BackendSpec) (*awacspb.Backend, error) {
	meta := &awacspb.BackendMeta{
		Id:          backendId,
		NamespaceId: namespaceId,
		Auth: &awacspb.Auth{
			Type: awacspb.Auth_STAFF,
		},
	}
	req := &awacspb.CreateBackendRequest{
		Meta: meta,
		Spec: spec,
	}
	rsp, err := c.c.CreateBackend(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetBackend(), nil
}

func (c *Client) UpdateBackend(ctx context.Context, namespaceId, backendId, version, comment string, spec *awacspb.BackendSpec) (*awacspb.Backend, error) {
	meta := &awacspb.BackendMeta{}
	meta.Id = backendId
	meta.NamespaceId = namespaceId
	meta.Version = version
	meta.Comment = comment
	req := &awacspb.UpdateBackendRequest{
		Meta: meta,
		Spec: spec,
	}
	rsp, err := c.c.UpdateBackend(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetBackend(), nil
}

func (c *Client) GetEndpointSet(ctx context.Context, namespaceId, endpointSetId string) (*awacspb.EndpointSet, error) {
	req := &awacspb.GetEndpointSetRequest{NamespaceId: namespaceId, Id: endpointSetId}
	rsp, err := c.c.GetEndpointSet(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetEndpointSet(), nil
}

func (c *Client) CreateEndpointSet(ctx context.Context, namespaceId, endpointSetId, backendVersion string, spec *awacspb.EndpointSetSpec) (*awacspb.EndpointSet, error) {
	meta := &awacspb.EndpointSetMeta{
		Id:          endpointSetId,
		NamespaceId: namespaceId,
		Auth: &awacspb.Auth{
			Type: awacspb.Auth_STAFF,
		},
	}
	req := &awacspb.CreateEndpointSetRequest{
		Meta:           meta,
		Spec:           spec,
		BackendVersion: backendVersion,
	}
	rsp, err := c.c.CreateEndpointSet(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetEndpointSet(), nil
}

func (c *Client) UpdateEndpointSet(ctx context.Context, namespaceId, endpointSetId, version, backendVersion, comment string, spec *awacspb.EndpointSetSpec) (*awacspb.EndpointSet, error) {
	meta := &awacspb.EndpointSetMeta{}
	meta.Id = endpointSetId
	meta.NamespaceId = namespaceId
	meta.Version = version
	meta.Comment = comment
	req := &awacspb.UpdateEndpointSetRequest{
		Meta:           meta,
		Spec:           spec,
		BackendVersion: backendVersion,
	}
	rsp, err := c.c.UpdateEndpointSet(ctx, req)
	if err != nil {
		return nil, err
	}
	return rsp.GetEndpointSet(), nil
}
