/*
Copyright 2021.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package controllers

import (
	awacspb "a.yandex-team.ru/infra/awacs/proto"
	"context"
	"errors"
	"fmt"
	"os"
	"strings"
	"time"

	"github.com/go-logr/logr"
	"k8s.io/apimachinery/pkg/runtime"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"

	awacsv1 "a.yandex-team.ru/infra/awacs/kube/example/api/v1"
	awacsclient "a.yandex-team.ru/infra/awacs/kube/example/client"
)

const AwacsNamespaceID = "kubertc-romanovich-tmp"
const DefaultAwacsAPIURL = "http://test-awacs.n.yandex-team.ru/api/"

// DnsRecordReconciler reconciles a DnsRecord object
type DnsRecordReconciler struct {
	client.Client
	Log    logr.Logger
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=awacs.yandex-team.ru,resources=dnsrecords,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=awacs.yandex-team.ru,resources=dnsrecords/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=awacs.yandex-team.ru,resources=dnsrecords/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the DnsRecord object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.8.3/pkg/reconcile
func (r *DnsRecordReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	log := r.Log.WithValues("dnsrecord", req.NamespacedName)
	errResult := ctrl.Result{RequeueAfter: time.Second * 30}
	var dnsRecord awacsv1.DnsRecord
	if err := r.Get(ctx, req.NamespacedName, &dnsRecord); err != nil {
		log.Error(err, "unable to fetch DnsRecord")
		// we'll ignore not-found errors, since they can't be fixed by an immediate
		// requeue (we'll need to wait for a new notification), and we can get them
		// on deleted requests.
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	fqdn := dnsRecord.Spec.FQDN
	ipv6Address := dnsRecord.Spec.IPv6Address

	awacsToken := os.Getenv("AWACS_TOKEN")
	if awacsToken == "" {
		log.Error(errors.New("env var AWACS_TOKEN is not set"), "")
		os.Exit(1)
	}
	awacsAPIURL := os.Getenv("AWACS_API_URL")
	if awacsAPIURL == "" {
		awacsAPIURL = DefaultAwacsAPIURL
	}
	awacsClient := awacsclient.NewAwacsClient(awacsAPIURL, awacsToken)

	awacsBackendSpec := &awacspb.BackendSpec{
		Selector: &awacspb.BackendSelector{
			Type: awacspb.BackendSelector_MANUAL,
		},
	}
	awacsBackend, err := awacsClient.GetBackend(ctx, AwacsNamespaceID, fqdn)
	if err != nil {
		if awacsclient.IsNotFound(err) {
			awacsBackend = nil
		} else {
			return errResult, fmt.Errorf("failed to communicate with awacs API: %w", err)
		}
	}
	if awacsBackend == nil {
		awacsBackend, err = awacsClient.CreateBackend(ctx, AwacsNamespaceID, fqdn, awacsBackendSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to create awacs backend: %w", err)
		}
	} else {
		awacsBackend, err = awacsClient.UpdateBackend(ctx, AwacsNamespaceID, fqdn, awacsBackend.Meta.Version,
			"Hello from KUBERTC", awacsBackendSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to update awacs backend: %w", err)
		}
	}
	log.Info(fmt.Sprintf("Synced backend %s:%s to version %s", AwacsNamespaceID, fqdn, awacsBackend.Meta.Version))

	awacsEndpointSetSpec := &awacspb.EndpointSetSpec{
		Instances: []*awacspb.EndpointSetSpec_Instance{
			{
				Host:     fqdn,
				Port:     31337,
				Weight:   1,
				Ipv6Addr: ipv6Address,
			},
		},
	}
	awacsEndpointSet, err := awacsClient.GetEndpointSet(ctx, AwacsNamespaceID, fqdn)
	if err != nil {
		if awacsclient.IsNotFound(err) {
			awacsEndpointSet = nil
		} else {
			return errResult, fmt.Errorf("failed to communicate with awacs API: %w", err)
		}
	}
	if awacsEndpointSet == nil {
		awacsEndpointSet, err = awacsClient.CreateEndpointSet(ctx, AwacsNamespaceID, fqdn, awacsBackend.Meta.Version, awacsEndpointSetSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to create awacs endpoint set: %w", err)
		}
	} else {
		awacsEndpointSet, err = awacsClient.UpdateEndpointSet(ctx, AwacsNamespaceID, fqdn, awacsEndpointSet.Meta.Version,
			awacsBackend.Meta.Version, "Hello from KUBERTC", awacsEndpointSetSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to update awacs endpoint set: %w", err)
		}
	}
	log.Info(fmt.Sprintf("Synced endpoint set %s:%s to version %s", AwacsNamespaceID, fqdn, awacsEndpointSet.Meta.Version))

	var hostname string
	var nameServerId string
	if strings.HasSuffix(fqdn, ".in.yandex.net") {
		nameServerId = "in.yandex.net"
		hostname = strings.TrimSuffix(fqdn, ".in.yandex.net")
	} else if strings.HasSuffix(fqdn, ".in.yandex-team.ru") {
		nameServerId = "in.yandex-team.ru"
		hostname = strings.TrimSuffix(fqdn, ".in.yandex-team.ru")
	} else {
		return errResult, fmt.Errorf("can not process FQDN %s", fqdn)
	}
	awacsDnsRecordSpec := &awacspb.DnsRecordSpec{
		NameServer: &awacspb.NameServerFullId{NamespaceId: "infra", Id: nameServerId},
		Address: &awacspb.DnsRecordSpec_Address{
			Zone: hostname,
			Backends: &awacspb.DnsBackendsSelector{
				Backends: []*awacspb.DnsBackendsSelector_Backend{
					{Id: fqdn},
				},
			},
		},
	}

	awacsDnsRecord, err := awacsClient.GetDnsRecord(ctx, AwacsNamespaceID, fqdn)
	if err != nil {
		if awacsclient.IsNotFound(err) {
			awacsDnsRecord = nil
		} else {
			return errResult, fmt.Errorf("failed to communicate with awacs API: %w", err)
		}
	}
	if awacsDnsRecord == nil {
		awacsDnsRecord, err = awacsClient.CreateDnsRecord(ctx, AwacsNamespaceID, fqdn, awacsDnsRecordSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to create awacs DNS record: %w", err)
		}
	} else {
		awacsDnsRecord, err = awacsClient.UpdateDnsRecord(ctx, AwacsNamespaceID, fqdn, awacsDnsRecord.Meta.Version,
			"Hello from KUBERTC", awacsDnsRecordSpec)
		if err != nil {
			return errResult, fmt.Errorf("failed to update awacs DNS record: %w", err)
		}
	}
	log.Info(fmt.Sprintf("Synced DNS record %s:%s to version %s", AwacsNamespaceID, fqdn, awacsDnsRecord.Meta.Version))

	return ctrl.Result{}, nil
}

// SetupWithManager sets up the controller with the Manager.
func (r *DnsRecordReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&awacsv1.DnsRecord{}).
		Complete(r)
}
