# coding: utf-8
import calendar
import re
from datetime import datetime

import aniso8601
import click
import requests


# DEFAULT_NANNY_API_URL = 'http://nanny.test:8080/v2/'
DEFAULT_NANNY_API_URL = 'https://nanny.yandex-team.ru/v2/'
UTC_TZINFO = aniso8601.utcoffset.UTCOffset(name='utc', minutes=0)


class NannyClient(object):
    def __init__(self, api_url, oauth_token):
        self.api_url = api_url.rstrip('/') + '/'
        self.oauth_token = oauth_token
        self.s = requests.Session()
        self.s.headers = {
            'Authorization': 'OAuth {}'.format(self.oauth_token),
        }

    def iterate_all_services(self):
        url = self.api_url + 'services/'
        limit = 200
        skip = 0
        params = {'limit': limit}
        while True:
            params['skip'] = skip
            r = self.s.get(url, params=params)
            r.raise_for_status()
            services = r.json()['result']
            for service in services:
                yield service
            if len(services) != limit:
                break
            skip += limit

    def activate_snapshot(self, service_id, snapshot_id):
        data = {
            "type": "SET_TARGET_STATE",
            "content": {
                "is_enabled": True,
                "snapshot_id": snapshot_id,
                "comment": "Force activate snapshot",
                "prepare_recipe": "default",
                "recipe": "common"
            }
        }
        url = self.api_url + 'services/{}/events/'.format(service_id)
        r = self.s.post(url, json=data)
        r.raise_for_status()

    def activate_last_active_snapshot(self, service_id):
        data = {
            "type": "ROLLBACK",
            "content": {
                "comment": "Force activate snapshot",
                "recipe": "common",
                "set_as_current": True
            }
        }
        url = self.api_url + 'services/{}/events/'.format(service_id)
        r = self.s.post(url, json=data)
        r.raise_for_status()

    def pause_service(self, service_id):
        data = {
            "type": "PAUSE_ACTIONS",
            "content": {
                "comment": "Pause actions in service",
            }
        }
        url = self.api_url + 'services/{}/events/'.format(service_id)
        r = self.s.post(url, json=data)
        r.raise_for_status()

    def resume_service(self, service_id):
        data = {
            "type": "RESUME_ACTIONS",
            "content": {
                "comment": "Resume actions in service",
            }
        }
        url = self.api_url + 'services/{}/events/'.format(service_id)
        r = self.s.post(url, json=data)
        r.raise_for_status()


def iso8601_to_utc_datetime(iso_ctime):
    return aniso8601.parse_datetime(iso_ctime)


def test_iso8601_to_utc_datetime():
    # UTC 09.08.2005 13:31:42
    start_ctime = '2005-08-09T18:31:42+05'
    start_dt = iso8601_to_utc_datetime(start_ctime)

    # UTC 09.08.2005 18:31:42
    dt = datetime(year=2005, month=8, day=9, hour=18, minute=31, second=42)
    # Use calendar.timegm for UTC
    ts = calendar.timegm(dt.timetuple())
    last_runtime_dt = datetime.utcfromtimestamp(ts).replace(tzinfo=UTC_TZINFO)

    assert last_runtime_dt >= start_dt
    assert not last_runtime_dt <= start_dt

    # UTC 09.08.2005 18:31:42
    start_ctime = '2005-08-09T18:31:42+00'
    start_dt = iso8601_to_utc_datetime(start_ctime)
    assert last_runtime_dt == start_dt

    # UTC 09.08.2005 19:31:42
    start_ctime = '2005-08-09T18:31:42-01'
    start_dt = iso8601_to_utc_datetime(start_ctime)
    assert not last_runtime_dt >= start_dt
    assert last_runtime_dt <= start_dt


@click.group()
def cli():
    pass


@cli.command('force_activate')
@click.option('--nanny-token', envvar='NANNY_TOKEN', required=True)
@click.option('--nanny-url', default=DEFAULT_NANNY_API_URL)
@click.option('--select', '-s', type=click.Choice(['ALL', 'BY_IDS']), required=True)
@click.option('--start-ctime')
@click.option('--end-ctime')
@click.option('--ids-path')
@click.option('--comment-regexp')
@click.option('--op', type=click.Choice(['rollback-current-activating', 'force-activate-current']),
              default='force-activate-current')
def force_activate(nanny_token, nanny_url, select, op, start_ctime=None, end_ctime=None, ids_path=None,
                   comment_regexp=None):
    matchers = []
    match_service = lambda service: service['info_attrs']['content'].get('type') == 'AWACS_BALANCER'
    matchers.append(match_service)

    if select == 'BY_IDS':
        if not ids_path:
            click.echo('--ids-path must be given if selector is BY_IDS')
            return
        with open(ids_path) as f:
            service_ids = set(_id.rstrip() for _id in f.readlines())
        match_id = lambda service: service["_id"] in service_ids
        matchers.append(match_id)

    last_runtime_dt = lambda service: datetime.utcfromtimestamp(
        service['runtime_attrs']['change_info']['ctime'] // 1000).replace(tzinfo=UTC_TZINFO)
    if start_ctime:
        start_dt = iso8601_to_utc_datetime(start_ctime)
        match_start_dt = lambda service: last_runtime_dt(service) >= start_dt
        matchers.append(match_start_dt)
    if end_ctime:
        end_dt = iso8601_to_utc_datetime(end_ctime)
        match_end_dt = lambda service: last_runtime_dt(service) <= end_dt
        matchers.append(match_end_dt)

    if comment_regexp:
        match_comment = lambda service: bool(
            re.match(comment_regexp, service['runtime_attrs']['change_info']['comment']))
        matchers.append(match_comment)

    client = NannyClient(nanny_url, nanny_token)
    to_be_processed = []
    click.echo('The following services will be proccessed:')
    for service in client.iterate_all_services():
        if all(matcher(service) for matcher in matchers):
            service_id = service['_id']
            to_be_processed.append((service_id, service['runtime_attrs']['_id']))
            click.echo(service_id)

    if not to_be_processed:
        click.echo('No services to be processed')
        return

    click.echo("")
    if op == 'rollback-current-activating':
        click.echo("Do you want to activate last active snapshots? Print yes or no")
    elif op == 'force-activate-current':
        click.echo("Do you want to force activation?")
    c = click.prompt('Print yes or no', type=click.Choice(['yes', 'no']))
    if c == 'no':
        return

    for service_id, snapshot_id in to_be_processed:
        try:
            if op == 'rollback-current-activating':
                client.activate_last_active_snapshot(service_id)
            elif op == 'force-activate-current':
                client.activate_snapshot(service_id, snapshot_id)
        except requests.HTTPError as e:
            print('Failed on {}'.format(service_id))
            print(e)
            if 400 <= e.response.status_code < 500:
                print(e.response.json())


if __name__ == '__main__':
    test_iso8601_to_utc_datetime()
    cli()
