# coding: utf-8
import json
import time

import urllib3
import yaml


urllib3.disable_warnings()

from collections import defaultdict

import click

from infra.awacs.proto import api_pb2, model_pb2
from infra.awacs.tools.awacsalerting.src import context


@click.group('-draft-commands', help='Unstable commands')
def cli():
    pass


@cli.command('list-service-info-attrs')
@context.pass_cli_context
def list_service_info_attrs(cli_context):  # type: (context.CliContext) -> None
    req_pb = api_pb2.ListNamespacesRequest()
    req_pb.field_mask.paths.append('meta')
    req_pb.field_mask.paths.append('spec.alerting')
    req_pb.field_mask.paths.append('alerting_sync_status')
    resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
    i = 0
    balancers_count = 0
    for ns_pb in resp_pb.namespaces:
        if not ns_pb.spec.HasField('alerting'):
            continue
        ns_recipients = []
        for rule in ns_pb.spec.alerting.juggler_raw_notify_rules.balancer:
            ns_recipients += [l.replace('@', '') for l in yaml.safe_load(rule.template_kwargs).get('login')]

        ns_recipients = list(set(ns_recipients))

        balancer_pbs = cli_context.list_balancers_pbs(ns_pb.meta.id)
        logins_not_found = []
        groups_not_found = []
        for b_pb in balancer_pbs:
            nanny_service_id = b_pb.spec.config_transport.nanny_static_file.service_id
            info_attrs = cli_context.nanny_client.get_service_info_attrs(nanny_service_id)
            deploy_monitoring_settings = info_attrs['content']['monitoring_settings']['deploy_monitoring']
            logins = [login for login in deploy_monitoring_settings['content']['responsible']['logins'] if
                      login != 'cplb_admins']
            groups = [int(g) for g in deploy_monitoring_settings['content']['responsible']['groups']]
            groups_slags = []
            if groups:
                resp = cli_context.staff_client.get_groups_by_ids(groups)
                groups_slags = [s['url'] for s in resp.values()]

            logins_not_found += [l for l in logins if l not in ns_recipients]
            groups_not_found += [g for g in groups_slags if g not in ns_recipients]

        if logins_not_found or groups_not_found:
            i += 1
            nanny_recipients = list(set(logins_not_found)) + list(set(groups_not_found))
            click.echo("{:62s}{:62s} | {}".format(
                ns_pb.meta.id,
                ','.join(nanny_recipients),
                ','.join(ns_recipients),
            )
            )


@cli.command('fill-yp-lite-balancers-instance-tags')
@click.option('--does-not-skip-taxi-or-maps', is_flag=True, default=False)
@click.option('--balancers-limit', type=click.INT, default=27)
@click.option('--balancers-limit-countdown', type=click.INT, default=300)
@context.pass_cli_context
def fill_yp_lite_balancers_instance_tags(cli_context, does_not_skip_taxi_or_maps, balancers_limit,
                                         balancers_limit_countdown):
    """

    https://st.yandex-team.ru/SWATOPS-101

    :type cli_context: context.CliContext
    :type does_not_skip_taxi_or_maps: bool
    :type balancers_limit: int
    :type balancers_limit_countdown: int
    """
    req_pb = api_pb2.ListNamespacesRequest()
    req_pb.field_mask.paths.append('meta.id')
    req_pb.field_mask.paths.append('spec.alerting')
    resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
    total = 0
    in_progress_balancers = {}
    for i, ns_pb in enumerate(resp_pb.namespaces):
        click.echo('[{}/{:4s}]{:60s}'.format(str(i + 1), str(len(resp_pb.namespaces)), ns_pb.meta.id), nl=False)
        if ns_pb.spec.HasField('alerting'):
            click.echo('{:10s}skip, alerting enabled'.format(''))
            continue
        balancer_pbs = cli_context.list_balancers_pbs(ns_pb.meta.id)
        balancer_w_location_pbs = [b_pb for b_pb in balancer_pbs
                                   if b_pb.meta.location.type == cli_context.LOCATION_YP_CLUSTER]
        if not balancer_w_location_pbs:
            click.echo('{:10s}skip, no yp-lite balancers'.format(''))
            continue

        click.echo('{:10s}https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'.format('', ns_pb.meta.id))
        total += 1

        for b_pb in balancer_pbs:
            click.echo("{:60s}".format(b_pb.meta.id), nl=False)

            if not b_pb.meta.HasField('location'):
                click.echo('SKIP, meta.location is empty'.format(
                    model_pb2.BalancerMeta.Location.Type.Name(b_pb.meta.location.type)
                ))
                continue

            if b_pb.meta.location.type != cli_context.LOCATION_YP_CLUSTER:
                click.echo('SKIP, meta.location.type has unsupported value'.format(
                    model_pb2.BalancerMeta.Location.Type.Name(b_pb.meta.location.type)
                ))
                continue

            click.echo("{:10s}".format(b_pb.meta.location.yp_cluster), nl=False)

            if b_pb.spec.config_transport.nanny_static_file.HasField('instance_tags'):
                instance_tags_pb = b_pb.spec.config_transport.nanny_static_file.instance_tags
                click.echo('SKIP, already filled: itype: {}, ctype: {}, prj: {}'.format(
                    instance_tags_pb.itype, instance_tags_pb.ctype, instance_tags_pb.prj))
                continue

            if not does_not_skip_taxi_or_maps and ('taxi' in b_pb.meta.id or 'maps.yandex.net' in b_pb.meta.id):
                click.echo('SKIP, is taxi or maps'.format(
                    model_pb2.BalancerMeta.Location.Type.Name(b_pb.meta.location.type)
                ))
                continue

            spec_is_active = (b_pb.status.validated.status == 'True' and
                              b_pb.status.in_progress.status == 'False' and
                              b_pb.status.active.status == 'True')

            if not spec_is_active:
                click.echo('SKIP, spec is not active')
                continue

            aspects_tags_pb = cli_context.get_balancer_aspects_tags_pb(ns_pb.meta.id, b_pb.meta.id)
            if not aspects_tags_pb.itype or not aspects_tags_pb.ctype or not aspects_tags_pb.prj:
                click.echo('SKIP, aspects_tags_pb have empty tags value: {}, {}, {}'.format(
                    aspects_tags_pb.itype, aspects_tags_pb.ctype, aspects_tags_pb.prj))
                continue

            click.echo("itype:{:10s} ctype:{:10s} prj:{:60s}".format(
                aspects_tags_pb.itype[0], aspects_tags_pb.ctype[0], aspects_tags_pb.prj[0]), nl=False)

            update_balancer_request_pb = api_pb2.UpdateBalancerRequest()
            update_balancer_request_pb.meta.CopyFrom(b_pb.meta)
            update_balancer_request_pb.meta.comment = (
                "https://st.yandex-team.ru/SWATOPS-101, fill spec.config_transport.nanny_static_file.instance_tags "
                "with actual values")
            instance_tags_pb = b_pb.spec.config_transport.nanny_static_file.instance_tags
            instance_tags_pb.itype = aspects_tags_pb.itype[0]
            instance_tags_pb.ctype = aspects_tags_pb.ctype[0]
            instance_tags_pb.prj = aspects_tags_pb.prj[0]

            update_balancer_request_pb.spec.CopyFrom(b_pb.spec)
            cli_context.balancer_stub.update_balancer(update_balancer_request_pb)
            click.echo('DONE')
            in_progress_balancers[b_pb.meta.id] = time.time()

        current_time = time.time()
        in_progress_balancers = {b_id: started_at for b_id, started_at in in_progress_balancers.items()
                                 if started_at + balancers_limit_countdown > current_time}

        started_times = [current_time - started_at for started_at in in_progress_balancers.values()
                         if started_at + balancers_limit_countdown > current_time]

        if len(started_times) > balancers_limit:
            min_started_time = min(sorted(started_times, reverse=True)[:3])
            click.echo('Sleeping {} seconds...'.format(balancers_limit_countdown - min_started_time))
            time.sleep(balancers_limit_countdown - min_started_time)


@cli.command('fill-gencfg-balancers-instance-tags')
@click.option('--gencfg-tags-file', type=click.File(), required=True)
@click.option('--balancers-limit', type=click.INT, default=27)
@click.option('--balancers-limit-countdown', type=click.INT, default=300)
@context.pass_cli_context
def fill_gencfg_balancers_instance_tags(cli_context, gencfg_tags_file, balancers_limit, balancers_limit_countdown):
    """
    https://st.yandex-team.ru/SWATOPS-105

    :type cli_context: context.CliContext
    :type balancers_limit: int
    :type gencfg_tags_file: str
    :type balancers_limit_countdown: int
    """
    gencfg_tags = json.load(open(gencfg_tags_file))
    req_pb = api_pb2.ListNamespacesRequest()
    req_pb.field_mask.paths.append('meta.id')
    req_pb.field_mask.paths.append('spec.alerting')
    resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
    total = 0
    in_progress_balancers = {}
    i = 0
    for ns_pb in resp_pb.namespaces:
        # i += 1
        # click.echo('[{:4s}/{:4s}]{:60s}'.format(str(i + 1), str(len(resp_pb.namespaces)), ns_pb.meta.id), nl=False)
        if any([v in ns_pb.meta.id for v in ('taxi', 'maps', 'alice')]):
            # click.echo('SKIP exlude names')
            continue
        if ns_pb.spec.HasField('alerting'):
            # click.echo('SKIP alerting enabled')
            continue
        balancer_pbs = cli_context.list_balancers_pbs(ns_pb.meta.id)
        balancer_w_yp_lite_location_pbs = [b_pb for b_pb in balancer_pbs
                                           if b_pb.meta.location.type == cli_context.LOCATION_YP_CLUSTER]
        if balancer_w_yp_lite_location_pbs:
            # click.echo('SKIP has yp lite balancers')
            continue

        balancer_wo_location_pbs = [b_pb for b_pb in balancer_pbs if not b_pb.meta.HasField('location')]

        if not balancer_wo_location_pbs:
            # click.echo('SKIP withot locations not found')
            continue

        if any([not cli_context.balancer_spec_is_active(b_pb) for b_pb in balancer_wo_location_pbs]):
            # click.echo('SKIP has not active spec')
            continue

        balancers_tags = {b_pb.meta.id: cli_context.get_balancer_tags(b_pb, gencfg_tags)
                          for b_pb in balancer_wo_location_pbs}

        if any([v is None for v in balancers_tags.values()]):
            # click.echo('SKIP empty tags')
            continue

        dcs = [v['dc'] for v in balancers_tags.values()]
        if len(dcs) != len(set(dcs)):
            # click.echo('SKIP cross dc')
            continue
        i += 1
        click.echo('[{:4s}/{:4s}]https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'.format(
            str(i + 1), str(len(resp_pb.namespaces)), ns_pb.meta.id))
        # click.echo('PROCESS')
        for b_pb in balancer_wo_location_pbs:
            balancer_tags = balancers_tags[b_pb.meta.id]
            total += 1
            click.echo("[{:4s}/    ]{:60s}{:10s}".format(str(total), b_pb.meta.id, balancer_tags['dc']), nl=False)

            if b_pb.spec.config_transport.nanny_static_file.HasField('instance_tags'):
                instance_tags_pb = b_pb.spec.config_transport.nanny_static_file.instance_tags
                click.echo('SKIP, already filled: itype: {}, ctype: {}, prj: {}'.format(
                    instance_tags_pb.itype, instance_tags_pb.ctype, instance_tags_pb.prj))
                continue

            click.echo("itype:{:10s} ctype:{:10s} prj:{:60s}".format(
                balancer_tags['itype'], balancer_tags['ctype'], balancer_tags['prj']), nl=False)

            update_balancer_request_pb = api_pb2.UpdateBalancerRequest()
            update_balancer_request_pb.meta.CopyFrom(b_pb.meta)
            update_balancer_request_pb.meta.location.type = model_pb2.BalancerMeta.Location.GENCFG_DC
            update_balancer_request_pb.meta.location.gencfg_dc = balancer_tags['dc'].upper()
            update_balancer_request_pb.meta.comment = (
                "https://st.yandex-team.ru/SWATOPS-105, fill spec.config_transport.nanny_static_file.instance_tags "
                "with actual values")
            instance_tags_pb = b_pb.spec.config_transport.nanny_static_file.instance_tags
            instance_tags_pb.itype = balancer_tags['itype']
            instance_tags_pb.ctype = balancer_tags['ctype']
            instance_tags_pb.prj = balancer_tags['prj']

            update_balancer_request_pb.spec.CopyFrom(b_pb.spec)
            cli_context.balancer_stub.update_balancer(update_balancer_request_pb)
            click.echo('DONE')
            in_progress_balancers[(ns_pb.meta.id, b_pb.meta.id)] = time.time()
        current_time = time.time()
        in_progress_balancers = {b_id: started_at for b_id, started_at in in_progress_balancers.items()
                                 if started_at + balancers_limit_countdown > current_time}

        started_times = [current_time - started_at for started_at in in_progress_balancers.values()
                         if started_at + balancers_limit_countdown > current_time]

        if len(started_times) > balancers_limit:
            min_started_time = min(sorted(started_times, reverse=True)[:3])
            click.echo('Sleeping {} seconds...'.format(balancers_limit_countdown - min_started_time), nl=False)
            wakeup_time = time.time() + (balancers_limit_countdown - min_started_time)

            while time.time() < wakeup_time:
                all_statuses = {}
                for ns_id, balancer_id in in_progress_balancers.keys():
                    all_statuses[ns_id, balancer_id] = cli_context.check_balancer_spec_is_active(ns_id, balancer_id)
                in_progress_count = len([s for s in all_statuses.values() if not s])
                if in_progress_count:
                    click.echo("\rSleeping... {} in progress".format(in_progress_count), nl=False)
                    time.sleep(20)
                else:
                    sleep_left = wakeup_time - time.time()
                    click.echo("\rAll done, continue after {} seconds...".format(sleep_left))
                    time.sleep(sleep_left)


@cli.command('migrate-nanny-deploy-monitoring-to-awacs')
@context.pass_cli_context
def migrate_nanny_deploy_monitoring_to_awacs(cli_context):  # type: (context.CliContext) -> None

    for namespace in cli_context.list_namespaces():
        if not namespace.is_alerting_enabled:
            continue

        for balancer in namespace.list_balancers():
            nanny_service_info_attrs = balancer.nanny_service_info_attrs
            assert nanny_service_info_attrs

    # for ns_pb in resp_pb.namespaces:
    #     # if not ns_pb.spec.HasField('alerting'):
    #     #     continue
    #     # if semantic_version.Version(ns_pb.spec.alerting.version) < semantic_version.Version('0.0.4'):
    #     #     continue
    #     if ns_pb.meta.id != 'i-dyachkov-testing':
    #         continue
    #
    #     awacs_recipients = []
    #     for rule in ns_pb.spec.alerting.juggler_raw_notify_rules.balancer:
    #         awacs_recipients += [l.replace('@', '') for l in yaml.safe_load(rule.template_kwargs).get('login')]
    #
    #     awacs_recipients = list(set(awacs_recipients))
    #
    #     nanny_logins = set()
    #     nanny_groups = set()
    #     nanny_telegram_chats = set()
    #
    #     balacers_pbs = cli_context.list_balancers_pbs(ns_pb.meta.id)
    #     for b_pb in balacers_pbs:
    #         nanny_service_id = b_pb.spec.config_transport.nanny_static_file.service_id
    #         info_attrs = cli_context.nanny_client.get_service_info_attrs(nanny_service_id)
    #         deploy_monitoring_settings = info_attrs['content']['monitoring_settings']['deploy_monitoring']
    #         is_enabled = deploy_monitoring_settings['is_enabled']
    #         if is_enabled:
    #             logins = deploy_monitoring_settings['content']['responsible']['logins']
    #             for login in logins:
    #                 if login == 'cplb_admins':
    #                     continue
    #                 person_data = cli_context.staff_client.get_person_by_login(login)
    #                 if person_data is None:
    #                     nanny_telegram_chats.add(login)
    #                     continue
    #                 if person_data['official']['is_robot'] or person_data['official']['is_dismissed']:
    #                     continue
    #                 nanny_logins.add(login)
    #
    #             group_ids = {int(g) for g in deploy_monitoring_settings['content']['responsible']['groups']}
    #
    #             # cli_context.nanny_client.disable_deploy_monitoring(


@cli.command('find-namespaces-by-prj')
@click.argument('input-file', type=click.File())
@context.pass_cli_context
def find_namespaces_by_prj(cli_context, input_file):  # type: (context.CliContext, file) -> None
    prj_to_namespace = defaultdict(lambda: set())

    for namespace in cli_context.list_namespaces():

        for balancer in namespace.list_balancers():
            if balancer.instance_tags is not None:
                prj_to_namespace[balancer.instance_tags.prj].add(namespace.namespace_id)

    assert prj_to_namespace

    for line in input_file.readlines():
        gencfg_group, tags_raw = line.strip().split(':')

        tags = {t.split('=')[0]: t.split('=')[1] for t in tags_raw.strip().split(';')}

        prj_tag = tags['prj']

        namespace_ids = prj_to_namespace.get(prj_tag, None)

        click.echo("{} -> {}".format(line.strip(), ','.join(namespace_ids) if namespace_ids else None))


if __name__ == '__main__':
    cli(obj=context.CliContext())
