# coding: utf-8
import time
import typing

import click
import nanny_rpc_client
import semantic_version
import six
import yaml
from infra.awacs.proto import api_pb2, model_pb2
from tabulate import tabulate

from infra.awacs.tools.awacsalerting.src import context, models


@click.group('stable')
@click.option('--awacs-token', envvar='AWACS_TOKEN', required=True)
@click.option('--nanny-token', envvar='NANNY_TOKEN', required=True)
@click.option('--abc-token', envvar='ABC_TOKEN', required=False)
@click.option('--staff-token', envvar='STAFF_TOKEN', required=False)
@click.option('--vcr-cassette-dir', envvar='VCR_CASSETTE_DIR', required=False)
@click.option('--awacs-rpc-url', envvar='AWACS_RPC_URL', default=context.DEFAULT_AWACS_RPC_URL)
@click.pass_context
def cli(ctx, awacs_rpc_url, abc_token, awacs_token, nanny_token, staff_token, vcr_cassette_dir):
    awacs_rpc = nanny_rpc_client.RetryingRpcClient(rpc_url=awacs_rpc_url, oauth_token=awacs_token)
    ctx.obj.init(
        awacs_rpc=awacs_rpc,
        abc_token=abc_token,
        nanny_token=nanny_token,
        staff_token=staff_token,
        vcr_cassette_library_dir=vcr_cassette_dir
    )


@cli.command('list-enabled', help="List awacs namespaces with enabled alerting")
@click.option('--order-by', type=click.Choice(['namespace_id', 'last_attempt']), default='namespace_id')
@click.option('--desc', is_flag=True)
@context.pass_cli_context
def list_enabled(cli_context, order_by, desc):
    current_time = time.time()
    req_pb = api_pb2.ListNamespacesRequest()
    req_pb.field_mask.paths.append('meta')
    req_pb.field_mask.paths.append('spec.alerting')
    req_pb.field_mask.paths.append('alerting_sync_status')
    resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
    i = 0

    def sorted_key_func(_ns_pb):
        if order_by == 'namespace_id':
            return _ns_pb.meta.id
        elif order_by == 'last_attempt':
            return _ns_pb.alerting_sync_status.last_attempt.finished_at.seconds
        else:
            return _ns_pb.meta.id

    result = [('#', 'Namespace Ui Url', 'Ver', 'Ctl Status', 'Time ago')]

    for ns_pb in sorted(resp_pb.namespaces, key=sorted_key_func, reverse=desc):
        if not ns_pb.spec.HasField('alerting'):
            continue
        i += 1
        result.append((
            i,
            'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/alerting/'.format(ns_pb.meta.id),
            ns_pb.spec.alerting.version,
            ns_pb.alerting_sync_status.last_attempt.succeeded.status,
            str(int((current_time - ns_pb.alerting_sync_status.last_attempt.finished_at.seconds) / 60)) + ' m'
        ))

    click.echo(tabulate(result, headers='firstrow'))


@cli.command('list-errors', help="List errors awacs namespace-ctl(detailed)")
@context.pass_cli_context
def list_errors(cli_context):  # type: (context.CliContext) -> None
    current_time = time.time()
    req_pb = api_pb2.ListNamespacesRequest()
    req_pb.field_mask.paths.append('meta')
    req_pb.field_mask.paths.append('spec.alerting')
    req_pb.field_mask.paths.append('alerting_sync_status')
    resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
    i = 0

    for ns_pb in resp_pb.namespaces:
        if not ns_pb.spec.HasField('alerting'):
            continue
        if ns_pb.alerting_sync_status.last_attempt.succeeded.status != 'False':
            continue

        i += 1
        click.echo("{:5s}{:120s}{:10s}{:10s}{:10s}{}".format(
            str(i),
            'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/alerting/'.format(ns_pb.meta.id),
            ns_pb.spec.alerting.version,
            ns_pb.alerting_sync_status.last_attempt.succeeded.status,
            str(int((current_time - ns_pb.alerting_sync_status.last_attempt.finished_at.seconds) / 60)) + ' m',
            ns_pb.alerting_sync_status.last_attempt.succeeded.message,
        ))


def _iter_namespaces(cli_context):
    limit = 1000
    page = 0
    total = 1
    while total > page * limit:
        req_pb = api_pb2.ListNamespacesRequest(limit=limit, skip=page * limit)
        resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
        total = resp_pb.total
        page += 1
        for ns_pb in resp_pb.namespaces:
            yield ns_pb


@cli.command('bump-version', help='Bump alerting version')
@click.option('--new-version', type=click.STRING, required=True)
@click.option('--st-task-key', type=click.STRING, required=True)
@click.option('--limit', type=click.INT, default=100)
@click.option('--sleep-after-sync', type=click.INT, default=3)
@click.option('--namespace-id', type=click.STRING)
@context.pass_cli_context
def bump_version(cli_context, new_version, st_task_key, limit, sleep_after_sync, namespace_id=None):
    """

    :type cli_context: context.CliContext
    :type new_version: six.text_type
    :type st_task_key: six.text_type
    :type limit: int
    :type sleep_after_sync: int
    :type namespace_id: typing.Optional[six.text_type]
    """

    total = 0
    if namespace_id is not None:
        namespaces = [cli_context.list_namespaces([namespace_id])[0]._pb]
    else:
        namespaces = _iter_namespaces(cli_context)

    for ns_pb in namespaces:
        if not ns_pb.spec.HasField('alerting'):
            continue
        if semantic_version.Version(ns_pb.spec.alerting.version) >= semantic_version.Version(new_version):
            continue

        total += 1
        update_ns_request_pb = api_pb2.UpdateNamespaceRequest()
        update_ns_request_pb.meta.CopyFrom(ns_pb.meta)
        update_ns_request_pb.meta.comment = 'https://st.yandex-team.ru/{}: bump alerting version to {}'.format(
            st_task_key, new_version)
        update_ns_request_pb.spec.CopyFrom(ns_pb.spec)
        update_ns_request_pb.spec.alerting.version = new_version
        cli_context.namespace_stub.update_namespace(update_ns_request_pb)
        last_sync_time = ns_pb.alerting_sync_status.last_attempt.finished_at.seconds
        current_sync_time = ns_pb.alerting_sync_status.last_attempt.finished_at.seconds

        click.echo("\r[{:3s}]https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/alerting/ - {} -> {}".format(
            str(total), ns_pb.meta.id, ns_pb.spec.alerting.version, new_version))
        while current_sync_time <= last_sync_time:
            resp = cli_context.namespace_stub.get_namespace_alerting_config(api_pb2.GetNamespaceAlertingConfigRequest(
                namespace_id=ns_pb.meta.id))
            current_sync_time = resp.alerting_sync_status.last_attempt.finished_at.seconds
            click.echo("\r{}".format('Not synced'), nl=False)
            time.sleep(sleep_after_sync)
        click.echo("\r{}".format('Synced'), nl=False)
        time.sleep(sleep_after_sync)

        if total > limit:
            break


@cli.command('set-preset', help='Set alerting preset for namespace')
@click.argument('namespace', type=str)
@click.argument('preset', type=str)
@context.pass_cli_context
def set_preset(
    cli_context,
    namespace,
    preset,
):
    req_pb = api_pb2.ApplyNamespaceAlertingPresetRequest(namespace_id=namespace)
    req_pb.preset = getattr(model_pb2.NamespaceSpec, preset)
    cli_context.namespace_stub.apply_namespace_alerting_preset(req_pb)


@cli.command('enable-alerting', help='Enable alerting for YP-Lite powered namespaces')
@click.option('--allow-ns-filters', default='')
@click.option('--skip-ns-filters', default='taxi,maps,alice')
@click.option('--exact-namespaces', default='')
@click.option('--limit', type=click.INT, default=1)
@click.option('--alerting-version', default='0.1.0')
@click.option('--sleep', type=click.INT, default=30)
@click.option('--no-act', is_flag=True)
@context.pass_cli_context
def enable_alerting(
    cli_context,
    skip_ns_filters, allow_ns_filters,
    exact_namespaces,
    limit,
    alerting_version='0.1.0',
    sleep=30,
    no_act=False,
):
    """

    :type cli_context:
    :param skip_ns_filters:
    :param allow_ns_filters:
    :param limit:
    :return:
    """
    exact_namespaces = [f.strip() for f in exact_namespaces.split(' ') if f.strip()]

    if exact_namespaces:
        req_pb = api_pb2.ListNamespacesRequest()
        req_pb.query.id_in.extend(exact_namespaces)
        resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
        namespaces = resp_pb.namespaces
    else:
        namespaces = list(_iter_namespaces(cli_context))

    skip_ns_filters = [f.strip() for f in skip_ns_filters.split(',') if f.strip()]
    allow_ns_filters = [f.strip() for f in allow_ns_filters.split(',') if f.strip()]
    total = 0
    for i, ns_pb in enumerate(namespaces):
        if not limit:
            break
        if allow_ns_filters and not any([f in ns_pb.meta.id for f in allow_ns_filters]):
            # click.echo('{} not found in ns.meta.id'.format(allow_ns_filters))
            continue

        if not allow_ns_filters and skip_ns_filters and any([f in ns_pb.meta.id for f in skip_ns_filters]):
            # click.echo('{} found in ns.meta.id'.format(skip_ns_filters))
            continue

        if exact_namespaces and ns_pb.meta.id not in exact_namespaces:
            continue

        if not ns_pb.meta.abc_service_id:
            click.echo('{}: meta.abc_service_id has empty value'.format(ns_pb.meta.id))
            continue

        ns_is_ready, reason = cli_context.is_ready_namespace_for_alerting(ns_pb)

        if not ns_is_ready:
            # click.echo('{}: skip: {}'.format(ns_pb.meta.id, reason))
            continue

        _, recipients, recipients_count = cli_context.get_default_alerting_recipients(ns_pb.meta.abc_service_id)

        if not recipients:
            click.echo('{}: skip, recipients not found'.format(ns_pb.meta.id))
            continue
        total += 1
        alerting_url = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/alerting/'.format(ns_pb.meta.id)
        namespace_url = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'.format(ns_pb.meta.id)
        click.echo('[{}/{:4s}]{}  '.format(str(total), str(len(namespaces)), alerting_url), nl=False)

        limit -= 1

        annotations_to_set = {}

        if recipients_count == 0:
            annotations_to_set['alerting_no_receivers'] = 'yes'
        if recipients_count < 2:
            annotations_to_set['alerting_too_little_receivers'] = 'yes'
        elif recipients_count > 15:
            annotations_to_set['alerting_too_many_receivers'] = 'yes'

        if any(recipient.endswith('_services_management') for recipient in recipients):
            annotations_to_set['alerting_service_management_included'] = 'yes'

        abc_service = str(cli_context.abc_client.get_abc_service(ns_pb.meta.abc_service_id)['slug'].lower())
        abc_url = 'https://abc.yandex-team.ru/services/{}/'.format(abc_service)
        all_owner_logins = set(ns_pb.meta.auth.staff.owners.logins)
        all_owner_logins.update(
            cli_context.staff_client.get_groups_members_by_ids(ns_pb.meta.auth.staff.owners.group_ids))
        all_owner_logins.update(cli_context.staff_client.get_groups_members_by_slugs([
            "svc_" + recipient for recipient in recipients
            if recipient.endswith('_services_management')
        ]))
        all_owner_logins = [str(login) for login in all_owner_logins][:30]

        if not no_act:
            cli_context.enable_namespace_alerting(ns_pb, recipients, alerting_version, annotations_to_set)

            if 'alerting_service_management_included' in annotations_to_set:
                with open('management_tickets.yaml', 'a') as f:
                    f.write(yaml.dump({
                        ns_pb.meta.id: {
                            'assignee': all_owner_logins[0] if all_owner_logins else None,
                            'context': {
                                'owner_logins': all_owner_logins,
                                'abc_url': abc_url,
                                'abc_slug': abc_service,
                                'alerting_url': alerting_url,
                                'namespace_url': namespace_url,
                                'namespace_name': ns_pb.meta.id,
                            },
                            'invitee': all_owner_logins,
                        }}))

        click.echo('done with {} {} - {}'.format(recipients_count, recipients, ', '.join(
            '{}={}'.format(k, v) for k, v in annotations_to_set.items())))

        if not no_act:
            time.sleep(5)
            resp = cli_context.namespace_stub.get_namespace_alerting_config(api_pb2.GetNamespaceAlertingConfigRequest(
                namespace_id=ns_pb.meta.id))
            click.echo(resp.current_config.yasm_alerts_panel_url)
            st = None
            while not resp.alerting_sync_status.last_attempt.succeeded.status:
                resp = cli_context.namespace_stub.get_namespace_alerting_config(
                    api_pb2.GetNamespaceAlertingConfigRequest(
                        namespace_id=ns_pb.meta.id))
                st = resp.alerting_sync_status.last_attempt.succeeded.status
                click.echo("\r{}".format('Not started' if not st else st), nl=False)
                time.sleep(5)
            click.echo('\rSynced with status {}, sleep {} seconds...'.format(st, sleep))
            time.sleep(sleep)


@cli.command('check-balancers-instance-tags-status',
             help='Check awacs balancers instance tags with nanny service attrs')
@context.pass_cli_context
def check_balancers_instance_tags_status(cli_context):
    """

    :type cli_context: context.CliContext
    """
    exclude_namespaces = []
    desync_instance_tags_balancers = []  # type: typing.List[typing.Tuple[models.Balancer, model_pb2.InstanceTags, model_pb2.InstanceTags]]

    xdc_balancers_result = []  # type: typing.List[typing.Tuple[models.Balancer, typing.List[six.text_type]]]
    empty_nanny_tags_balancers_result = []  # type: typing.List[models.Balancer]

    not_full_yp_lite_instance_tags = []
    setup_instance_tags = []  # type: typing.List[typing.Tuple[models.Balancer, model_pb2.InstanceTags]]

    unexpected_errors_balancers = []

    namespaces = cli_context.list_namespaces()
    namespace_count, namespace_processed = len(namespaces), 0
    for namespace in namespaces:
        namespace_processed += 1
        for balancer in namespace.list_balancers():
            if balancer.platform == balancer.PLATFORM_YP_LITE:
                try:
                    active_revision_data = balancer.get_nanny_active_revision_data()
                except Exception as e:
                    unexpected_errors_balancers.append((balancer, e))
                    continue

                if not active_revision_data.instance_tags.is_full:
                    not_full_yp_lite_instance_tags.append((balancer, active_revision_data.instance_tags))
                    continue

                nanny_instance_tags = model_pb2.InstanceTags()
                nanny_instance_tags.ctype = active_revision_data.instance_tags.ctype[0]
                nanny_instance_tags.itype = active_revision_data.instance_tags.itype[0]
                nanny_instance_tags.prj = active_revision_data.instance_tags.prj[0]

                if balancer.instance_tags is None:
                    setup_instance_tags.append((balancer, nanny_instance_tags))
                elif balancer.instance_tags != nanny_instance_tags:

                    desync_instance_tags_balancers.append((balancer, balancer.instance_tags, nanny_instance_tags))

            elif balancer.platform == balancer.PLATFORM_GENCFG:
                try:
                    runtime_attrs = balancer.get_service_runtime_attrs()
                except Exception as e:
                    unexpected_errors_balancers.append((balancer, e))
                    continue

                gencfg_fixed_orthogonal_tags = runtime_attrs.gencfg_fixed_orthogonal_tags
                if gencfg_fixed_orthogonal_tags:
                    nanny_instance_tags = model_pb2.InstanceTags()
                    nanny_instance_tags.ctype = gencfg_fixed_orthogonal_tags['ctype']
                    nanny_instance_tags.itype = gencfg_fixed_orthogonal_tags['itype']
                    nanny_instance_tags.prj = gencfg_fixed_orthogonal_tags['prj']

                    if balancer.instance_tags is None:
                        setup_instance_tags.append((balancer, nanny_instance_tags))
                    elif balancer.instance_tags != nanny_instance_tags:
                        desync_instance_tags_balancers.append((balancer, balancer.instance_tags, nanny_instance_tags))
                else:
                    try:
                        active_revision_data = balancer.get_nanny_active_revision_data()
                    except Exception as e:
                        unexpected_errors_balancers.append((balancer, e))
                        continue

                    if len(active_revision_data.locations) > 1:
                        xdc_balancers_result.append((balancer, active_revision_data.locations))
                    else:
                        empty_nanny_tags_balancers_result.append(balancer)

                    exclude_namespaces.append(balancer.namespace_id)

        click.echo(
            '\rProcessing... {}/{} (setup:{}, desync: {}, xdc:{}, empty:{}, not_full_yp:{}, errors:{})'.format(
                namespace_processed,
                namespace_count,
                len(setup_instance_tags),
                len(desync_instance_tags_balancers),
                len(xdc_balancers_result),
                len(empty_nanny_tags_balancers_result),
                len(not_full_yp_lite_instance_tags),
                len(unexpected_errors_balancers),
            ), nl=False)

    click.echo(' done')

    for_setup_instance_tags = [(i + 1, b.namespace_id, b.balancer_id, b.platform, t.itype, t.ctype, t.prj)
                               for i, (b, t) in enumerate(setup_instance_tags) if
                               b.namespace_id not in exclude_namespaces]
    click.echo('\nSet instance tags balancers:')
    headers = ['#', 'namespace', 'balancer', 'platform', 'itype', 'ctype', 'prj']
    click.echo(tabulate(for_setup_instance_tags, headers=headers))

    click.echo('\nDesync instance tags balancers:')
    headers = ['#', 'namespace', 'balancer', 'platform', 'itype', 'ctype', 'prj']
    click.echo(tabulate(((i + 1, b.namespace_id, b.balancer_id, b.platform,
                          "{} != {}".format(ta.itype, tn.itype) if ta.itype != tn.itype else "",
                          "{} != {}".format(ta.ctype, tn.ctype) if ta.ctype != tn.ctype else "",
                          "{} != {}".format(ta.prj, tn.prj)) if ta.prj != tn.prj else ""
                         for i, (b, ta, tn) in enumerate(desync_instance_tags_balancers)
                         if b.namespace_id not in exclude_namespaces), headers=headers))

    click.echo('\nXDC balancers:')
    click.echo(
        tabulate(((i + 1, b.namespace_id, b.balancer_id, b.platform, ",".join(l))
                  for i, (b, l) in enumerate(xdc_balancers_result)),
                 headers=['#', 'namespace', 'balancer', 'platform', 'locations']))
    #
    click.echo('\nW/O nanny orthogonal tags balancers:')
    click.echo(tabulate(((i + 1, b.namespace_id, b.balancer_id, b.platform)
                         for i, b in enumerate(empty_nanny_tags_balancers_result)),
                        headers=['#', 'namespace', 'balancer', 'platform']))

    click.echo('\nNot full yp lite instance tags:')
    click.echo(tabulate(((i + 1, b.namespace_id, b.balancer_id, t.itype, t.ctype, t.prj)
                         for i, (b, t) in enumerate(not_full_yp_lite_instance_tags)),
                        headers=['#', 'namespace', 'balancer', 'itype', 'ctype', 'prj']))
    #
    click.echo('\nUnexpected errors balancers:')
    click.echo(tabulate(((i + 1, b.namespace_id, b.balancer_id, str(err))
                         for i, (b, err) in enumerate(unexpected_errors_balancers)),
                        headers=['#', 'Namespace ID', 'Balancer ID', 'Error']))


@cli.command('sync-balancers-instance-tags', help='Set nanny instance tags to awacs balancer spec')
@click.argument('st-task-id', type=click.STRING)
@click.option('--namespace-id', type=click.STRING)
@click.option('--namespace-ids-file', type=click.File())
@context.pass_cli_context
def sync_balancers_instance_tags(cli_context, st_task_id, namespace_id, namespace_ids_file):
    """

    :type cli_context: context.CliContext
    :type namespace_id: six.text_type
    :type namespace_ids_file: click.File
    :type st_task_id: six.text_type
    """
    if (not namespace_id and not namespace_ids_file) or (namespace_id and namespace_ids_file):
        raise click.BadOptionUsage('--namespace-id or --namespace-ids-file should be passed')

    namespace_ids = [namespace_id] if namespace_id else [ns.strip() for ns in namespace_ids_file.readlines()
                                                         if ns.strip()]
    empty_nanny_tags_balancers_result = []
    spec_is_not_active_balancer = []
    namespaces = cli_context.list_namespaces(namespace_ids)
    namespace_count, namespace_processed = len(namespaces), 0
    balancer_update_breaker = models.BalancersSpecUpdateBreaker(
        max_in_flight=10,
        in_flight_period=300
    )
    for namespace in namespaces:
        click.echo("[###] Processing https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/ "
                   "namespace...".format(namespace.namespace_id))
        namespace_processed += 1
        for balancer in namespace.list_balancers():
            if not balancer.spec_is_active:
                spec_is_not_active_balancer.append(balancer)
                click.echo('{} - spec is not active'.format(balancer.balancer_id))
                continue

            try:
                nanny_instance_tags = balancer.get_nanny_service_instance_tags()
            except Exception as e:
                click.echo('{} - Error({}): {}'.format(balancer.balancer_id, e.__class__.__name__, e.message))
                continue

            if nanny_instance_tags is None:
                empty_nanny_tags_balancers_result.append(balancer)
                click.echo('{} - nanny tags empty'.format(balancer.balancer_id))
            elif nanny_instance_tags != balancer.instance_tags:
                balancer.pb.spec.config_transport.nanny_static_file.instance_tags.CopyFrom(nanny_instance_tags)
                update_balancer_request_pb = api_pb2.UpdateBalancerRequest()
                update_balancer_request_pb.meta.CopyFrom(balancer.pb.meta)
                update_balancer_request_pb.meta.comment = (
                    "{}: update spec.config_transport.nanny_static_file.instance_tags "
                    "with actual values".format(st_task_id)
                )
                update_balancer_request_pb.spec.CopyFrom(balancer.pb.spec)
                cli_context.balancer_stub.update_balancer(update_balancer_request_pb)
                balancer_update_breaker.apply(balancer.balancer_id)
                click.echo('{} - update itype: {}, ctype: {}, prj: {}'.format(
                    balancer.balancer_id, nanny_instance_tags.itype, nanny_instance_tags.ctype,
                    nanny_instance_tags.prj))
            else:
                click.echo('{} - skip'.format(balancer.balancer_id))
                continue

        balancer_update_breaker.wait()


@cli.command('find-020-ready', help='Find namespaces with alerting ready for 0.2.0')
@context.pass_cli_context
def find_020_ready(cli_context):
    for namespace_pb in _iter_namespaces(cli_context):
        if not namespace_pb.spec.alerting.version or namespace_pb.spec.alerting.version == '0.2.0':
            continue

        balancers = cli_context.list_balancers_pbs(namespace_pb.meta.id)
        upstreams = cli_context.list_upstreams_pbs(namespace_pb.meta.id)
        has_health_upstream = any(
            upstream.meta.id == 'awacs-balancer-health-check'
            and (
                (
                    upstream.spec.yandex_balancer.config.HasField('regexp_section')
                    and upstream.spec.yandex_balancer.config.regexp_section.matcher.HasField('match_fsm')
                )
                or upstream.spec.yandex_balancer.config.HasField('regexp_path_section')
            )
            for upstream in upstreams
        )
        if all(
            balancer_pb.spec.WhichOneof('spec') == 'yandex_balancer'
            and (
                balancer_pb.spec.yandex_balancer.config.WhichOneof('module') == 'l7_macro'
                and (
                    balancer_pb.spec.yandex_balancer.config.l7_macro.HasField('health_check_reply')
                    or has_health_upstream
                )
                and (
                    balancer_pb.spec.yandex_balancer.config.l7_macro.HasField('http')
                    or balancer_pb.spec.yandex_balancer.config.l7_macro.HasField('https')
                )

                # and (not balancer_pb.spec.yandex_balancer.config.l7_macro.http.ports or 80 in balancer_pb.spec.yandex_balancer.config.l7_macro.http.ports)
            ) or (
                balancer_pb.spec.yandex_balancer.config.WhichOneof('module') == 'instance_macro'
                and any(
                section.key in ('http_section', 'https_section')
                for section in balancer_pb.spec.yandex_balancer.config.instance_macro.sections
            )
                and has_health_upstream
            )
            for balancer_pb in balancers
        ):
            click.echo(namespace_pb.meta.id)


@cli.command('remove-nanny-check', help='Remove nanny check from balancers')
@click.option('--limit', type=click.INT, default=1)
@click.option('--exact-namespaces', default='')
@click.option('--forbidden-balancers',
              default='rtc_balancer_frontend_vh_yandex_ru_yp_man'
                      ' rtc_balancer_frontend_vh_yandex_ru_yp_sas'
                      ' rtc_balancer_frontend_vh_yandex_ru_yp_vla'
                      ' rtc_balancer_internal_vh_man'
                      ' rtc_balancer_internal_vh_vla'
                      ' production_balancer_wizard'
                      ' rtc_balancer_news_production_knoss_load_man'
                      ' rtc_balancer_news_production_knoss_load_vla'
                      ' rtc_balancer_news_production_knoss_load_sas'
                      ' rtc_balancer_news_production_knoss_man'
                      ' rtc_balancer_news_production_knoss_vla'
                      ' rtc_balancer_news_production_knoss_sas'
                      ' production_balancer_entitysearch_test_iss'
                      ' production_balancer_mobile_ads_beta_msk'
                      ' production_balancer_services_msk'
                      ' production_balancer_answers_test'
              )
@click.option('--no-act', is_flag=True)
@click.option('--start-from', default='')
@click.argument('check_names', nargs=-1)
@context.pass_cli_context
def remove_nanny_checks(cli_context, limit, exact_namespaces, forbidden_balancers, no_act, start_from, check_names):
    total = 0
    forbidden_balancers = [f.strip() for f in forbidden_balancers.split(' ') if f.strip()]
    exact_namespaces = [f.strip() for f in exact_namespaces.split(' ') if f.strip()]

    if exact_namespaces:
        req_pb = api_pb2.ListNamespacesRequest()
        req_pb.query.id_in.extend(exact_namespaces)
        resp_pb = cli_context.namespace_stub.list_namespaces(req_pb)
        namespaces = resp_pb.namespaces
    else:
        namespaces = list(_iter_namespaces(cli_context))

    for i, ns_pb in enumerate(namespaces):
        if exact_namespaces and ns_pb.meta.id not in exact_namespaces:
            continue

        if start_from and ns_pb.meta.id < start_from:
            continue

        namespace = models.Namespace(cli_context, ns_pb)

        for balancer in namespace.list_balancers():
            if not limit:
                return

            if balancer.nanny_service_id in forbidden_balancers:
                continue

            try:
                removed = balancer.remove_nanny_checks(check_names, no_act)
            except Exception as e:
                click.echo("\rFailed: %s -> %s -> %s" % (balancer.namespace_id, balancer.balancer_id, e))

            if removed:
                total += 1
                limit -= 1

                click.echo('\r[%5d] %s -> %s -> %s' % (
                    total,
                    balancer.namespace_id,
                    balancer.balancer_id,
                    'https://nanny.yandex-team.ru/ui/#/services/catalog/%s/monitoring_settings' % (
                        balancer.nanny_service_id,),
                ))
            else:
                click.echo("\rSkip: %s -> %s" % (balancer.namespace_id, balancer.balancer_id), nl=False)
