from collections import defaultdict

import requests
import urllib3
import vcr


urllib3.disable_warnings()

import click

from infra.swatlib.auth import staff
from infra.awacs.proto import api_pb2, api_stub, model_pb2
from infra.awacs.tools.awacsalerting.src.models import Namespace


DEFAULT_AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'

DEFAULT_WARN_BALANCER_NOTIFY_RULE_KWARGS_TPL = """status:
    - from: OK
      to: WARN
login:
{}
method:
    - email
"""

DEFAULT_CRIT_BALANCER_NOTIFY_RULE_TPL = """status:
    - from: WARN
      to: CRIT
    - from: OK
      to: CRIT
login:
{}
method:
    - telegram
    - email
"""


class AbcClient:
    DEFAULT_ABC_API_URL = 'https://abc-back.yandex-team.ru/api/v4/'

    def __init__(self, vcr, abc_token):
        self.vcr = vcr
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(abc_token)

    def get_abc_service(self, abc_service_id):
        with self.vcr.use_cassette('abc_service_{}.yml'.format(abc_service_id)):
            url = self.DEFAULT_ABC_API_URL + 'services/{}/'.format(abc_service_id)
            resp = self.session.get(url)
            resp.raise_for_status()
            return resp.json()

    def get_abc_service_members(self, abc_service_id):
        """

        :type abc_service_id: int
        :rtype: list[dict]
        """
        with self.vcr.use_cassette('abc_service_{}_members.yml'.format(abc_service_id)):
            url = self.DEFAULT_ABC_API_URL + 'services/members/?service={}'.format(abc_service_id)
            resp = self.session.get(url)
            resp.raise_for_status()
            return resp.json().get('results', [])


class NannyClient(object):
    NANNY_DEFAULT_URL = 'https://nanny.yandex-team.ru/'
    DEFAULT_COMMENT = 'Updated by nctl'

    def __init__(self, token, url=None):
        url = url or self.NANNY_DEFAULT_URL
        self.url = url.rstrip('/') + '/'
        self.token = token
        self.session = requests.Session()
        self.session.headers['Content-Type'] = 'application/json'
        self.session.headers['Authorization'] = 'OAuth {}'.format(token)

    def _get_segment_attrs(self, service_id, segment):
        url = self.url + 'v2/services/' + service_id + '/{}_attrs/'.format(segment)
        r = self.session.get(url)
        r.raise_for_status()
        content = r.json()
        return {
            'content': content['content'],
            'snapshot_id': content['_id'],
            'comment': self.DEFAULT_COMMENT
        }

    def get_active_revision_data(self, service_id):
        """

        :rtype: dict
        """
        url = self.url + 'v2/services/_helpers/get_active_revision_id/{service_id}/'.format(service_id=service_id)
        r = self.session.get(url)
        r.raise_for_status()
        return r.json()

    def get_service_info_attrs(self, service_id):
        return self._get_segment_attrs(service_id, 'info')

    def get_service_runtime_attrs(self, service_id):
        return self._get_segment_attrs(service_id, 'runtime')

    def update_service_info_attrs(self, service_id, info_attrs, comment):
        """
        :type service_id: str
        :type info_attrs: models.NannyInfoAttrs
        :type comment: str
        """
        url = self.url + 'v2/services/{service_id}/info_attrs/'.format(service_id=service_id)
        r = self.session.put(url, json={
            'comment': comment,
            'snapshot_id': info_attrs.snapshot_id,
            'content': info_attrs.content,
        })
        r.raise_for_status()


class StaffClient(object):
    def __init__(self, token):
        self._staff_client = staff.StaffClient.from_config({
            'api_url': 'https://staff-api.yandex-team.ru/v3/',
            'oauth_token': token,
            'req_timeout': 5,
            'verify_ssl': False,
        })

    def get_groups_members_by_ids(self, group_ids):
        """
        :type group_ids: list[int]
        :rtype: list[str]
        """
        if not group_ids:
            return []

        spec = {
            "group.id": ",".join(str(group_id) for group_id in group_ids),
            "person.official.is_robot": "false",
        }
        data = self._staff_client.list_groupmembership(spec, fields=('person.login',), limit=200)
        return [item['person']['login'] for item in data['result']]

    def get_groups_members_by_slugs(self, group_slugs):
        """
        :type group_ids: list[str]
        :rtype: list[str]
        """
        if not group_slugs:
            return []

        spec = {
            "group.url": ",".join(group_slug for group_slug in group_slugs),
            "person.official.is_robot": "false",
        }
        data = self._staff_client.list_groupmembership(spec, fields=('person.login',), limit=200)
        return [item['person']['login'] for item in data['result']]

    def get_groups_by_ids(self, group_ids, fields=('id', 'url', 'type')):
        """

        :type group_ids: list[int]
        :type fields: list[str] | tuple[str]
        :rtype: dict[int, dict]
        """
        spec = {
            'id': ",".join([str(g_id) for g_id in group_ids]),
            '_limit': len(group_ids)
        }
        data = self._staff_client.list_groups(spec, fields=fields)
        return {int(v['id']): v for v in data.get('result', [])}


class CliContext(object):
    namespace_stub = None  # type: api_stub.NamespaceServiceStub
    balancer_stub = None  # type: api_stub.BalancerServiceStub
    upstream_stub = None  # type: api_stub.UpstreamServiceStub
    abc_client = None  # type: AbcClient
    nanny_client = None  # type: NannyClient
    staff_client = None  # type: StaffClient
    vcr = None  # type: vcr.VCR
    juggler_api_token = None  # type: six.text_type

    LOCATION_YP_CLUSTER = model_pb2.BalancerMeta.Location.YP_CLUSTER
    LOCATION_GENCFG_DC = model_pb2.BalancerMeta.Location.GENCFG_DC
    LOCATION_TYPES = (LOCATION_YP_CLUSTER, LOCATION_GENCFG_DC)

    _cache = None

    def init(self, awacs_rpc, nanny_token, abc_token=None, staff_token=None, vcr_cassette_library_dir=None):
        self.balancer_stub = api_stub.BalancerServiceStub(awacs_rpc)
        self.upstream_stub = api_stub.UpstreamServiceStub(awacs_rpc)
        self.namespace_stub = api_stub.NamespaceServiceStub(awacs_rpc)
        self.nanny_client = NannyClient(nanny_token)

        self.staff_client = StaffClient(staff_token) if staff_token else None
        self.vcr = vcr.VCR(
            cassette_library_dir=vcr_cassette_library_dir,
            record_mode='new_episodes',
            filter_headers=['authorization'],
            match_on=['method', 'scheme', 'host', 'port', 'path', 'query', 'raw_body'],
        ) if vcr_cassette_library_dir else None
        self.abc_client = AbcClient(self.vcr, abc_token) if abc_token and vcr_cassette_library_dir else None
        self._cache = defaultdict(lambda: {})

    @property
    def cache(self):
        return self._cache

    def list_namespaces(self, namespace_id_in=None):
        """
        :type namespace_id_in: list[six.text_type] | None
        :rtype: list[Namespace]
        """
        req_pb = api_pb2.ListNamespacesRequest()
        if namespace_id_in is not None:
            req_pb.query.id_in.extend(namespace_id_in)
        resp_pb = self.namespace_stub.list_namespaces(req_pb)
        return [Namespace(self, ns_pb) for ns_pb in resp_pb.namespaces]

    def save_to_paste(self, content):
        pass

    def list_upstreams_pbs(self, namespace_id):
        resp = self.upstream_stub.list_upstreams(
            list_upstreams_request=api_pb2.ListUpstreamsRequest(
                namespace_id=namespace_id
            ))
        return resp.upstreams

    def list_balancers_pbs(self, namespace_id):
        resp = self.balancer_stub.list_balancers(
            list_balancers_request=api_pb2.ListBalancersRequest(
                namespace_id=namespace_id
            ))
        return resp.balancers

    def get_balancer_aspects_tags_pb(self, namespace_id, balancer_id):
        get_balancer_aspects_set_request = api_pb2.GetBalancerAspectsSetRequest()
        get_balancer_aspects_set_request.id = balancer_id
        get_balancer_aspects_set_request.namespace_id = namespace_id
        aspect_resp = self.balancer_stub.get_balancer_aspects_set(get_balancer_aspects_set_request)
        return aspect_resp.aspects_set.content.cluster.content.tags

    def get_balancer_aspects(self, namespace_id, balancer_id):
        get_balancer_aspects_set_request = api_pb2.GetBalancerAspectsSetRequest()
        get_balancer_aspects_set_request.id = balancer_id
        get_balancer_aspects_set_request.namespace_id = namespace_id
        aspect_resp = self.balancer_stub.get_balancer_aspects_set(get_balancer_aspects_set_request)
        return aspect_resp.aspects_set

    def get_abc_service_members_count_by_roles(self, abc_service_id):
        """

        :type abc_service_id: int
        :type: dict[six.text_type, int]
        """
        service_members_resp = self.abc_client.get_abc_service_members(abc_service_id)
        res = defaultdict(lambda: 0)
        for member in service_members_resp:
            res[member['role']['scope']['slug']] += 1
        return res

    def is_ready_namespace_for_alerting(self, namespace_pb, balancer_pbs=None):
        if namespace_pb.spec.HasField('alerting'):
            return False, 'Alerting already enabled'

        if namespace_pb.spec.incomplete:
            return False, 'Namespace Order is Incomplete'

        _balancers_pbs = balancer_pbs or self.list_balancers_pbs(namespace_pb.meta.id)

        if not _balancers_pbs:
            return False, 'balancers not found'

        needs_instance_tags = [b_pb for b_pb in _balancers_pbs
                               if not b_pb.spec.config_transport.nanny_static_file.HasField('instance_tags')]

        if needs_instance_tags:
            return False, 'Balancers without instance tags'

        return True, None

    def get_default_alerting_recipients(self, abc_service_id):
        abc_service_slug = self.abc_client.get_abc_service(abc_service_id)['slug'].lower()
        res = self.get_abc_service_members_count_by_roles(abc_service_id)
        administrations_count, developments_count, devops_count = res['administration'], res['development'], res[
            'devops']
        smanagement_count = res['services_management']
        recipients = []
        total_recipients = 0
        if 0 < administrations_count <= 15:
            if administrations_count == 1:
                if 0 < devops_count < 15:
                    recipients = ['{}_administration'.format(abc_service_slug),
                                  '{}_devops'.format(abc_service_slug)]
                    total_recipients = administrations_count + devops_count
                elif 0 < developments_count < 15:
                    recipients = ['{}_administration'.format(abc_service_slug),
                                  '{}_development'.format(abc_service_slug)]
                    total_recipients = administrations_count + developments_count
                else:
                    recipients = ['{}_administration'.format(abc_service_slug)]
                    total_recipients = administrations_count
            else:
                recipients = ['{}_administration'.format(abc_service_slug)]
                total_recipients = administrations_count
        elif 0 < developments_count <= 15:
            if developments_count == 1:
                if 0 < devops_count < 15:
                    recipients = ['{}_administration'.format(abc_service_slug),
                                  '{}_devops'.format(abc_service_slug)]
                    total_recipients = administrations_count + devops_count
                elif 0 < smanagement_count:
                    recipients = ['{}_services_management'.format(abc_service_slug),
                                  '{}_development'.format(abc_service_slug)]
                    total_recipients = smanagement_count + developments_count
                else:
                    recipients = ['{}_development'.format(abc_service_slug)]
                    total_recipients = developments_count
            else:
                recipients = ['{}_development'.format(abc_service_slug)]
                total_recipients = developments_count

        if total_recipients < 2 and 0 < smanagement_count <= 15:
            recipients.append('{}_services_management'.format(abc_service_slug))
            total_recipients += smanagement_count

        return abc_service_slug, recipients, total_recipients

    def enable_namespace_alerting(self, ns_pb, recipients, alerting_version='0.0.1', annotations=None):
        """

        :type ns_pb: model_pb2.Namespace
        :type recipients: list[six.text_type]
        :type alerting_version: six.text_type
        :type annotations: dict[six.text_type, six.text_type]
        :return:
        """
        update_ns_request_pb = api_pb2.UpdateNamespaceRequest()
        update_ns_request_pb.meta.CopyFrom(ns_pb.meta)
        update_ns_request_pb.spec.CopyFrom(ns_pb.spec)
        update_ns_request_pb.spec.alerting.version = alerting_version
        recipients_text = "\n".join(["    - '@svc_{}'".format(r) for r in recipients])

        for k, v in (annotations or {}).items():
            update_ns_request_pb.meta.annotations[k] = v

        project = ns_pb.meta.annotations.get('project')

        crit_juggler_nr_pb = update_ns_request_pb.spec.alerting.juggler_raw_notify_rules.balancer.add()
        crit_juggler_nr_pb.template_name = 'on_status_change'
        crit_juggler_nr_pb.template_kwargs = DEFAULT_CRIT_BALANCER_NOTIFY_RULE_TPL.format(recipients_text)

        warn_juggler_nr_pb = update_ns_request_pb.spec.alerting.juggler_raw_notify_rules.balancer.add()
        warn_juggler_nr_pb.template_name = 'on_status_change'
        warn_juggler_nr_pb.template_kwargs = DEFAULT_WARN_BALANCER_NOTIFY_RULE_KWARGS_TPL.format(recipients_text)

        if project == 'taxi':
            update_ns_request_pb.spec.alerting.notify_rules_disabled = True
            update_ns_request_pb.spec.preset = model_pb2.NamespaceSpec.PR_WITHOUT_NOTIFICATIONS
        elif project == 'maps':
            update_ns_request_pb.spec.alerting.balancer_checks_disabled = True
            update_ns_request_pb.spec.preset = model_pb2.NamespaceSpec.PR_PLATFORM_CHECKS_ONLY

        self.namespace_stub.update_namespace(update_ns_request_pb)

    def get_balancer_tags(self, balancer_pb, gencfg_tags):
        nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id

        return gencfg_tags.get(nanny_service_id)

    def balancer_spec_is_active(self, b_pb):
        return (b_pb.status.validated.status == 'True' and
                b_pb.status.in_progress.status == 'False' and
                b_pb.status.active.status == 'True')

    def check_balancer_spec_is_active(self, namespace_id, balancer_id):
        resp = self.balancer_stub.get_balancer(api_pb2.GetBalancerRequest(
            namespace_id=namespace_id,
            id=balancer_id
        ))
        return self.balancer_spec_is_active(resp.balancer)

    def disable_nanny_deploy_monitoring(self, namespace_pb):
        balancer_pbs = self.list_balancers_pbs(namespace_pb.meta.id)


pass_cli_context = click.make_pass_decorator(CliContext, ensure=True)
