# encoding: utf-8
import time

from infra.awacs.proto import api_pb2, model_pb2


class NannyInstanceTags(object):
    def __init__(self, itype, ctype, prj):
        """

        :type itype: list[six.text_type]
        :type ctype: list[six.text_type]
        :type prj: list[six.text_type]
        """
        self._itype = itype
        self._ctype = ctype
        self._prj = prj

    @property
    def itype(self):
        return self._itype

    @property
    def ctype(self):
        return self._ctype

    @property
    def prj(self):
        return self._prj

    def __str__(self):
        return "{}, {}, {}".format("|".join(self.itype), "|".join(self.ctype), "|".join(self.prj))

    def __repr__(self):
        return "InstanceTags({})".format(self)

    @property
    def is_full(self):
        return self.itype and self.ctype and self.prj


class NannyActiveRevisionData(object):
    def __init__(self, locations, instance_tags):
        """

        :type locations: list[six.text_type]
        :type instance_tags: NannyInstanceTags
        """
        self._locations = locations
        self._instance_tags = instance_tags

    @property
    def locations(self):
        return self._locations

    @property
    def instance_tags(self):
        return self._instance_tags


class NannyRuntimeAttrs(object):
    def __init__(self, snapshot_id, content):
        self._snapshot_id = snapshot_id
        self._content = content

    @property
    def instances_chosen_type(self):
        return self._content['instances']['chosen_type']

    @property
    def gencfg_fixed_orthogonal_tags(self):
        """

        :rtype: dict
        """

        if self.instances_chosen_type == 'EXTENDED_GENCFG_GROUPS':
            instances_data = self._content.get('instances')
            return instances_data['extended_gencfg_groups'].get('orthogonal_tags')


class NannyDeployMonitoringRecipients(object):
    def __init__(self, logins, group_ids):
        self._logins = logins
        self._group_ids = group_ids


class NannyInfoAttrs(object):
    def __init__(self, ctx, snapshot_id, content):
        self._ctx = ctx
        self._snapshot_id = snapshot_id
        self._content = content

    @property
    def deploy_monitoring(self):
        return self._content['monitoring_settings']['deploy_monitoring']

    @property
    def active_checks(self):
        return self._content['monitoring_settings']['juggler_settings']['content']['active_checks']

    @property
    def snapshot_id(self):
        return self._snapshot_id

    @property
    def content(self):
        return self._content


class AbcService(object):
    ABC_SERVICE_INFO_BY_ID_CACHE_KEY = 'abc_service_info_by_id'

    def __init__(self, ctx, service_id):
        """

        :type ctx: CliContext
        :type service_id: int
        """
        self._ctx = ctx
        self._service_id = int(service_id)
        if self._service_id not in self._ctx.cache[self.ABC_SERVICE_INFO_BY_ID_CACHE_KEY]:
            self._ctx.cache[self.ABC_SERVICE_INFO_BY_ID_CACHE_KEY][
                self._service_id] = self._ctx.abc_client.get_abc_service(self._service_id)

        self._service_info = self._ctx.cache[self.ABC_SERVICE_INFO_BY_ID_CACHE_KEY][self._service_id]

    @property
    def slug(self):
        return self._service_info['slug']

    @property
    def service_id(self):
        return self._service_id

    @property
    def ui_url(self):
        return 'https://abc.yandex-team.ru/services/{}'.format(self.slug)

    def __hash__(self):
        return self._service_id

    def __eq__(self, other):
        if isinstance(other, AbcService):
            return self.service_id == other.service_id
        return False

    def __str__(self):
        return self.slug

    def __repr__(self):
        return "AbcService(slug={})".format(self.slug)


class StaffGroup(object):
    GROUP_INFO_CACHE_KEY = 'staff_group_info_by_id'

    def __init__(self, ctx, group_id):
        """

        :type ctx: CliContext
        :type group_id: six.text_type
        """
        self._ctx = ctx
        self._group_id = int(group_id)

        # if self._group_id not in self._ctx.cache[self.GROUP_INFO_CACHE_KEY]:
        #     self._ctx.cache[self.GROUP_INFO_CACHE_KEY][
        #         self._group_id] = self._ctx.staff_client.get_groups_by_ids([self._group_id]).get(self._group_id, None)
        #
        # self._service_info = self._ctx.cache[self.GROUP_INFO_CACHE_KEY][self._group_id]

    @property
    def group_id(self):
        return self._group_id


class Namespace(object):
    def __init__(self, ctx, namespace_pb):
        """

        :type ctx: infra.awacs.tools.awacsalerting.src.context.CliContext
        :type namespace_pb:
        """

        self._ctx = ctx
        self._pb = namespace_pb

    @property
    def is_alerting_enabled(self):
        return (
            self._pb.HasField('spec') and
            self._pb.spec.HasField('alerting')
        )

    @property
    def namespace_id(self):
        return self._pb.meta.id

    @property
    def ui_url(self):
        return 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/'.format(self.namespace_id)

    @property
    def abc_service(self):
        return AbcService(self._ctx, self._pb.meta.abc_service_id)

    @property
    def category(self):
        return self._pb.meta.category.split('/')

    @property
    def owners_logins(self):
        return list(self._pb.meta.auth.staff.owners.logins)

    @property
    def owners_groups(self):
        return [StaffGroup(self._ctx, group_id) for group_id in self._pb.meta.auth.staff.owners.group_ids]

    def get_abc_service(self, abc_client, service_info_cache):
        """

        :type abc_client: AbcClient
        :rtype:
        """
        if self._pb.meta.abc_service_id not in service_info_cache:
            service_info_cache[self._pb.meta.abc_service_id] = abc_client.get_abc_service(
                self._pb.meta.abc_service_id)
        return service_info_cache[self._pb.meta.abc_service_id]

    def list_balancer_pbs(self):
        """

        :rtype: list[model_pb2.Balancer]
        """
        resp = self._ctx.balancer_stub.list_balancers(
            list_balancers_request=api_pb2.ListBalancersRequest(
                namespace_id=self._pb.meta.id
            ))
        return resp.balancers

    def list_balancers(self):
        """

        :rtype: list[Balancer]
        """

        return [Balancer(self._ctx, b_pb) for b_pb in self.list_balancer_pbs()]

    def __str__(self):
        return self.namespace_id

    def __repr__(self):
        return "Namespace(id={})".format(self.namespace_id)


class Balancer(object):
    PLATFORM_UNDEFINED = 'UNDEFINED'
    PLATFORM_GENCFG = model_pb2.ClusterAspectsContent.Platform.Name(model_pb2.ClusterAspectsContent.GENCFG)
    PLATFORM_YP_LITE = model_pb2.ClusterAspectsContent.Platform.Name(model_pb2.ClusterAspectsContent.YP_LITE)
    LOCATION_TYPE_TO_PLATFORM = {
        model_pb2.BalancerMeta.Location.GENCFG_DC: PLATFORM_GENCFG,
        model_pb2.BalancerMeta.Location.YP_CLUSTER: PLATFORM_YP_LITE,
    }

    def __init__(self, ctx, balancer_pb):
        """

        :type ctx: infra.awacs.tools.awacsalerting.src.context.CliContext
        :type balancer_pb: infra.awacs.proto.model_pb2.Balancer
        """
        self._ctx = ctx
        self._pb = balancer_pb
        self._nanny_service_active_revision_data = None
        self._nanny_service_runtime_attrs = None
        self._nanny_service_info_attrs = None
        self._platform = None

    def __repr__(self):
        return "Balancer(id={})".format(self.balancer_id)

    @property
    def pb(self):
        return self._pb

    @property
    def namespace_id(self):
        return self._pb.meta.namespace_id

    @property
    def balancer_id(self):
        return self._pb.meta.id

    @property
    def nanny_service_id(self):
        return self._pb.spec.config_transport.nanny_static_file.service_id

    @property
    def ui_url(self):
        return "https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/balancers/list/{}/show/".format(
            self.namespace_id, self.balancer_id)

    @property
    def location_type(self):
        return self._pb.meta.location.type

    @property
    def location_type_str(self):
        return model_pb2.BalancerMeta.Location.Type.Name(self.location_type)

    @property
    def location(self):
        if self.location_type == model_pb2.BalancerMeta.Location.YP_CLUSTER:
            return self._pb.meta.location.yp_cluster
        elif self.location_type == model_pb2.BalancerMeta.Location.GENCFG_DC:
            return self._pb.meta.location.gencfg_dc
        else:
            return None

    @property
    def spec_is_active(self):
        return (self._pb.status.validated.status == 'True' and
                self._pb.status.in_progress.status == 'False' and
                self._pb.status.active.status == 'True')

    @property
    def instance_tags(self):
        """

        :rtype: infra.awacs.proto.model_pb2.InstanceTags | None
        """
        if self._pb.spec.config_transport.nanny_static_file.HasField('instance_tags'):
            return self._pb.spec.config_transport.nanny_static_file.instance_tags

    @instance_tags.setter
    def instance_tags(self, value):
        assert isinstance(value, model_pb2.InstanceTags)
        self._pb.spec.config_transport.nanny_static_file.instance_tags.CopyFrom(value)

    @property
    def instance_tags_str(self):
        if self.instance_tags is not None:
            return "{}, {}, {}".format(self.instance_tags.itype, self.instance_tags.ctype, self.instance_tags.prj)
        return 'None'

    def get_nanny_active_revision_data(self):
        """

        :rtype: NannyActiveRevisionData
        """
        active_nanny_service_id = self.nanny_service_id

        if not active_nanny_service_id:
            raise ValueError('balancer with id {} does not have valid nanny service id'.format(self.balancer_id))
        resp_data = self._ctx.nanny_client.get_active_revision_data(active_nanny_service_id)

        active_conf_id = resp_data.get('active_revision_id', '')
        if not active_conf_id:
            raise ValueError('Nanny service "{}" does not have an active configuration'.format(active_nanny_service_id))

        orthogonal_tags = resp_data.get('full_orthogonal_tags', None)
        if not orthogonal_tags:
            raise ValueError('full_orthogonal_tags not found in active revision')

        def get_tags(tags_data, key):
            tags_combinations = tags_data.get(key, [])
            tags = set()
            for tags_combination in tags_combinations:
                for tag in tags_combination:
                    if tag not in (None, 'none', 'unknown', ''):
                        tags.add(tag)
            return sorted(tags)

        instance_tags = NannyInstanceTags(
            itype=get_tags(orthogonal_tags, 'itype'),
            ctype=get_tags(orthogonal_tags, 'ctype'),
            prj=get_tags(orthogonal_tags, 'prj'),
        )
        return NannyActiveRevisionData(
            locations=resp_data.get('locations', []),
            instance_tags=instance_tags
        )

    @property
    def nanny_service_active_revision_data(self):
        if self._nanny_service_active_revision_data is None:
            self._nanny_service_active_revision_data = self.get_nanny_active_revision_data()
        return self._nanny_service_active_revision_data

    def get_service_runtime_attrs(self):
        """

        :rtype: NannyRuntimeAttrs
        """

        runtime_attrs = self._ctx.nanny_client.get_service_runtime_attrs(self.nanny_service_id)
        return NannyRuntimeAttrs(runtime_attrs['snapshot_id'], runtime_attrs['content'])

    @property
    def nanny_service_runtime_attrs(self):
        if self._nanny_service_runtime_attrs is None:
            self._nanny_service_runtime_attrs = self.get_service_runtime_attrs()
        return self._nanny_service_runtime_attrs

    def remove_nanny_checks(self, check_names, no_act=False):
        info_attrs = self.get_service_info_attrs()
        active_checks = info_attrs.active_checks
        checks_found = []

        if 'deploy_status' in check_names:
            deploy_monitoring = info_attrs.deploy_monitoring
            if deploy_monitoring['is_enabled']:
                checks_found.append('deploy_status')
                deploy_monitoring['is_enabled'] = False
                deploy_monitoring.pop('content', None)

        for active_check in active_checks:
            passive_checks = active_check['passive_checks']
            for idx, check in reversed(list(enumerate(passive_checks))):
                if check['juggler_service_name'] in check_names:
                    passive_checks.pop(idx)
                    checks_found.append(check['juggler_service_name'])

        if checks_found and not no_act:
            self._ctx.nanny_client.update_service_info_attrs(
                self.nanny_service_id,
                info_attrs,
                'Remove check(s): {}'.format(', '.join(checks_found))
            )
        return checks_found

    def get_service_info_attrs(self):
        """

        :rtype: NannyInfoAttrs
        """
        assert self.nanny_service_id
        info_attrs = self._ctx.nanny_client.get_service_info_attrs(self.nanny_service_id)
        return NannyInfoAttrs(self._ctx, info_attrs['snapshot_id'], info_attrs['content'])

    @property
    def nanny_service_info_attrs(self):
        if self._nanny_service_info_attrs is None:
            self._nanny_service_info_attrs = self.get_service_info_attrs()
        return self._nanny_service_info_attrs

    @property
    def platform(self):
        """

        :rtype: six.text_type
        """
        if self._platform is None:
            self._platform = self.determine_platform()
        return self._platform

    def determine_platform(self):
        """

        :rtype: six.text_type
        """
        platform = self.PLATFORM_UNDEFINED
        if self._pb.meta.HasField('location'):
            platform = self.LOCATION_TYPE_TO_PLATFORM.get(self._pb.meta.location.type, self.PLATFORM_UNDEFINED)

        if platform != self.PLATFORM_UNDEFINED:
            return platform

        get_balancer_aspect_set_request = api_pb2.GetBalancerAspectsSetRequest()
        get_balancer_aspect_set_request.id = self._pb.meta.id
        get_balancer_aspect_set_request.namespace_id = self._pb.meta.namespace_id
        resp = self._ctx.balancer_stub.get_balancer_aspects_set(get_balancer_aspect_set_request)
        return model_pb2.ClusterAspectsContent.Platform.Name(resp.aspects_set.content.cluster.content.platform)

    def get_nanny_service_instance_tags(self):
        """

        :rtype: model_pb2.InstanceTags
        :raises: GetNannyServiceInstanceTagsError, ValueError
        """
        balancer_platform = self.determine_platform()

        if balancer_platform == self.PLATFORM_YP_LITE:
            try:
                active_revision_data = self.get_nanny_active_revision_data()
            except Exception as e:
                raise ValueError('Get nanny active revision data error: {}'.format(e))

            if not active_revision_data.instance_tags.is_full:
                raise ValueError('nanny instance tags is not full {}'.format(active_revision_data.instance_tags))

            nanny_instance_tags = model_pb2.InstanceTags()
            nanny_instance_tags.ctype = active_revision_data.instance_tags.ctype[0]
            nanny_instance_tags.itype = active_revision_data.instance_tags.itype[0]
            nanny_instance_tags.prj = active_revision_data.instance_tags.prj[0]
        elif balancer_platform == self.PLATFORM_GENCFG:
            try:
                runtime_attrs = self.get_service_runtime_attrs()
            except Exception as e:
                raise ValueError('Get nanny active revision data error: {}'.format(e))

            gencfg_fixed_orthogonal_tags = runtime_attrs.gencfg_fixed_orthogonal_tags
            if gencfg_fixed_orthogonal_tags is None:
                raise ValueError('nanny gencfg_fixed_orthogonal_tags does not set')
            nanny_instance_tags = model_pb2.InstanceTags()
            nanny_instance_tags.ctype = gencfg_fixed_orthogonal_tags['ctype']
            nanny_instance_tags.itype = gencfg_fixed_orthogonal_tags['itype']
            nanny_instance_tags.prj = gencfg_fixed_orthogonal_tags['prj']
        else:
            raise ValueError('balancer platform "{}" unimplemented'.format(balancer_platform))

        return nanny_instance_tags


class BalancersSpecUpdateBreaker(object):

    def __init__(self, max_in_flight, in_flight_period):
        self._max_in_flight = max_in_flight
        self._in_flight_period = in_flight_period
        self._statuses = {}

    def apply(self, balancer_id):
        self._statuses[balancer_id] = time.time()

    def wait(self):
        skip_before = time.time() - self._in_flight_period

        self._statuses = {b_id: started_time for b_id, started_time in self._statuses.items()
                          if started_time > skip_before}

        if len(self._statuses) > self._max_in_flight:
            time.sleep(5)
            return self.wait()
