from __future__ import print_function

import base64
import collections
import io
import logging
import random
import re
import tarfile
import uuid
from datetime import datetime, timedelta

import click
import requests
import six
from infra.awacs.proto import (model_pb2 as awacs_model_pb2,
                               api_pb2 as awacs_api_pb2,
                               api_stub as awacs_api_stub)
from instancectl.clients.vault import client as nannyvault
from library.python import resource
from nanny_repo import repo_api_stub, repo_api_pb2
from nanny_rpc_client import RequestsRpcClient, exceptions
from six.moves import input
from sepelib.core import config

from awacs.lib import certs
from awacs.lib.certs import get_end_entity_cert, fill_cert_fields, ensure_single_file_match
from awacs.lib.ya_vault import YaVaultClient


config.load()

DEFAULT_VAULT_URL = 'https://nanny-vault.yandex-team.ru'
DEFAULT_NANNY_RPC_URL = 'https://nanny.yandex-team.ru/api/repo/'
DEFAULT_AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'


@click.group()
@click.option('--awacs-rpc-url', envvar='AWACS_RPC_URL', default=DEFAULT_AWACS_RPC_URL)
@click.option('--awacs-token', envvar='AWACS_TOKEN', required=True)
@click.option('--nanny-rpc-url', envvar='NANNY_RPC_URL', default=DEFAULT_NANNY_RPC_URL)
@click.option('--nanny-token', envvar='NANNY_TOKEN', required=True)
@click.option('--vault-url', envvar='VAULT_RPC_URL', default=DEFAULT_VAULT_URL)
@click.pass_context
def cli(ctx, awacs_rpc_url, awacs_token, nanny_rpc_url, nanny_token, vault_url):
    ctx.obj = {
        'awacs_rpc_url': awacs_rpc_url,
        'awacs_token': awacs_token,
        'nanny_rpc_url': nanny_rpc_url,
        'nanny_token': nanny_token,
        'vault_url': vault_url
    }


def fill_cert_storage(storage_pb, secret_pb):
    storage_pb.type = storage_pb.NANNY_VAULT
    nanny_vault_secret_pb = storage_pb.nanny_vault_secret
    nanny_vault_secret_pb.keychain_id = secret_pb.keychain_id
    nanny_vault_secret_pb.secret_id = secret_pb.secret_id
    nanny_vault_secret_pb.secret_revision_id = secret_pb.secret_revision_id


def get_secret_pbs(nanny_rpc, service_id):
    rv = []
    nanny_repo_stub = repo_api_stub.RepoServiceStub(nanny_rpc)
    get_service_req_pb = repo_api_pb2.GetServiceRequest(service_id=service_id)
    resp_pb = nanny_repo_stub.get_service(get_service_req_pb)
    for volume_pb in resp_pb.service.spec.instance_spec.volume:
        if not volume_pb.name.startswith('secrets'):
            continue
        if volume_pb.HasField('secret_volume') and volume_pb.secret_volume.keychain_secret.keychain_id:
            rv.append((u'nanny_vault', volume_pb.secret_volume.keychain_secret))
        elif volume_pb.HasField('vault_secret_volume'):
            rv.append((u'yav', volume_pb.vault_secret_volume.vault_secret))
        else:
            raise RuntimeError(u'WTF is that?\n{}'.format(volume_pb))
    return rv


def read_secrets(nanny_rpc, vault_url, service_id, secret_ids_to_ignore=()):
    secret_pbs = get_secret_pbs(nanny_rpc, service_id)
    if not secret_pbs:
        raise RuntimeError('nanny service "{}" does not have secret volumes starting with "secrets"')
    token = resource.find('/secrets/nanny_vault_oauth_token_for_certs').strip()
    vault_client = nannyvault.VaultClient(vault_url, service_id=service_id, token=token)
    for secret_type, secret_pb in secret_pbs:
        if secret_pb.secret_id in secret_ids_to_ignore:
            continue
        if secret_type == u'yav':
            print(u'yav', secret_pb)
        elif secret_type == u'nanny_vault':
            secret_data = vault_client.get_secret(secret_pb)
            tarball_data = secret_data.get('secrets.tgz')
            if not tarball_data:
                raise RuntimeError(
                    'secret volume for "{}:{}" does not contain secrets.tgz'.format(secret_pb.keychain_id,
                                                                                    secret_pb.secret_id))
            with tarfile.open(fileobj=io.BytesIO(tarball_data), mode='r:gz') as t:
                yield secret_pb, t


def validate_secrets_tgz(t, cert_id):
    expected_filenames = {
        filename.format(cert_id) for filename in ('./allCAs-{}.pem',
                                                  './priv',
                                                  './priv/1st.{}.key',
                                                  './priv/2nd.{}.key',
                                                  './priv/3rd.{}.key',
                                                  './priv/{}.pem')
    }
    actual_filenames = set(t.getnames())
    if actual_filenames != expected_filenames:
        raise RuntimeError('\nactual filenames:\n{}\nexpected filenames:\n{}\ndiff:'.format(
            sorted(actual_filenames), sorted(expected_filenames)))


def get_supposed_cert_id(t):
    for filename in t.getnames():
        m = re.match(r'\./allCAs-(.+)\.pem', filename)
        if m is not None:
            return m.group(1)
    return None


def list_nanny_service_ids(awacs_rpc, namespace_id):
    awacs_balancer_service_stub = awacs_api_stub.BalancerServiceStub(awacs_rpc)
    list_balancers_req_pb = awacs_api_pb2.ListBalancersRequest(namespace_id=namespace_id)
    rv = []
    for balancer_pb in awacs_balancer_service_stub.list_balancers(list_balancers_req_pb).balancers:
        rv.append(balancer_pb.spec.config_transport.nanny_static_file.service_id)
    return rv


def create_cert(awacs_cert_service_stub, namespace_id, cert_id, spec_pb, meta_pb=None):
    create_cert_req_pb = awacs_api_pb2.CreateCertificateRequest(spec=spec_pb, meta=meta_pb)
    create_cert_req_pb.meta.auth.type = create_cert_req_pb.meta.auth.STAFF
    create_cert_req_pb.meta.namespace_id = namespace_id
    create_cert_req_pb.meta.id = cert_id
    print(create_cert_req_pb)
    awacs_cert_service_stub.create_certificate(create_cert_req_pb)


@cli.command('import')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--secret-ids-to-ignore', '-i', multiple=True)
@click.option('--just-print', '-p', is_flag=True)
def import_(cfg, namespace_id, secret_ids_to_ignore=(), just_print=False):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    nanny_rpc = RequestsRpcClient(rpc_url=cfg['nanny_rpc_url'], oauth_token=cfg['nanny_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)

    awacs_namespace_service_stub = awacs_api_stub.NamespaceServiceStub(awacs_rpc)
    get_namespace_req_pb = awacs_api_pb2.GetNamespaceRequest(id=namespace_id)
    resp_pb = awacs_namespace_service_stub.get_namespace(get_namespace_req_pb)
    namespace_pb = resp_pb.namespace

    service_ids = list_nanny_service_ids(awacs_rpc, namespace_id)
    if not service_ids:
        raise RuntimeError('no balancers found in {}'.format(namespace_id))

    prev_secret_pbs = None
    certs = {}
    vault_url = cfg['vault_url']
    for service_id in service_ids:
        print('Processing "{}" secrets'.format(service_id))
        secret_pbs = []
        for (secret_pb, t) in read_secrets(nanny_rpc, vault_url, service_id, secret_ids_to_ignore=secret_ids_to_ignore):
            if just_print:
                print(secret_pb)
                print('has following files')
                print(sorted(t.getnames()))
                cert_id = get_supposed_cert_id(t)
                print('get_supposed_cert_id', cert_id)
                print()
                print()
                print()
            else:
                cert_id = get_supposed_cert_id(t)
                if not cert_id:
                    raise RuntimeError('"secrets.tgz" (secret id "{}") does not look like '
                                       'a correctly packaged certificate'.format(secret_pb.secret_id))
                validate_secrets_tgz(t, cert_id)
                allcas_f = t.extractfile('./allCAs-{}.pem'.format(cert_id))
                allcas = get_end_entity_cert(allcas_f.read())
                certs[cert_id] = (allcas, secret_pb)
                secret_pbs.append(secret_pb)
        secret_pbs = sorted(secret_pbs, key=lambda i: i.secret_id)
        if prev_secret_pbs:
            if prev_secret_pbs != secret_pbs:
                raise RuntimeError('not all "secrets*" volumes are the same')
        prev_secret_pbs = secret_pbs

    if certs:
        for (cert_id, (allcas, secret_pb)) in sorted(certs.items()):
            spec_pb = awacs_model_pb2.CertificateSpec()
            fill_cert_fields(spec_pb.fields, allcas)
            fill_cert_storage(spec_pb.storage, secret_pb)
            spec_pb.source = spec_pb.IMPORTED
            spec_pb.imported.abc_service_id = namespace_pb.meta.abc_service_id
            assert spec_pb.imported.abc_service_id
            create_cert(awacs_cert_service_stub, namespace_id, cert_id, spec_pb)
            print(u'Imported certificate {}:{}'.format(namespace_id, cert_id))
    else:
        print(u'No certificates to import')


def extract_certs_from_yav_secret(cert_secret):
    """
    :type cert_secret: dict
    :rtype bytes, bytes
    :raises ValueError

    Look for files named "*_certificate" and "*_private_key".
    """
    public_key = private_key = None

    for cert_key, cert_value in six.iteritems(cert_secret):
        if cert_key.endswith('_certificate'):
            public_key = ensure_single_file_match('public key', cert_key, bool(public_key), cert_value)
        elif cert_key.endswith('_private_key'):
            private_key = ensure_single_file_match('private key', cert_key, bool(private_key), cert_value)

    if public_key and private_key:
        return public_key, private_key
    else:
        raise ValueError("secret doesn't contain both public and private cert parts")


@cli.command('import-external')
@click.pass_obj
@click.option('--certs-robot-ssh-key-path', envvar='ROBOT_AWACS_CERTS_SSH_KEY_PATH', required=True)
@click.option('--namespace-id', required=True)
@click.option('--cert-id', required=True)
@click.option('--yav-secret-id', required=True)
@click.option('--startrek-ticket', required=True)
def import_external(cfg, certs_robot_ssh_key_path, namespace_id, cert_id, yav_secret_id, startrek_ticket):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    yav = YaVaultClient(yav_url='https://vault-api.passport.yandex.net',
                        nanny_tvm_id='',
                        rsa_login='robot-awacs-certs',
                        rsa_key_path=certs_robot_ssh_key_path,
                        connection_timeout=10)
    _import_external(awacs_rpc, yav, namespace_id, cert_id, yav_secret_id, startrek_ticket)


def _import_external(awacs_rpc, yav, namespace_id, cert_id, yav_secret_id, startrek_ticket):
    """
    Robot's SSH key to authorize in YaVault:
    https://yav.yandex-team.ru/secret/sec-01dbytp5w5b1a1tj2w6m6f4db6/
    """
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)

    awacs_namespace_service_stub = awacs_api_stub.NamespaceServiceStub(awacs_rpc)
    get_namespace_req_pb = awacs_api_pb2.GetNamespaceRequest(id=namespace_id)
    resp_pb = awacs_namespace_service_stub.get_namespace(get_namespace_req_pb)
    namespace_pb = resp_pb.namespace
    assert namespace_pb.meta.abc_service_id

    cert_secret = yav.get_version(version=yav_secret_id)
    secret_ver, cert_secret = cert_secret['version'], cert_secret['value']
    public_key, private_key = extract_certs_from_yav_secret(cert_secret)
    assert public_key, private_key

    spec_pb = awacs_model_pb2.CertificateSpec()
    fill_cert_fields(spec_pb.fields, certs.get_end_entity_cert(public_key))
    spec_pb.source = spec_pb.IMPORTED
    spec_pb.imported.abc_service_id = namespace_pb.meta.abc_service_id
    spec_pb.storage.type = spec_pb.storage.YA_VAULT
    spec_pb.storage.ya_vault_secret.secret_id = yav_secret_id
    spec_pb.storage.ya_vault_secret.secret_ver = secret_ver

    meta_pb = awacs_model_pb2.CertificateMeta()
    meta_pb.unrevokable.value = True
    meta_pb.unrevokable.author = u'robot-awacs-certs'
    meta_pb.unrevokable.comment = u'Imported external certificate: {}'.format(startrek_ticket)

    create_cert(awacs_cert_service_stub, namespace_id, cert_id, spec_pb)
    print(u'Imported external certificate {}:{}, ticket {}'.format(namespace_id, cert_id, startrek_ticket))


def _create_secret_copy(yav_client, cert_pb):
    cert_secret = yav_client.get_version(version=cert_pb.spec.storage.ya_vault_secret.secret_id)
    cert_secret = cert_secret[u'value']
    cert_sn = cert_pb.spec.fields.serial_number
    public_key, private_key, cert_tarball = certs.extract_certs_from_yav_secret(
        log=logging.getLogger(u'copy_cert'),
        flat_cert_id='{}/{}'.format(cert_pb.meta.namespace_id, cert_pb.meta.id),
        serial_number=cert_sn,
        cert_secret=cert_secret)
    new_secret_name = u'certificate_{}_private_key_copy_{}'.format(cert_sn, uuid.uuid4())
    new_secret = [{
        u'key': u'{}_certificate'.format(cert_sn),
        u'value': public_key.decode('ascii'),
    }, {
        u'key': u'{}_private_key'.format(cert_sn),
        u'value': private_key.decode('ascii'),
    }]
    if cert_tarball:
        new_secret.append({
            u'key': u'secrets.tgz',
            u'value': base64.b64encode(cert_tarball).decode('ascii'),
            u'encoding': u'base64'
        })
    secret_uuid = yav_client.create_secret(new_secret_name)
    yav_client.create_secret_version(secret_uuid, new_secret)
    print(secret_uuid)
    return secret_uuid


@cli.command('copy')
@click.pass_obj
@click.option('--certs-robot-ssh-key-path', envvar='ROBOT_AWACS_CERTS_SSH_KEY_PATH', required=True)
@click.option('--namespace-id', required=True)
@click.option('--cert-id', required=True)
@click.option('--from-namespace-id', required=True)
@click.option('--from-cert-id', required=True)
@click.option('--startrek-ticket', required=True)
def copy_cert(cfg, certs_robot_ssh_key_path, namespace_id, cert_id, from_namespace_id, from_cert_id, startrek_ticket):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)
    from_cert = awacs_cert_service_stub.get_certificate(awacs_api_pb2.GetCertificateRequest(id=from_cert_id, namespace_id=from_namespace_id)).certificate
    yav = YaVaultClient(yav_url='https://vault-api.passport.yandex.net',
                        nanny_tvm_id='',
                        rsa_login='robot-awacs-certs',
                        rsa_key_path=certs_robot_ssh_key_path,
                        connection_timeout=10)
    new_secret_id = _create_secret_copy(yav, from_cert)
    _import_external(awacs_rpc, yav, namespace_id, cert_id, new_secret_id, startrek_ticket)
    print(u'Copied certificate from {}:{} to {}:{}, ticket {}'.format(from_namespace_id, from_cert_id, namespace_id, cert_id, startrek_ticket))


@cli.command('create-fake-empty-cert')
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--cert-id', required=True)
def create_fake_empty_cert(cfg, namespace_id, cert_id):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)
    spec_pb = awacs_model_pb2.CertificateSpec()
    storage_pb = spec_pb.storage
    storage_pb.type = storage_pb.NANNY_VAULT
    nanny_vault_secret_pb = storage_pb.nanny_vault_secret
    nanny_vault_secret_pb.keychain_id = 'a'
    nanny_vault_secret_pb.secret_id = 'b'
    nanny_vault_secret_pb.secret_revision_id = 'c'
    spec_pb.fields.SetInParent()
    create_cert(awacs_cert_service_stub, namespace_id, cert_id, spec_pb)
    print('Created fake empty certificate {}:{}'.format(namespace_id, cert_id))


def list_balancers(balancer_service_stub, namespace_id):
    req_pb = awacs_api_pb2.ListBalancersRequest(namespace_id=namespace_id)
    resp_pb = balancer_service_stub.list_balancers(req_pb)
    return resp_pb.balancers


def pause_balancer(balancer_service_stub, namespace_id, balancer_id):
    req_pb = awacs_api_pb2.UpdateBalancerTransportPausedRequest(namespace_id=namespace_id,
                                                                id=balancer_id)
    req_pb.transport_paused.value = True
    balancer_service_stub.update_balancer_transport_paused(req_pb)


def unpause_balancer(balancer_service_stub, namespace_id, balancer_id):
    req_pb = awacs_api_pb2.UpdateBalancerTransportPausedRequest(namespace_id=namespace_id,
                                                                id=balancer_id)
    req_pb.transport_paused.value = False
    balancer_service_stub.update_balancer_transport_paused(req_pb)


@cli.command('unpause-location')
@click.pass_obj
@click.option('--cert-full-ids-path', required=True, type=click.File('r'))
@click.option('--location-to-unpause', required=True)
def unpause_location(cfg, cert_full_ids_path, location_to_unpause):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    balancer_service_stub = awacs_api_stub.BalancerServiceStub(awacs_rpc)

    ns_ids = set()
    for line in cert_full_ids_path:
        line = line.strip()
        if not line:
            continue
        ns_id, cert_id = line.split()
        ns_ids.add(ns_id)

    click.echo('unpausing balancers in {}'.format(location_to_unpause))
    for ns_id in ns_ids:
        for balancer_pb in list_balancers(balancer_service_stub, ns_id):
            balancer_location = (
                balancer_pb.meta.location.yp_cluster or
                balancer_pb.meta.location.gencfg_dc or
                'XDC'
            )
            if balancer_location.lower() == location_to_unpause.lower():
                b_id = balancer_pb.meta.id
                unpause_balancer(balancer_service_stub, ns_id, b_id)
                print('unpaused balancer {}:{}'.format(ns_id, b_id))
    click.echo('unpaused balancers in {}'.format(location_to_unpause))


@cli.command('list-in-progress-statuses')
@click.pass_obj
@click.option('--cert-full-ids-path', required=True, type=click.File('r'))
@click.option('--location', required=True)
def list_in_progress_statuses(cfg, cert_full_ids_path, location):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    balancer_service_stub = awacs_api_stub.BalancerServiceStub(awacs_rpc)
    cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)

    cert_ids = []
    ns_ids = set()
    for line in cert_full_ids_path:
        line = line.strip()
        if not line:
            continue
        ns_id, cert_id = line.split()
        cert_ids.append((ns_id, cert_id))
        ns_ids.add(ns_id)

    ns_id_to_balancer_ids = collections.defaultdict(set)
    balancer_readiness = {}
    for ns_id in sorted(ns_ids):
        for balancer_pb in list_balancers(balancer_service_stub, ns_id):
            balancer_location = (
                balancer_pb.meta.location.yp_cluster or
                balancer_pb.meta.location.gencfg_dc or
                'XDC'
            )
            if balancer_location.lower() == location.lower():
                resp = requests.post(
                    'https://hq.{}-swat.yandex-team.ru/rpc/instances/GetRevisionStats/'.format(
                        'msk' if location.lower() in ('iva', 'myt') else location.lower()),
                    json={
                        'filter': {'serviceId': balancer_pb.spec.config_transport.nanny_static_file.service_id},
                    })
                data = resp.json()
                if data['stats']:
                    _, stat = max([(s['revision'], s) for s in data['stats']])
                    readiness = '{}/{}'.format(stat['readyCount'], stat['totalCount'])
                else:
                    readiness = 'X/X'
                balancer_readiness[(ns_id, balancer_pb.meta.id)] = readiness
                ns_id_to_balancer_ids[ns_id].add(balancer_pb.meta.id)

    for ns_id, cert_id in sorted(cert_ids):
        cert_pb = cert_service_stub.get_certificate(awacs_api_pb2.GetCertificateRequest(
            namespace_id=ns_id, id=cert_id)).certificate

        used = False
        for rev_pb in cert_pb.statuses:
            if rev_pb.validated:
                used = True
        if not used:
            print(ns_id, cert_id)
            print(' not used')
            continue
        rev_pb = cert_pb.statuses[-1]
        assert rev_pb.id == cert_pb.meta.version

        if not ns_id_to_balancer_ids[ns_id]:
            continue

        for b_id in ns_id_to_balancer_ids[ns_id]:
            print(ns_id, cert_id)
            in_progress_status = rev_pb.in_progress[ns_id + ':' + b_id].status
            active_status = rev_pb.active[ns_id + ':' + b_id].status

            if in_progress_status == 'True':
                fg = 'blue'
            elif active_status == 'True':
                fg = 'green'
            else:
                fg = 'red'
            click.echo(
                click.style(' * in progress ' + in_progress_status + ' at ' + b_id +
                            ' ' + balancer_readiness[(ns_id, b_id)], fg=fg))


@cli.command('batch-apply-renewals')
@click.pass_obj
@click.option('--cert-full-ids-path', required=True, type=click.File('r'))
def batch_apply_renewals(cfg, cert_full_ids_path):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)
    balancer_service_stub = awacs_api_stub.BalancerServiceStub(awacs_rpc)

    cert_ids_by_namespace_ids = collections.defaultdict(set)
    for line in cert_full_ids_path:
        line = line.strip()
        if not line:
            continue
        ns_id, cert_id = line.split()
        cert_ids_by_namespace_ids[ns_id].add(cert_id)

    all_cert_renewals_are_ready = True
    all_balancers_are_ready = True
    for ns_id, cert_ids in sorted(cert_ids_by_namespace_ids.items()):
        incomplete_renewal_cert_ids = set()
        for cert_id in cert_ids:
            get_cert_renewal_req_pb = awacs_api_pb2.GetCertificateRenewalRequest(id=cert_id, namespace_id=ns_id)
            cert_renewal_pb = awacs_cert_service_stub.get_certificate_renewal(
                get_cert_renewal_req_pb).certificate_renewal
            if cert_renewal_pb.spec.incomplete:
                incomplete_renewal_cert_ids.add(cert_id)
        if incomplete_renewal_cert_ids:
            click.echo(click.style(
                'namespace {}: certs {} are incomplete'.format(ns_id, incomplete_renewal_cert_ids), fg='red'))
            all_cert_renewals_are_ready = False

        paused_balancer_ids = set()
        for balancer_pb in list_balancers(balancer_service_stub, ns_id):
            if balancer_pb.meta.transport_paused.value:
                paused_balancer_ids.add(balancer_pb.meta.id)
        if paused_balancer_ids:
            click.echo(click.style(
                'namespace {}: balancers {} are paused'.format(ns_id, paused_balancer_ids), fg='red'))
            all_balancers_are_ready = False

    if not all_cert_renewals_are_ready:
        click.echo(click.style('not all certs are ready', fg='red'))
        exit(1)

    if not all_balancers_are_ready:
        click.echo(click.style('not all balancers are ready', fg='red'))
        exit(1)

    click.echo('pausing all balancers')
    for ns_id, cert_ids in cert_ids_by_namespace_ids.items():
        for balancer_pb in list_balancers(balancer_service_stub, ns_id):
            b_id = balancer_pb.meta.id
            pause_balancer(balancer_service_stub, ns_id, b_id)
            print('paused balancer {}:{}'.format(ns_id, b_id))
    click.echo('paused all balancers')

    click.echo('unpausing all renewals')
    for ns_id, cert_ids in cert_ids_by_namespace_ids.items():
        for cert_id in cert_ids:
            get_cert_renewal_req_pb = awacs_api_pb2.GetCertificateRenewalRequest(id=cert_id, namespace_id=ns_id)
            cert_renewal_pb = awacs_cert_service_stub.get_certificate_renewal(
                get_cert_renewal_req_pb).certificate_renewal
            if cert_renewal_pb.meta.paused.value:
                unpause_cert_renewal_req_pb = awacs_api_pb2.UnpauseCertificateRenewalRequest(
                    id=cert_id, namespace_id=ns_id, version=cert_renewal_pb.meta.version)
                awacs_cert_service_stub.unpause_certificate_renewal(unpause_cert_renewal_req_pb)
                print('unpaused renewal {}:{}'.format(ns_id, cert_id))
            else:
                print('renewal {}:{} is already unpaused'.format(ns_id, cert_id))
    click.echo('unpaused all renewals')


@cli.command('force-renewal')
@click.pass_obj
@click.option('--cert-full-ids-path', required=True, type=click.File('r'))
@click.option('--out-path', required=True, type=click.File('w+'))
@click.option('--confirmation-step', type=int, required=True)
def force_renewal(cfg, cert_full_ids_path, out_path, confirmation_step):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)
    cert_full_ids = []
    for line in cert_full_ids_path:
        line = line.strip()
        if not line:
            continue
        cert_full_ids.append(tuple(line.split()))
    for i, (namespace_id, cert_id) in enumerate(cert_full_ids):
        if i % confirmation_step == 0:
            while 1:
                print('OK? y/n')
                yn = input()
                if yn == 'y':
                    break
                elif yn == 'n':
                    exit(0)

        get_cert_req_pb = awacs_api_pb2.GetCertificateRequest(id=cert_id, namespace_id=namespace_id)
        try:
            cert_pb = awacs_cert_service_stub.get_certificate(get_cert_req_pb).certificate
        except exceptions.NotFoundError:
            print('{}:{} not found'.format(namespace_id, cert_id))
            continue
        if cert_pb.meta.force_renewal.value:
            print('{}:{} is already force-renewed by {}, skipping...'.format(
                namespace_id, cert_id, cert_pb.meta.force_renewal.author))
            continue
        get_cert_renewal_req_pb = awacs_api_pb2.GetCertificateRenewalRequest(id=cert_id, namespace_id=namespace_id)
        try:
            awacs_cert_service_stub.get_certificate_renewal(get_cert_renewal_req_pb).certificate_renewal
        except exceptions.NotFoundError:
            pass
        else:
            click.echo(click.style('> {}:{} already has renewal, skipping...'.format(namespace_id, cert_id), fg='red'))
            continue
        force_cert_renewal_req_pb = awacs_api_pb2.ForceCertificateRenewalRequest()
        force_cert_renewal_req_pb.id = cert_id
        force_cert_renewal_req_pb.namespace_id = namespace_id
        force_cert_renewal_req_pb.version = cert_pb.meta.version
        force_cert_renewal_req_pb.cert_ttl = random.randint(150, 182)
        awacs_cert_service_stub.force_certificate_renewal(force_cert_renewal_req_pb)
        click.echo(click.style('force-renewed {}:{}'.format(namespace_id, cert_id), fg='green'))
        out_path.write('{} {}\n'.format(namespace_id, cert_id))


@cli.command('list-ready-renewals')
@click.pass_obj
def list_ready_renewals(cfg):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)

    req_pb = awacs_api_pb2.ListCertificateRenewalsRequest()
    renewal_pbs = awacs_cert_service_stub.list_certificate_renewals(req_pb).certificate_renewals
    for renewal_pb in renewal_pbs:
        if not renewal_pb.meta.paused.value:
            continue
        assert renewal_pb.meta.paused.author == 'nanny-robot'
        assert renewal_pb.meta.paused.comment == 'Automatic renewal paused due to awacs configuration'
        ns_id, cert_id = renewal_pb.meta.namespace_id, renewal_pb.meta.id
        if not renewal_pb.spec.incomplete:
            print('XXX', ns_id, cert_id)
        if (
            not renewal_pb.spec.incomplete and
            renewal_pb.meta.paused.mtime.ToDatetime() <= datetime.utcnow() - timedelta(hours=1) and
            renewal_pb.meta.mtime.ToDatetime() <= datetime.utcnow() - timedelta(hours=1)
        ):
            print(ns_id, cert_id)


@cli.command('list-invalid-renewals')
@click.pass_obj
def list_invalid_renewals(cfg):
    awacs_rpc = RequestsRpcClient(rpc_url=cfg['awacs_rpc_url'], oauth_token=cfg['awacs_token'])
    awacs_cert_service_stub = awacs_api_stub.CertificateServiceStub(awacs_rpc)

    req_pb = awacs_api_pb2.ListCertificateRenewalsRequest()
    renewal_pbs = awacs_cert_service_stub.list_certificate_renewals(req_pb).certificate_renewals
    for renewal_pb in renewal_pbs:
        if renewal_pb.meta.paused.value:
            continue
        if renewal_pb.meta.paused.comment != 'Unpaused renewal automation':
            continue
        if renewal_pb.meta.paused.author != 'romanovich':
            continue
        target_rev = renewal_pb.meta.target_rev
        cert_rev_pb = awacs_cert_service_stub.get_certificate_revision(
            awacs_api_pb2.GetCertificateRevisionRequest(id=target_rev)).revision
        if 'Russian Federation' in cert_rev_pb.spec.fields.subject:
            print(renewal_pb.meta.namespace_id, renewal_pb.meta.id)


if __name__ == '__main__':
    cli()
