# coding: utf-8
from __future__ import print_function

import json

import click
import click as c
from google.protobuf import text_format
from infra.awacs.proto import model_pb2

from infra.awacs.tools.awacstoolslib.app import Selector
from infra.awacs.tools.awacstoolslib.util import (clone_pb, cli, wait_for_confirmation,
                                                  update_l7_macro_version_preserving_comments,
                                                  update_l7_macro_version, update_base_layer_version,
                                                  update_instancectl_version, update_juggler_checks_bundle_version,
                                                  update_pginx_version, update_instancectl_conf_version,
                                                  update_awacslet_version, add_or_update_endpoint_root_certs_version,
                                                  update_shawshank_version, update_l7_macro)


@cli.command()
@c.option('--selector', type=Selector.parse, required=True)
@c.option('--unprocessed-only', type=bool, default=False, is_flag=True)
@c.option('--processed-only', type=bool, default=False, is_flag=True)
@c.pass_obj
def ls(app, selector, unprocessed_only=False, processed_only=False):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    :type selector: Selector
    :type unprocessed_only: bool
    :type processed_only: bool
    """
    assert not (unprocessed_only and processed_only)
    full_balancer_ids, processed_full_balancer_ids, unprocessed_full_balancer_ids = \
        split_full_balancer_ids(app, selector)

    full_balancer_ids_to_list = full_balancer_ids
    if unprocessed_only:
        full_balancer_ids_to_list = unprocessed_full_balancer_ids
    elif processed_only:
        full_balancer_ids_to_list = processed_full_balancer_ids
    for ns_id, b_id in sorted(full_balancer_ids_to_list):
        c.echo(ns_id + '/' + b_id)


@cli.command('find_in_progress')
@click.option('--playbook', required=True, type=click.File('r'))
@click.option('--index', required=True, type=click.File('r'))
@click.option('--stage', required=True)
@click.option('--location', required=True)
@click.option('--login', required=True)
@click.pass_obj
def find_in_progress(ctx, playbook, index, stage, location, login):
    playbook_data = json.load(playbook)
    ns_ids = playbook_data[stage]

    nanny_client = ctx['nanny']
    index_data = json.load(index)

    service_ids = []
    click.echo('fetching balancers from awacs')
    for ns_id in sorted(ns_ids):
        ns_index_data = sorted(index_data.get(ns_id, {}).items())
        for location_, (balancer_id, service_id) in ns_index_data:
            if location.lower() == location_.lower():
                service_ids.append((ns_id, balancer_id, service_id))

    click.echo('found {} services in {}'.format(len(service_ids), location))
    for ns_id, b_id, service_id in service_ids:
        click.echo('looking at {}:{} (service {})'.format(ns_id, b_id, service_id))
        curr_snapshot_data = nanny_client.get_service_runtime_attrs(service_id, exclude_runtime_attrs=True)
        state_data = nanny_client.get_state(service_id)
        if state_data['current_state']['content']['summary']['value'] != 'ONLINE':
            click.echo(click.style('{}:{} {} is not ONLINE'.format(ns_id, b_id, service_id), fg='blue'))

        if not state_data['target_state']['content']['is_enabled']:
            click.echo(click.style('{}:{} {} is OFFLINE'.format(ns_id, b_id, service_id), fg='red'))
            continue
        if (state_data['current_state']['content']['summary']['value'] != 'ONLINE' and
            (state_data['target_state']['content']['snapshot_info']['author'] == login or
             curr_snapshot_data['change_info']['author'] == login)):
            click.echo(click.style('{}:{} {} is still deploying'.format(ns_id, b_id, service_id), fg='red'))


def _bump_l7_macro(app, balancer_pb, l7_macro_version, ticket):
    namespace_id = balancer_pb.meta.namespace_id
    balancer_id = balancer_pb.meta.id

    update_msg_parts = []
    non_update_msg_parts = []
    tickets = set()

    updated = False
    spec_pb = balancer_pb.spec

    if spec_pb.yandex_balancer.config.HasField('l7_macro'):
        if '### balancer_deployer_sign' in spec_pb.yandex_balancer.yaml:
            c.secho('Not editing YAML, balancer seems to be sedem-managed', fg='yellow')
        elif spec_pb.yandex_balancer.config.l7_macro.HasField('antirobot'):
            c.secho('Not editing YAML, l7_macro.antirobot is set', fg='yellow')
        else:
            if '#' in spec_pb.yandex_balancer.yaml:
                c.secho('l7_macro YAML has comments, using update_l7_macro_version_preserving_comments', fg='yellow')
                upd, msg = update_l7_macro_version_preserving_comments(spec_pb, l7_macro_version)
            else:
                upd, msg = update_l7_macro_version(spec_pb, l7_macro_version)

            if upd:
                updated = True
                update_msg_parts.append(msg)
                tickets.add(ticket)
            else:
                non_update_msg_parts.append(msg)
    else:
        c.secho('Not proceeding, balancer is not l7_macro', fg='red')
        return False, False

    if updated:
        comment = '{}: {}'.format(', '.join(sorted(tickets)), '; '.join(update_msg_parts))
        try:
            app.awacs_client.update_balancer(namespace_id=namespace_id,
                                             balancer_id=balancer_id,
                                             version=balancer_pb.meta.version,
                                             spec_pb=balancer_pb.spec,
                                             comment=comment)
        except Exception as e:
            c.secho('failed to update {}:{}: {}'.format(namespace_id, balancer_id, e), fg='red')
            return False, False
        else:
            c.secho(
                'Updated balancer spec, see https://nanny.yandex-team.ru/ui/#/awacs/'
                'namespaces/list/{}/monitoring/common/'.format(namespace_id), fg='green')
            for msg in update_msg_parts:
                c.secho(' * ' + msg, fg='green')
            return True, True
    else:
        c.echo('Did not update {}:{}'.format(namespace_id, balancer_id))
        assert not update_msg_parts
        for msg in non_update_msg_parts:
            c.echo(' * ' + msg)
        return False, True


@cli.command('bump_easy_mode_version')
@click.option('--selector', type=Selector.parse, required=True)
@click.option('--confirm-every', type=int, default=1)
@click.option('--l7-macro-version', required=True)
@click.option('--ticket', required=True)
@click.pass_obj
def bump_easy_mode_version(app, selector, confirm_every, l7_macro_version, ticket):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    :type selector: Selector
    :type confirm_every: int
    :type l7_macro_version: six.text_type
    :type ticket: six.text_type
    """
    if app.op is None:
        c.secho('Op must be configured (see help)', fg='red')
        return

    full_balancer_ids, processed_full_balancer_ids, unprocessed_full_balancer_ids = \
        split_full_balancer_ids(app, selector)

    c.echo('found {} unprocessed balancers in {}'.format(
        len(unprocessed_full_balancer_ids),
        selector.expr))

    if not unprocessed_full_balancer_ids:
        c.secho('No unprocessed balancers found', fg='red')
        return

    i = 1
    if not wait_for_confirmation('Let\'s start?'):
        return

    rps_data = app.awacs_client.get_yesterday_max_rps_stats_by_balancer()
    for balancer_pb in app.awacs_client.iter_all_balancers(skip_incomplete=True,
                                                           # yp_lite_only=True,
                                                           full_balancer_id_in=unprocessed_full_balancer_ids):
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id

        b_rps = rps_data.get((namespace_id, balancer_id), -1)
        rps = str(int(b_rps)) if b_rps != -1 else 'UNKNOWN'

        c.secho('Looking at {}:{}, {} RPS...'.format(namespace_id, balancer_id, rps), fg='blue')
        if i % confirm_every == 0 and not wait_for_confirmation(
            'Going to update {}:{}'.format(namespace_id, balancer_id)):
            c.echo('Skipped {}:{}...'.format(namespace_id, balancer_id))
            continue

        updated, processed = _bump_l7_macro(app, balancer_pb, l7_macro_version=l7_macro_version, ticket=ticket)
        if updated:
            i += 1
        if processed:
            processed_full_balancer_ids.add((namespace_id, balancer_id))
            save_processed_balancers(app.op, processed_full_balancer_ids)


def bump_components_according_to_swatops_312(balancer_pb):
    tickets = set()
    changes = []
    notchanges = []

    spec_pb = balancer_pb.spec
    prev_spec_pb = clone_pb(spec_pb)

    # https://st.yandex-team.ru/SWATOPS-282#5fd78c1d6a347306eced19c1
    must_set_allow_webdav = spec_pb.components.pginx_binary.version == '212-3'

    updated = False
    updated_, msg = update_pginx_version(
        spec_pb.components.pginx_binary,
        new_version='227-2',
        expected_versions=('198-4', '203-7', '203-11', '208-3', '212-3', '220-1', '221-1', '227-2'))
    if updated_:
        updated = True
        changes.append(msg)
        tickets.add('SWATOPS-282')
        if updated_ and must_set_allow_webdav:
            def f(l7_macro_bp):
                l7_macro_bp.core.allow_webdav = True
                return True, 'set l7_macro.core.allow_webdav'

            update_l7_macro(spec_pb, f)
    else:
        notchanges.append(msg)

    updated_, msg = update_base_layer_version(spec_pb,
                                              to_version='xenial-1',
                                              expected_versions=('precise-1', 'xenial-0', 'xenial-1'))
    if updated_:
        updated = True
        changes.append(msg)
        tickets.add('SWATOPS-103')
    else:
        notchanges.append(msg)

    updated_, msg = update_instancectl_version(spec_pb.components.instancectl,
                                               new_version='2.54',
                                               expected_versions=('2.1', '2.9', '2.54'))
    if updated_:
        updated = True
        changes.append(msg)
        tickets.add('SWATOPS-310')
    else:
        notchanges.append(msg)

    shawshank_layer_pb = spec_pb.components.shawshank_layer
    if shawshank_layer_pb.state == shawshank_layer_pb.SET:
        updated_, msg = update_shawshank_version(spec_pb.components.shawshank_layer,
                                                 new_version='0.0.5',
                                                 expected_versions=('0.0.1-prehistoric', '0.0.2', '0.0.3', '0.0.4'))
        if updated_:
            updated = True
            changes.append(msg)
            tickets.add('SWATOPS-285')
        else:
            notchanges.append(msg)

    instancectl_conf_pb = spec_pb.components.instancectl_conf
    if instancectl_conf_pb.state == instancectl_conf_pb.SET:
        updated_, msg = update_instancectl_conf_version(spec_pb,
                                                        to_version='0.1.7',
                                                        to_pushclient_version='0.1.7-pushclient')
        if updated_:
            updated = True
            changes.append(msg)
            tickets.add('SWATOPS-311')
        else:
            notchanges.append(msg)

    awacslet_pb = spec_pb.components.awacslet
    if awacslet_pb.state == awacslet_pb.SET:
        updated_, msg = update_awacslet_version(spec_pb,
                                                to_version='0.0.4',
                                                to_pushclient_version='0.0.4-pushclient')
        if updated_:
            updated = True
            changes.append(msg)
            tickets.add('SWATOPS-311')
        else:
            notchanges.append(msg)

    updated_, msg = update_juggler_checks_bundle_version(spec_pb.components.juggler_checks_bundle, '0.0.3')
    if updated_:
        updated = True
        changes.append(msg)
        tickets.add('SWATOPS-314')
    else:
        notchanges.append(msg)

    updated_, msg = add_or_update_endpoint_root_certs_version(spec_pb, '0.3.0')
    if updated_:
        updated = True
        changes.append(msg)
        tickets.add('SWATOPS-308')
    else:
        notchanges.append(msg)

    if spec_pb.ctl_version < 5:
        updated = True
        spec_pb.ctl_version = 5
        changes.append('increase max_execution_time for iss_hook_stop')
        tickets.add('SWATOPS-313')
    else:
        notchanges.append('ctl_version is already >= 5')

    if updated:
        assert prev_spec_pb != spec_pb, str(changes)
        return True, '{}: {}'.format(', '.join(sorted(tickets)), ', '.join(changes)), ', '.join(notchanges) or 'nil'
    else:
        assert prev_spec_pb == spec_pb
        return False, 'nil', ', '.join(notchanges) or 'nil'


def save_previous_components_spec(op, namespace_id, balancer_id, components_spec, changelog):
    flat_balancer_id = '{}/{}'.format(namespace_id, balancer_id)
    previous_components_specs = op.data.get('previous_components_specs', {})
    previous_components_specs[flat_balancer_id] = text_format.MessageToString(components_spec)
    op.data['previous_components_specs'] = previous_components_specs

    changelogs = op.data.get('changelogs', {})
    changelogs[flat_balancer_id] = changelog
    op.data['changelogs'] = changelogs
    op.save()


def split_full_balancer_ids(app, selector):
    op = app.op
    processed_full_balancer_ids = set()
    for flat_balancer_id in op.data.get('processed_flat_balancer_ids', []):
        processed_full_balancer_ids.add(tuple(flat_balancer_id.split('/', 1)))
    full_balancer_ids = selector.resolver(app)
    return full_balancer_ids, full_balancer_ids & processed_full_balancer_ids, full_balancer_ids - processed_full_balancer_ids


def save_processed_balancers(op, full_balancer_ids):
    op.data['processed_flat_balancer_ids'] = sorted(['/'.join(full_balancer_id)
                                                     for full_balancer_id in full_balancer_ids])
    op.save()


@cli.command('rollback')
@click.option('--selector', type=Selector.parse, required=True)
@click.option('--confirm-every', type=int, default=1)
@click.pass_obj
def rollback(app, selector, confirm_every):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    :type selector: Selector
    :type confirm_every: int
    """
    if app.op is None:
        click.secho('Op must be configured (see help)', fg='red')
        return

    awacs_client = app.awacs_client

    full_balancer_ids, processed_full_balancer_ids, unprocessed_full_balancer_ids = \
        split_full_balancer_ids(app, selector)

    click.echo('found {} balancers in {}, {} processed'.format(len(full_balancer_ids),
                                                               selector.expr,
                                                               len(processed_full_balancer_ids)))
    c = 0

    for balancer_pb in awacs_client.iter_all_balancers(skip_incomplete=True,
                                                       full_balancer_id_in=processed_full_balancer_ids):
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id
        flat_balancer_id = '{}/{}'.format(namespace_id, balancer_id)

        curr_components_spec_pb = balancer_pb.spec.components

        prev_components_spec_pb = model_pb2.BalancerSpec.ComponentsSpec()
        text_format.Parse(app.op.data['previous_components_specs'][flat_balancer_id], prev_components_spec_pb)
        changelog = app.op.data['changelogs'][flat_balancer_id]

        changed = curr_components_spec_pb != prev_components_spec_pb

        if changed:
            balancer_pb.spec.ClearField('components')
            balancer_pb.spec.components.CopyFrom(prev_components_spec_pb)

            if c % confirm_every == 0 and not wait_for_confirmation(
                'going to rollback {}:{}'.format(namespace_id, balancer_id)):
                click.echo('skipped {}:{}'.format(namespace_id, balancer_id))
                continue

            click.echo(
                'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/monitoring/common/'.format(namespace_id))
            click.echo()

            try:
                awacs_client.update_balancer(namespace_id=namespace_id,
                                             balancer_id=balancer_id,
                                             version=balancer_pb.meta.version,
                                             spec_pb=balancer_pb.spec,
                                             comment='rollback "{}"'.format(changelog))
            except Exception as e:
                click.secho('failed to update {}:{}: {}'.format(namespace_id, balancer_id, e), fg='red')
            else:
                c += 1

        click.secho('rolled back {} balancers'.format(c), fg='green')


@cli.command('to_swatops_312')
@click.option('--selector', type=Selector.parse, required=True)
@click.option('--confirm-every', type=int, default=1)
@click.option('--confirm-automatically-after', type=int, default=-1)
@click.pass_obj
def to_swatops_312(app, selector, confirm_every, confirm_automatically_after=-1):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    :type selector: Selector
    :type confirm_every: int
    :type confirm_automatically_after: int
    """
    if app.op is None:
        click.secho('Op must be configured (see help)', fg='red')
        return

    awacs_client = app.awacs_client
    rps_data = awacs_client.get_yesterday_max_rps_stats_by_balancer()

    full_balancer_ids, processed_full_balancer_ids, unprocessed_full_balancer_ids = \
        split_full_balancer_ids(app, selector)

    click.echo('found {} balancers in {}, {} unprocessed'.format(len(full_balancer_ids),
                                                                 selector.expr,
                                                                 len(unprocessed_full_balancer_ids)))
    c = 0
    if not wait_for_confirmation('Let\'s start?'):
        return
    just_confirmed = True
    for balancer_pb in awacs_client.iter_all_balancers(skip_incomplete=True,
                                                       full_balancer_id_in=unprocessed_full_balancer_ids):
        namespace_id = balancer_pb.meta.namespace_id
        balancer_id = balancer_pb.meta.id
        b_rps = rps_data.get((namespace_id, balancer_id), -1)
        if b_rps > 100:
            rps = click.style(str(int(b_rps)), fg='red')
        else:
            rps = str(int(b_rps)) if b_rps != -1 else 'UNKNOWN'
        click.echo('\nlooking at {}:{}, {} RPS'.format(namespace_id, balancer_id, rps))

        prev_components_spec_pb = clone_pb(balancer_pb.spec.components)
        changed, changelog, nonchangelog = bump_components_according_to_swatops_312(balancer_pb)
        if changed:
            save_previous_components_spec(app.op, namespace_id, balancer_id, prev_components_spec_pb, changelog)

        click.secho('Changelog: ' + changelog, fg='blue')
        click.secho('Nonchangelog: ' + nonchangelog, fg='yellow')

        if changed:
            if not just_confirmed and c % confirm_every == 0:
                if not wait_for_confirmation(
                    'Going to update {}:{}'.format(namespace_id, balancer_id),
                    confirm_automatically_after=confirm_automatically_after):
                    click.echo('Skipped {}:{}...'.format(namespace_id, balancer_id))
                    continue
                else:
                    just_confirmed = True

            click.echo(
                'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/monitoring/common/'.format(namespace_id))
            click.echo()

            try:
                awacs_client.update_balancer(namespace_id=namespace_id,
                                             balancer_id=balancer_id,
                                             version=balancer_pb.meta.version,
                                             spec_pb=balancer_pb.spec,
                                             comment=changelog)
            except Exception as e:
                click.secho('failed to update {}:{}: {}'.format(namespace_id, balancer_id, e), fg='red')
            else:
                just_confirmed = False
                c += 1
                processed_full_balancer_ids.add((namespace_id, balancer_id))
                save_processed_balancers(app.op, processed_full_balancer_ids)
        else:
            processed_full_balancer_ids.add((namespace_id, balancer_id))
            save_processed_balancers(app.op, processed_full_balancer_ids)
        import time
        time.sleep(2)

    click.secho('updated {} balancers'.format(c), fg='green')


if __name__ == '__main__':
    cli()
