# coding: utf-8
from __future__ import print_function, absolute_import

import importlib
import sys
import traceback
import copy

import click
import nanny_rpc_client
import six
from infra.awacs.proto import model_pb2, modules_pb2
from six.moves import input
import yaml

import core
import util
from awacs.model.balancer.generator import get_yandex_config_pb
from awacs.model.balancer.vector import UpstreamVersion
from awacs.wrappers.base import Holder, ValidationCtx
from awacs.lib.yamlparser.wrappers_util import dump_tlem_pb, dump_uem_pb, float_representer
from awacs.yamlparser.util import AwacsYamlDumper
from rules.model import clone_pb, clone_pb_dict, BaseBalancerSuggester, BaseUpstreamSuggester


DEFAULT_AWACS_RPC_URL = 'https://awacs.yandex-team.ru/api/'

YamlDumper = copy.deepcopy(AwacsYamlDumper)
YamlDumper.add_representer(float, float_representer)


def dump_pb(pb):
    from infra.awacs.tools.awacsfmt.util import pb_to_dict, parse
    assert isinstance(pb, modules_pb2.Holder)
    d = pb_to_dict(pb)
    yml = yaml.dump(d, default_flow_style=False, Dumper=YamlDumper, width=200)
    pb.DiscardUnknownFields() # TODO del after vendoring lastest awacs version
    assert parse(modules_pb2.Holder, yml) == pb
    return yml



def create_awacs_upstream_href(namespace_id, upstream_id):
    return 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{0}/upstreams/list/{1}/show/'.format(
        namespace_id, upstream_id)


def create_awacs_balancer_href(namespace_id, balancer_id):
    return 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{0}/balancers/list/{1}/show/'.format(
        namespace_id, balancer_id)


def l7_macro_to_instance_macro(balancer_spec_pb, ctx):
    balancer_pb = get_yandex_config_pb(balancer_spec_pb)
    bal = Holder(balancer_pb)
    bal.expand_immediate_contained_macro(ctx=ctx)
    return bal


@click.group()
@click.option('--awacs-token', envvar='AWACS_TOKEN', required=True)
@click.option('--awacs-rpc-url', envvar='AWACS_RPC_URL', default=DEFAULT_AWACS_RPC_URL)
@click.pass_context
def cli(ctx, awacs_rpc_url, awacs_token):
    ctx.obj = core.App(awacs_rpc_url=awacs_rpc_url, awacs_token=awacs_token)


def suggest_(d, namespace_id, balancer_id, rule, do_update, update_with_diff, ticket,
             upstream_id=None, extra_opts=None):
    if extra_opts is None:
        extra_opts = {}

    try:
        balancer_state_pb = d.get_balancer_state(namespace_id, balancer_id)
        namespace_pb = d.get_namespace(namespace_id)
    except nanny_rpc_client.exceptions.NotFoundError:
        click.echo('Balancer {}:{} is not found'.format(namespace_id, balancer_id))
        return False
    active_vector = core.balancer_state_to_active_vector(namespace_id, balancer_id, balancer_state_pb)
    if not active_vector.balancer_version:
        click.echo('No active configuration for balancer {}:{}, nothing to check'.format(namespace_id, balancer_id))
        return False
    (balancer_spec_pb,
     domain_spec_pbs,
     upstream_spec_pbs,
     backend_spec_pbs,
     endpoint_set_spec_pbs,
     knob_spec_pbs,
     cert_spec_pbs) = core.vector_to_specs(d, active_vector)

    # we want to use all upstreams for core.vector_to_config_holder
    # because original instance_macro and new one may differ in included upstreams
    for upstream in d.list_upstreams(namespace_id):
        ver = UpstreamVersion.from_pb(upstream)
        upstream_spec_pbs[ver] = upstream.spec

    # instance_macro_yml = balancer_spec_pb.yandex_balancer.yaml
    # instance_macro_yml = dump_pb(balancer_spec_pb.yandex_balancer.config)

    if rule == 'tlem':
        pb = copy.deepcopy(balancer_spec_pb.yandex_balancer.config)
        from awacs.wrappers.base import Holder
        Holder(pb).to_normal_form_XXX()
        instance_macro_yml = dump_pb(pb)
        del pb

        if not instance_macro_yml.startswith('l7_macro:'):
            print(instance_macro_yml)
    elif rule == 'uem':
        is_upstream_already_easy_mode2 = False
        upstream_version = None
        for v, upstream_spec_pb in six.iteritems(upstream_spec_pbs):
            if v.upstream_id[1] == upstream_id:
                if upstream_spec_pb.yandex_balancer.mode == upstream_spec_pb.yandex_balancer.EASY_MODE2:
                    is_upstream_already_easy_mode2 = True
                upstream_version = v
                break

        upstream_spec_pb = upstream_spec_pbs[upstream_version]
        pb = copy.deepcopy(upstream_spec_pb.yandex_balancer.config)
        from awacs.wrappers.base import Holder
        Holder(pb).to_normal_form_XXX()
        up_yml = dump_pb(pb)
        del pb
        print(up_yml)

        if is_upstream_already_easy_mode2 and rule == 'uem':
            print(u'Upstream {} is already in EASY_MODE2'.format(upstream_id))
            return True

    # balancer = core.vector_to_config_holder(active_vector,
    #                                         clone_pb(namespace_pb),
    #                                         clone_pb(balancer_spec_pb),
    #                                         clone_pb_dict(domain_spec_pbs),
    #                                         clone_pb_dict(upstream_spec_pbs),
    #                                         clone_pb_dict(backend_spec_pbs),
    #                                         clone_pb_dict(endpoint_set_spec_pbs),
    #                                         clone_pb_dict(knob_spec_pbs),
    #                                         clone_pb_dict(cert_spec_pbs))

    impl = importlib.import_module('rules.{}'.format(rule))
    checker = impl.Checker(namespace_id, balancer_id, **extra_opts)

    if isinstance(checker, BaseBalancerSuggester):
        assert not upstream_id
        if rule == 'tlem2full':
            ok, msg, balancer_config_pb = checker.suggest(
                balancer_spec_pb,
                domain_spec_pbs={v.domain_id: pb for v, pb in domain_spec_pbs.items()},
                cert_spec_pbs={v.cert_id: pb for v, pb in cert_spec_pbs.items()})
        else:
            ok, msg, balancer_config_pb = checker.suggest(balancer_spec_pb)
    elif isinstance(checker, BaseUpstreamSuggester):
        assert upstream_id and upstream_version
        other_specs = upstream_spec_pbs.copy()
        del other_specs[upstream_version]
        ok, msg, upstream_config_pb = checker.suggest(
            upstream_id, upstream_spec_pb, other_specs)
    else:
        raise AssertionError()

    if ok:
        if upstream_id:
            l7_upstream_config_holder_pb = modules_pb2.Holder()
            if upstream_config_pb.DESCRIPTOR.name == 'L7UpstreamMacro':
                l7_upstream_config_holder_pb.l7_upstream_macro.CopyFrom(upstream_config_pb)
            elif upstream_config_pb.DESCRIPTOR.name == 'Holder':
                l7_upstream_config_holder_pb.CopyFrom(upstream_config_pb)
            else:
                raise AssertionError(upstream_config_pb.DESCRIPTOR.name)
            cloned_cert_spec_pbs = clone_pb_dict(cert_spec_pbs)
        else:
            l7_balancer_config_holder_pb = modules_pb2.Holder()
            if balancer_config_pb.DESCRIPTOR.name == 'L7Macro':
                l7_balancer_config_holder_pb.l7_macro.CopyFrom(balancer_config_pb)
            elif balancer_config_pb.DESCRIPTOR.name == 'Holder':
                l7_balancer_config_holder_pb.CopyFrom(balancer_config_pb)
            else:
                raise AssertionError(balancer_config_pb.DESCRIPTOR.name)
            cloned_cert_spec_pbs = clone_pb_dict(cert_spec_pbs)
            if balancer_config_pb.DESCRIPTOR.name == 'L7Macro' and balancer_config_pb.https.certs:
                cert_full_id = (namespace_id, balancer_config_pb.https.certs[0].id)
                cert_version = core.CertVersion(0, cert_full_id, '', False, False)
                cloned_cert_spec_pbs[cert_version] = model_pb2.CertificateSpec()

        balancer = core.vector_to_config_holder(active_vector,
                                                clone_pb(namespace_pb),
                                                clone_pb(balancer_spec_pb),
                                                clone_pb_dict(domain_spec_pbs),
                                                clone_pb_dict(upstream_spec_pbs),
                                                clone_pb_dict(backend_spec_pbs),
                                                clone_pb_dict(endpoint_set_spec_pbs),
                                                clone_pb_dict(knob_spec_pbs),
                                                cloned_cert_spec_pbs)

        balancer.expand_macroses()
        before_lua = balancer.to_config().to_top_level_lua()

        if upstream_id:
            upstream_spec_pb = upstream_spec_pbs[upstream_version]
            prev_l7_upstream_config_holder_pb = clone_pb(upstream_spec_pb.yandex_balancer.config)
            upstream_spec_pb.yandex_balancer.Clear()
            upstream_spec_pb.yandex_balancer.config.CopyFrom(l7_upstream_config_holder_pb)

            uem_yml = dump_uem_pb(l7_upstream_config_holder_pb)
            print(uem_yml)

            from awacs.wrappers.base import Holder
            l7_upstream_macro_holder = Holder(clone_pb(upstream_spec_pb.yandex_balancer.config))
            regexp_section_pb = l7_upstream_macro_holder.module.to_regexp_section_pb()
            l7_upstream_macro_holder.expand_immediate_contained_macro()
            l7_upstream_macro_holder.to_normal_form_XXX()
            regexp_section_pb.nested.CopyFrom(l7_upstream_macro_holder.pb)
            del l7_upstream_macro_holder

            expanded_l7_upstream_macro_holder = modules_pb2.Holder()
            expanded_l7_upstream_macro_holder.regexp_section.CopyFrom(regexp_section_pb)
            expanded_l7_upstream_macro_holder = Holder(expanded_l7_upstream_macro_holder)
        else:
            prev_balancer_config_holder_pb = clone_pb(balancer_spec_pb.yandex_balancer.config)
            balancer_spec_pb.yandex_balancer.Clear()
            balancer_spec_pb.yandex_balancer.config.CopyFrom(l7_balancer_config_holder_pb)

            tlem_yml = dump_tlem_pb(l7_balancer_config_holder_pb)
            print(tlem_yml)

            expanded_l7_macro_holder = l7_macro_to_instance_macro(
                balancer_spec_pb,
                ValidationCtx(
                    domain_config_pbs={v.domain_id: pb.yandex_balancer.config for v, pb in domain_spec_pbs.items()}
                )
            )
            expanded_l7_macro_holder.to_normal_form_XXX()

        balancer = core.vector_to_config_holder(active_vector,
                                                clone_pb(namespace_pb),
                                                clone_pb(balancer_spec_pb),
                                                clone_pb_dict(domain_spec_pbs),
                                                clone_pb_dict(upstream_spec_pbs),
                                                clone_pb_dict(backend_spec_pbs),
                                                clone_pb_dict(endpoint_set_spec_pbs),
                                                clone_pb_dict(knob_spec_pbs),
                                                cloned_cert_spec_pbs)
        balancer.expand_macroses()
        after_lua = balancer.to_config().to_top_level_lua()

        if rule in ('uem', 'tlem'):
            # tlem_yml = dump_tlem_pb(l7_balancer_config_holder_pb)
            # print(tlem_yml)

            def print_diff(a, b, name='', full_diff=True):
                if name != '' and not name.startswith(' '):
                    name = ' ' + name
                d = util.get_diff(a, b, n=3)
                print('{}:{}\'s diff is not zero, stopping:'.format(namespace_id, balancer_id))
                print('diff{}:'.format(name))
                print(d)
                print('end diff{}.'.format(name))
                if full_diff:
                    print('full diff{}:'.format(name))
                    print(util.get_diff(a, b, n=1000000))
                    print('end full diff{}.'.format(name))

            if before_lua != after_lua:
                print_diff(before_lua, after_lua)

            if rule == 'tlem':
                before_intance_macro = instance_macro_yml
                after_instance_macro = dump_pb(expanded_l7_macro_holder.pb)

                if before_intance_macro != after_instance_macro:
                    print_diff(before_intance_macro, after_instance_macro, 'instance_macro')
            elif rule == 'uem':
                before_up_yml = up_yml
                after_up_yml = dump_pb(expanded_l7_upstream_macro_holder.pb)
                if before_up_yml != after_up_yml:
                    print_diff(before_up_yml, after_up_yml, 'upstream_full_mode')

        if rule not in ('uem', 'tlem') or (before_lua == after_lua or update_with_diff):
            if upstream_id:
                if l7_upstream_config_holder_pb.HasField('l7_upstream_macro'):
                    uem_yml = dump_uem_pb(l7_upstream_config_holder_pb)
                else:
                    uem_yml = dump_pb(l7_upstream_config_holder_pb)
                # print(uem_yml)
                # diff = util.get_diff(before_lua, after_lua)
                # print('{}:{}\'s diff is not zero:'.format(namespace_id, balancer_id))
                # print(diff)
                # print(after_lua)
                # print('OK?')
                # if input() != 'y':
                #    1/0

                if rule == 'uemnocompat':
                    from awacs.wrappers.base import Holder
                    m = Holder(l7_upstream_config_holder_pb)
                    m.validate()
                    prev_uem_yml = dump_uem_pb(prev_l7_upstream_config_holder_pb)
                    uem_yml = dump_uem_pb(l7_upstream_config_holder_pb)
                    if prev_uem_yml == uem_yml:
                        print('{}:{} has no compat options, skipping...'.format(namespace_id, upstream_id))
                        return
                    print('--- {}:{}'.format(namespace_id, upstream_id))
                    print(util.get_diff(prev_uem_yml, uem_yml))
                    print('---')
                else:
                    print('{}:{}:{}\'s YAML:'.format(namespace_id, balancer_id, upstream_id))
                    print(upstream_id)
                    print(uem_yml)

                if do_update or update_with_diff:
                    print('OK?')
                    if input() == 'y':
                        upstream_pb = d.get_upstream(namespace_id, upstream_id)
                        spec_pb = upstream_pb.spec
                        spec_pb.yandex_balancer.yaml = uem_yml
                        if rule == 'uem':
                            spec_pb.yandex_balancer.mode = spec_pb.yandex_balancer.EASY_MODE2
                        elif rule == 'uem2full':
                            spec_pb.yandex_balancer.mode = spec_pb.yandex_balancer.FULL_MODE
                        else:
                            raise AssertionError
                        updated_upstream_pb = d.update_upstream(
                            namespace_id, upstream_id, upstream_pb.meta.version, spec_pb,
                            comment=u'https://st.yandex-team.ru/{}'.format(ticket))
                        print('Updated upstream', upstream_id)
                        print(create_awacs_upstream_href(namespace_id, upstream_id))
                        return True
                    else:
                        return False
                else:
                    return True
            else:
                if not l7_balancer_config_holder_pb.HasField('l7_macro'):
                    tlem_yml = dump_pb(l7_balancer_config_holder_pb)

                if rule == 'tlemnocompat':
                    from awacs.wrappers.base import Holder
                    m = Holder(l7_balancer_config_holder_pb)
                    m.validate()
                    prev_l7_macro_holder_pb = modules_pb2.Holder(l7_macro=prev_balancer_config_holder_pb)
                    prev_uem_yml = dump_tlem_pb(prev_l7_macro_holder_pb)
                    uem_yml = dump_tlem_pb(l7_balancer_config_holder_pb)
                    if prev_uem_yml == uem_yml:
                        print('{}:{} has no compat options, skipping...'.format(namespace_id, balancer_id))
                        return
                    print('--- {}:{}'.format(namespace_id, balancer_id))
                    print(util.get_diff(prev_uem_yml, uem_yml))
                    print('---')
                else:
                    print('{}:{}\'s YAML:'.format(namespace_id, balancer_id))
                    print(tlem_yml)

                if do_update or update_with_diff:
                    print('OK?')
                    if input() == 'y':
                        balancer_pb = d.get_balancer(namespace_id, balancer_id)
                        spec_pb = balancer_pb.spec
                        spec_pb.yandex_balancer.yaml = tlem_yml
                        spec_pb.yandex_balancer.mode = spec_pb.yandex_balancer.EASY_MODE
                        updated_balancer_pb = d.update_balancer(
                            namespace_id, balancer_id, balancer_pb.meta.version, spec_pb,
                            comment='https://st.yandex-team.ru/{}'.format(ticket))
                        print('Updated balancer', balancer_id)
                        print(create_awacs_balancer_href(namespace_id, balancer_id))
                        return True
                    else:
                        return False
                else:
                    return True

    else:
        print(namespace_id, balancer_id, 'does not seem ok')
        if upstream_id:
            print(create_awacs_upstream_href(namespace_id, upstream_id), msg)
        else:
            print(create_awacs_balancer_href(namespace_id, balancer_id), msg)
    return False


@cli.command()
@click.pass_obj
@click.option('--namespace-id', default=None)
@click.option('--balancer-id', default=None)
@click.option('--upstream-id', default=None)
@click.option('--input-file', default=None)
@click.option('--uem', default=False)
@click.option('--rule', required=True)
@click.option('--do-update', is_flag=True, default=False)
@click.option('--update-with-diff', is_flag=True, default=False)
@click.option('--l7-macro-version', default='0.0.1')
@click.option('--ticket', default=None)
def suggest(d, namespace_id, balancer_id, upstream_id, input_file, uem, rule, do_update, update_with_diff, l7_macro_version,
            ticket):
    if upstream_id is not None:
        assert uem

    if do_update:
        assert ticket is not None
        assert input_file != '-'

    if input_file == '-':
        assert not do_update

    if input_file is None:
        assert namespace_id is not None
    else:
        assert namespace_id is None and balancer_id is None and upstream_id is None

    def get_args_from_file(file, namespaces=True, balancers=True):
        namespace_ids = []
        balancer_ids = []
        for line in file.readlines():
            item = line.strip()

            fragments = item.split()
            i = 0
            if namespaces:
                namespace_ids.append(fragments[i])
                i += 1
            if balancers:
                balancer_id_list = fragments[i:]
                balancer_ids.append(balancer_id_list)
        return namespace_ids, balancer_ids

    from sepelib.core import config
    config.load()

    if input_file is not None:
        if input_file == '-':
            namespace_ids, balancer_ids = get_args_from_file(sys.stdin)
        else:
            with open(input_file) as f:
                namespace_ids, balancer_ids = get_args_from_file(f)

        upstream_ids = None
    else:
        namespace_ids = [namespace_id]

        balancer_ids = None
        if balancer_id is not None:
            balancer_ids = balancer_id

        upstream_ids = None
        if upstream_id is not None:
            upstream_ids = upstream_id

    try:
        for namespace_id in namespace_ids:
            if 'taxi' in namespace_id or 'maps.yandex' in namespace_id:
                continue

            if balancer_ids is None or balancer_ids == []:
                balancer_id_list = d.list_balancer_ids(namespace_id)
            elif isinstance(balancer_ids, list):
                balancer_id_list = balancer_ids.pop(0)
                if len(balancer_id_list) == 0:
                    balancer_id_list = d.list_balancer_ids(namespace_id)
            elif isinstance(balancer_ids, unicode):
                balancer_id_list = [balancer_ids]
            else:
                raise AssertionError('balancer_ids is a {}'.format(upstream_ids.__class__))

            if uem:
                if upstream_ids is None or upstream_ids == []:
                    upstream_id_list = d.list_upstream_ids(namespace_id)
                elif isinstance(upstream_ids, list):
                    upstream_id_list = upstream_ids.pop(0)
                    if len(upstream_id_list) == 0:
                        upstream_id_list = d.list_upstream_ids(namespace_id)
                elif isinstance(upstream_ids, unicode):
                    upstream_id_list = [upstream_ids]
                else:
                    raise AssertionError('upstream_ids is a {}'.format(upstream_ids.__class__))
            else:
                upstream_id_list = None

            try:
                if upstream_id_list is not None:
                    for balancer_id in balancer_id_list:
                        for upstream_id in copy.copy(upstream_id_list):
                            if upstream_id in ('awacs-balancer-health-check', 'slbping'):
                                continue
                            try:
                                print('namespace {}, balancer {}, upstream {}'.format(namespace_id, balancer_id,
                                                                                      upstream_id))
                                upstream_ok = suggest_(
                                    d, namespace_id, balancer_id, rule,
                                    upstream_id=upstream_id,
                                    do_update=do_update,
                                    update_with_diff=update_with_diff,
                                    ticket=ticket,
                                )
                            except Exception:
                                upstream_ok = False
                                print('failed to process upstream {} for {}:{}'.format(
                                    upstream_id, namespace_id, balancer_id))
                                traceback.print_exc(file=sys.stdout)
                            finally:
                                print('*****')
                            if not upstream_ok:
                                upstream_id_list.remove(upstream_id)
                else:
                    for balancer_id in balancer_id_list:
                        try:
                            print('namespace {}, balancer {}'.format(namespace_id, balancer_id))
                            extra_opts = {}
                            if rule == 'tlem':
                                extra_opts['l7_macro_version'] = l7_macro_version
                            suggest_(d, namespace_id, balancer_id, rule,
                                     ticket=ticket,
                                     do_update=do_update, update_with_diff=update_with_diff, extra_opts=extra_opts)
                        except Exception:
                            print('failed to process {}:{}'.format(namespace_id, balancer_id))
                            traceback.print_exc(file=sys.stdout)
                        finally:
                            print('*****')
            except Exception:
                traceback.print_exc(file=sys.stdout)
            finally:
                sys.stderr.flush()
                sys.stdout.flush()
    except:
        traceback.print_exc(file=sys.stdout)
        raise SystemExit(1)


@cli.command()
@click.pass_context
@click.option('--namespace-id', required=True)
@click.option('--do-update', is_flag=True, default=False)
@click.option('--update-with-diff', is_flag=True, default=False)
@click.option('--l7-macro-version', default='0.0.1')
def full2tlem(ctx, namespace_id, do_update, update_with_diff, l7_macro_version):
    ctx.invoke(suggest, namespace_id=namespace_id, do_update=do_update, update_with_diff=update_with_diff, rule='tlem',
               l7_macro_version=l7_macro_version)


@cli.command()
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--upstream-id', required=True)
def uem2full(d, namespace_id, upstream_id):
    upstream_pb = d.get_upstream(namespace_id, upstream_id)
    impl = importlib.import_module('rules.uem2full')
    ok, msg, upstream_config_pb = impl.Checker(namespace_id, upstream_id).suggest(upstream_id, upstream_pb.spec)
    print(dump_pb(upstream_config_pb))
    print('---')


@cli.command()
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--balancer-id', required=True)
def tlem2full(d, namespace_id, balancer_id):
    balancer_pb = d.get_balancer(namespace_id, balancer_id)
    impl = importlib.import_module('rules.tlem2full')
    domain_spec_pbs = {(dom.meta.namespace_id, dom.meta.id): dom.spec for dom in d.list_domains(namespace_id)}
    cert_spec_pbs = {(cert.meta.namespace_id, cert.meta.id): cert.spec for cert in d.list_certs(namespace_id)}
    ok, msg, balancer_config_pb = impl.Checker(namespace_id, balancer_id).suggest(balancer_pb.spec,
                                                                                  domain_spec_pbs=domain_spec_pbs,
                                                                                  cert_spec_pbs=cert_spec_pbs)
    print(dump_pb(balancer_config_pb))
    print('---')


cli.add_command(suggest, name='suggest')

if __name__ == '__main__':
    cli()
