# coding: utf-8
import difflib
import shlex
import subprocess
import tempfile

from colorama import Fore


def color_diff(diff):
    for line in diff:
        if line.startswith('+'):
            yield Fore.GREEN + line + Fore.RESET
        elif line.startswith('-'):
            yield Fore.RED + line + Fore.RESET
        elif line.startswith('^'):
            yield Fore.BLUE + line + Fore.RESET
        else:
            yield line


def get_diff1(a, b):
    a = a.splitlines(1)
    b = b.splitlines(1)
    diff = difflib.unified_diff(a, b)
    diff = color_diff(diff)
    return ''.join(diff)


def get_diff2(a, b, n, colorize=False, colorize_only=False, skip_spaces=False, color_type=Fore):
    assert not colorize_only or colorize

    diff = difflib.unified_diff(a, b, n=n)
    diff = list(diff)
    end = len(diff)

    for i in range(end):
        if any(map(diff[i].startswith, ('---', '+++', '@@'))):
            continue
        elif diff[i].startswith(' '):
            diff[i] = diff[i][1:]
        else:
            sign = diff[i][0]
            assert sign in ('+', '-')

            if colorize:
                # if s.strip() == '':
                #     color_type = Back
                # else:
                #     color_type = Fore

                if sign == '+':
                    color = color_type.GREEN
                else:
                    color = color_type.RED

                reset_color = color_type.RESET
            else:
                color = ''
                reset_color = ''

            if colorize_only:
                open = ''
                close = ''
            else:
                open = '{' + sign
                close = sign + '}'

            j = i + 1
            has_non_space = False
            while j < end and diff[j].startswith(sign):
                s = diff[j][1:]
                if s == ' ':
                    s = ' '
                else:
                    has_non_space = True
                diff[j] = ' ' + s
                j += 1

            s = diff[i][1:]
            if skip_spaces:
                if s == ' ':
                    s = ' '
                    diff[i] = ' ' + s

                    if not has_non_space:
                        continue

            diff[i] = color + open + s
            diff[j - 1] += close + reset_color
    return ''.join(diff)


def get_diff3(a, b, n=3, colorize=True):
    with tempfile.NamedTemporaryFile() as f:
        f.write(a)
        f.seek(0)
        cmd = shlex.split("diff -U {} -b {} -".format(n, f.name))

        process = subprocess.Popen(cmd,
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.STDOUT,
                                   stdin=subprocess.PIPE)
        stdout, _ = process.communicate(b)
    diff = stdout.splitlines(True)
    if colorize:
        diff = color_diff(diff)
    return ''.join(diff)


def get_diff(a, b, n=3, colorize=True):
    # return get_diff2(a, b, n=n, colorize=True, colorize_only=True, skip_spaces=True)
    return get_diff3(a, b, n=n, colorize=colorize)
