import sys
import tempfile

import click
import pymongo

from infra.awacs.proto import model_pb2
from infra.awacs.tools.awacstoolslib.util import cli, App, create_awacs_upstream_href
from infra.awacs.tools.awacsemtool2.core import NamespaceConfig
from infra.awacs.tools.awacsemtool2.rules import uem_anttsov, uem_simplified  # noqa
from infra.awacs.tools.awacsemtool2 import util
from awacs.lib.yamlparser.wrappers_util import dump_uem_pb


DEFAULT_CACHE_DIR = './ns-config-cache'
DEFAULT_MONGO_DB = 'production_awacs'


@cli.command()
@click.pass_obj
@click.option('--namespace-id', required=True)
@click.option('--upstream-id', required=True)
@click.option('--do-update', is_flag=True, default=False)
@click.option('--use-cache', is_flag=True, default=False)
@click.option('--cache-dir', envvar='CACHE_DIR', default=DEFAULT_CACHE_DIR, type=click.Path(resolve_path=True))
@click.option('--mongo-url', envvar='MONGO_URL', default=None,
              help='Something like mongodb://production_awacs:XXX@myt0-3095.search.yandex.net:27017/production_awacs')
@click.option('--mongo-db', envvar='MONGO_DB', default=DEFAULT_MONGO_DB)
def full2uem(app: App, namespace_id: str, upstream_id: str, do_update=False,
             use_cache=False, cache_dir=DEFAULT_CACHE_DIR,
             mongo_url=None, mongo_db=DEFAULT_MONGO_DB):
    # We read environment from awacs API,
    # but accessing MongoDB directly for revisions will speed up this process
    a = app.awacs_client
    db = None
    if mongo_url is not None:
        m = pymongo.MongoClient(mongo_url, readPreference='secondary')
        db = util.MongoStorage(m[mongo_db])

    # Cache-related logic
    if use_cache:
        ns_config = NamespaceConfig.maybe_from_cache(cache_dir, namespace_id)
        if ns_config is None:
            ns_config = NamespaceConfig.from_api(a, namespace_id, db=db)
            ns_config.to_cache(cache_dir)
    else:
        ns_config = NamespaceConfig.from_api(a, namespace_id, db=db)

    # Find our upstream
    new_ns_config = ns_config.copy()
    upstream_spec_pb = new_ns_config.must_get_upstream_spec(upstream_id)

    # Ask the checker for how we should modify it
    checker = uem_simplified.Checker(namespace_id, upstream_id)
    try:
        ok, msg, suggested_config_pb = checker.suggest(upstream_id,
                                                       holder_pb=upstream_spec_pb.yandex_balancer.config,
                                                       namespace_config=new_ns_config)
    except uem_simplified.MismatchError as e:
        click.secho(f'Failed to suggest easymode config:\n{e}', fg='red')
        sys.exit(1)

    if not ok:
        click.secho(f'Failed to suggest easymode config: {msg}', fg='red')
        if checker.warnings:
            click.secho('Warnings:', fg='red')
            for w in checker.warnings:
                click.secho(repr(w), fg='red')
        sys.exit(1)

    # Modify the upstream and update the env
    # It's important to update both .yaml and .config
    upstream_spec_pb.yandex_balancer.mode = model_pb2.YandexBalancerUpstreamSpec.EASY_MODE2
    upstream_spec_pb.yandex_balancer.yaml = dump_uem_pb(suggested_config_pb)
    upstream_spec_pb.yandex_balancer.config.CopyFrom(suggested_config_pb)
    new_ns_config.replace_upstream_spec(upstream_id, upstream_spec_pb)

    lua = ns_config.to_lua()
    new_lua = new_ns_config.to_lua()

    if lua == new_lua:
        click.secho('No diff found :-)', fg='green')
        if do_update:
            new_ns_config.sync_upstream_spec(a, upstream_id, comment='Hello from awacsemtool2!')
            click.secho(f'Updated {namespace_id}:{upstream_id}, '
                        f'see {create_awacs_upstream_href(namespace_id, upstream_id)}', fg='green')
    else:
        with tempfile.NamedTemporaryFile(delete=False, suffix='.lua') as ns_lua_f:
            ns_lua_f.write(lua.encode('utf-8'))
        with tempfile.NamedTemporaryFile(delete=False, suffix='.lua') as new_ns_lua_f:
            new_ns_lua_f.write(new_lua.encode('utf-8'))
        click.secho('Diff found :-(', fg='red')
        print(f'Prev config: file://{ns_lua_f.name}')
        print(f'New config: file://{new_ns_lua_f.name}')
        print(util.get_diff(lua, new_lua))
