from __future__ import unicode_literals

import argparse
import errno
import os
import sys

import jinja2
import six

from infra.awacs.tools.awacsj2.yav import YavGetter


def parse_args(argv):
    argv = [arg.decode("utf-8") for arg in argv]

    parser = argparse.ArgumentParser()
    parser.add_argument('--src', type=argparse.FileType('r'), required=True, help='Source template file')
    parser.add_argument('--dst', type=argparse.FileType('w'), required=True, help='Destination rendered result file')
    return parser.parse_args(argv)


def makedirs_ignore_existing(dir_path, mode=0o777):
    try:
        os.makedirs(dir_path, mode=mode)
    except OSError as e:
        if e.errno != errno.EEXIST:
            raise


def write(dst_path, content):
    dst_dir_path = os.path.dirname(dst_path)
    dst_file_name = os.path.basename(dst_path)
    path = os.path.join(dst_dir_path, dst_file_name)
    makedirs_ignore_existing(dst_dir_path)
    with open(path, 'w') as template_dst:
        if isinstance(content, six.text_type):
            content = content.encode('utf-8')
        template_dst.write(content)


def main():
    args = parse_args(sys.argv[1:])
    src_path = str(args.src.name)
    dst_path = str(args.dst.name)

    context = {
        'get_yav_with_oauth': YavGetter.get_yav_with_oauth,
        'get_yav_with_rsa_auth': YavGetter.get_yav_with_rsa_auth,
    }
    context.update(os.environ)
    env = jinja2.Environment(loader=jinja2.FileSystemLoader('.'), undefined=jinja2.StrictUndefined)

    try:
        content = env.get_template(src_path).render(**context)
    except Exception as e:
        raise RuntimeError('Cannot process template {}: {}'.format(src_path, e))

    try:
        write(dst_path, content)
    except Exception as e:
        raise RuntimeError('Cannot save {}: {}'.format(dst_path, e))

    print("Success: rendered template {} saved to {}".format(src_path, dst_path))


if __name__ == '__main__':
    main()
