# coding: utf-8
import collections
import importlib
import json
import random
import traceback

import click
import requests
import six
import urllib3
import yaml
from awacs.wrappers.base import ValidationCtx, ANY_MODULE
from awacs.yamlparser.util import AwacsYamlDumper
from infra.swatlib.auth import staff

import core
from infra.awacs.tools.awacstoolslib.util import cli
from rules import uem
from rules.model import visit, FullId, Warning


urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)


def check_namespace(awacs, namespace_id, rule, check_balancers=False, pretty=False):
    """
    :type awacs: infra.awacs.tools.awacstoolslib.awacsclient.AwacsClient
    """
    impl = importlib.import_module('rules.{}'.format(rule))
    warnings = set()

    if hasattr(impl, 'UpstreamChecker'):
        upstream_pbs = awacs.list_upstreams(namespace_id)
        for upstream_pb in upstream_pbs:
            upstream_id = upstream_pb.meta.id
            checker = impl.UpstreamChecker(full_id=FullId('upstream', namespace_id, upstream_id))
            upstream_config = core.get_upstream_config_holder_from_pb(upstream_pb)
            try:
                ctx = ValidationCtx(config_type=ValidationCtx.CONFIG_TYPE_UPSTREAM)
                upstream_config.validate(ctx=ctx, preceding_modules=[ANY_MODULE])
            except Exception as e:
                checker.warn(path=[], message=six.text_type(e), severity=10, tags={'invalid'})
            else:
                visit(upstream_config, checker)

                if rule == 'uem':
                    if checker.already_ok:
                        checker.warn(path=[], message='This upstream is already in UEM.', severity=1,
                                     tags={'already-uem'})
                    else:
                        if not checker.ok and not checker.warnings:
                            checker.warn(path=[], message='This upstream is not simple enough.', severity=1,
                                         tags={'not-uem'})
                        if upstream_id != 'slbping' and checker.ok and getattr(checker, 'two_level', False):
                            print(upstream_pb.meta.namespace_id, upstream_pb.meta.id, '<- TWOLEVEL')
            warnings.update(checker.warnings)
        if check_balancers:
            balancer_pbs = awacs.list_balancers(namespace_id)
            for balancer_pb in balancer_pbs:
                balancer_id = balancer_pb.meta.id
                checker = impl.UpstreamChecker(full_id=FullId('balancer', namespace_id, balancer_id))
                balancer_config = core.get_upstream_config_holder_from_pb(balancer_pb)
                visit(balancer_config, checker)
                warnings.update(checker.warnings)

    if hasattr(impl, 'BalancerChecker'):
        namespace_pb = awacs.get_namespace(namespace_id)
        balancer_ids = awacs.list_balancer_ids(namespace_id)

        for balancer_id in balancer_ids[:1]:
            checker = impl.BalancerChecker(full_id=FullId('balancer', namespace_id, balancer_id))
            balancer_config, err = core.get_balancer_config_holder(awacs, namespace_pb, namespace_id, balancer_id)
            if err is None:
                visit(balancer_config, checker)
                warnings.update(checker.warnings)
            else:
                checker.warn(path=[], message=six.text_type(err), severity=10, tags={'invalid'})
                click.echo(click.style('Failed to get config for {}: {}'.format(balancer_id, err), fg='red'))

    if warnings:
        click.echo(click.style('{} has {} violations of "{}" rule'.format(namespace_id, len(warnings), rule), fg='red'))
        for w in warnings:
            if pretty:
                click.echo(click.style('Balancer: {}'.format(w.full_id), fg='red'))
                path = ''
                for p in w.path:
                    path += '{} -> '.format(str(p))
                click.echo(click.style('Path: ' + path[: len(path) - 3], fg='red'))
                click.echo(click.style('Message: {}\n'.format(w.message), fg='red'))
            else:
                click.echo(click.style(json.dumps(w.to_json(), indent=4), fg='red'))
    else:
        click.echo(click.style('Namespace {} has no violations of "{}" rule'.format(namespace_id, rule), fg='green'))
    return sorted(warnings, key=lambda v: v.full_id.id)


@cli.command()
@click.pass_obj
@click.option('--namespace-id')
@click.option('--rule', required=True)
@click.option('--skip-large', default=False)
@click.option('--check-balancers', default=False)
@click.option('--pretty', is_flag=True)
def check(app, rule, namespace_id=None, skip_large=False, check_balancers=False, pretty=False):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    awacs = app.awacs_client
    from sepelib.core import config
    config.load()
    if namespace_id is None:
        namespace_ids = awacs.list_namespace_ids()
    else:
        namespace_ids = [namespace_id]
    warnings_by_ns_id = {}
    total_warnings_count = 0
    total_ns_w_warnings_count = 0
    ns_with_error = []
    for namespace_id in sorted(namespace_ids):
        if skip_large and namespace_id in ('s.yandex-team.ru', 'translate-internal.yandex.net'):
            print('skipped {} as large'.format(namespace_id))
            continue

        if rule == 'uem':
            balancer_pbs = awacs.list_balancers(namespace_id)
            if not balancer_pbs:
                print('skipped {} -- no balancers found'.format(namespace_id))
                continue

        try:
            ns_warnings = check_namespace(awacs, namespace_id, rule, check_balancers=check_balancers, pretty=pretty)
        except Exception as e:
            click.secho('failed to process namespace {}: {}\nTraceback:\n{}'.format(
                namespace_id, e, traceback.format_exc()), fg='white', bg='red')
            ns_with_error.append(namespace_id)
        else:
            total_warnings_count += len(ns_warnings)
            if ns_warnings:
                total_ns_w_warnings_count += 1
            warnings_by_ns_id[namespace_id] = [w.to_json() for w in ns_warnings]

    with open('./{}-warnings.json'.format(rule), 'w') as f:
        f.seek(0)
        json.dump(warnings_by_ns_id, f, indent=4, sort_keys=True)

    print('total_warnings_count', total_warnings_count)
    print('total_ns_w_warnings_count', total_ns_w_warnings_count)
    print('namespaces skipped by error:', ns_with_error)
    c = collections.Counter()
    for ns_id, ws in six.iteritems(warnings_by_ns_id):
        c[ns_id] += len(ws)
    print(c.most_common(30))

    if rule == 'uem':
        with open('./uem.json', 'w') as f:
            json.dump({
                'thresholds': dict(uem.THRESHOLDS),
                'outer_balancer2s': dict(uem.OUTER_BALANCER2S),
                'inner_balancer2s': dict(uem.INNER_BALANCER2S),
                'outer_by_name_calls': dict(uem.OUTER_BY_NAME_CALLS),
                'regexp_matchers': dict(uem.REGEXP_MATCHERS),
                'slb_ping_matchers': dict(uem.SLB_PING_MATCHERS),
                'platform_ping_matchers': dict(uem.PLATFORM_PING_MATCHERS),
            }, f)


class AbcClient(object):
    def __init__(self, token):
        self.token = token

    def abc_service_id_to_url(self, abc_service_id):
        """
        :type abc_service_id: int
        :rtype: str
        """
        url = 'https://abc-back.yandex-team.ru/api/v3/services/?id={}&fields=slug'.format(abc_service_id)
        headers = {
            'Authorization': 'OAuth {}'.format(self.token),
        }
        resp = requests.get(url, headers=headers, timeout=5)
        resp.raise_for_status()
        data = resp.json()
        if not data.get('results'):
            raise ValueError('ABC service with id "{}" does not exist'.format(abc_service_id))
        else:
            return str(data['results'][0]['slug'])

    def get_abc_role_members(self, staff_client, abc_svc_id, role):
        try:
            abc_svc_url = self.abc_service_id_to_url(abc_svc_id)
        except Exception as e:
            print(six.text_type(e))
            return set()

        resp = staff_client.list_groupmembership({
            'group.url': 'svc_{}_{}'.format(abc_svc_url.lower(), role.lower())
        })
        rv = set()
        for item in resp['result']:
            rv.add(item['person']['login'])
        return rv


def get_responsible_for_ns(awacs, staff_client, abc_client, namespace_id, affected_upstream_ids):
    """
    :type awacs: infra.awacs.tools.awacstoolslib.awacsclient.AwacsClient
    """

    def f(admins, owners):
        def is_good_login(login):
            return 'robot' not in login and login != 'keepclean'

        rv = list({login for login in admins if is_good_login(login)})
        random.seed(123)
        random.shuffle(rv)
        rv = rv[:3]
        for owner in owners:
            if owner not in rv and is_good_login(owner):
                rv.insert(0, owner)
        return rv[:5]

    namespace_pb = awacs.get_namespace(namespace_id)
    abc_svc_id = namespace_pb.meta.abc_service_id

    owners = []
    owners.extend(namespace_pb.meta.auth.staff.owners.logins)
    for upstream_id in affected_upstream_ids:
        upstream_pb = awacs.get_upstream(namespace_id, upstream_id)
        owners.extend(upstream_pb.meta.auth.staff.owners.logins)

    admins = abc_client.get_abc_role_members(staff_client, abc_svc_id, 'administration')
    if admins:
        return f(admins, owners)

    developers = abc_client.get_abc_role_members(staff_client, abc_svc_id, 'development')
    if len(developers) > 1:
        return f(developers, owners)

    if not owners:
        for group_id in namespace_pb.meta.auth.staff.owners.group_ids:
            owners.extend(staff.get_group_member_logins(staff_client, group_id))

    return f(owners, owners)


def multiline_representer(dumper, data):
    return dumper.represent_scalar(u"tag:yaml.org,2002:str", data,
                                   style="|" if "\n" in data else None)


AwacsYamlDumper.add_representer(six.text_type, multiline_representer)


@cli.command('warnings_to_startreker_task')
@click.pass_obj
@click.option('--staff-token', envvar='STAFF_TOKEN', required=True)
@click.option('--abc-token', envvar='ABC_TOKEN', required=True)
@click.option('--warnings-path', required=True)
def warnings_to_startreker_task(app, staff_token, abc_token, warnings_path):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    staff_client = staff.StaffClient.from_config({
        'api_url': 'https://staff-api.yandex-team.ru/v3/',
        'oauth_token': staff_token,
        'req_timeout': 5,
        'verify_ssl': False,
    })
    abc_client = AbcClient(abc_token)

    with open(warnings_path) as f:
        warnings_by_ns_id = json.load(f)

    issues = collections.OrderedDict()
    for ns_id, warning_jsons in sorted(six.iteritems(warnings_by_ns_id)):
        ns_warnings_by_id = collections.defaultdict(list)
        for warning_json in warning_jsons:
            w = Warning.from_json(warning_json)
            assert w.full_id.type == 'upstream'
            assert w.full_id.namespace_id == ns_id
            assert w.full_id.id not in ('slbping', 'slb_ping')  # for arl151119
            ns_warnings_by_id[w.full_id.id].append(w)

        if ns_warnings_by_id:
            tags = set()
            lines = []
            print('Warnings found in', ns_id)
            responsible = get_responsible_for_ns(app.awacs_client, staff_client, abc_client, ns_id,
                                                 list(ns_warnings_by_id.keys()))
            for id, warnings in sorted(six.iteritems(ns_warnings_by_id)):
                for w in warnings:
                    tags.update(w.tags)
                warnings_by_message = collections.defaultdict(list)
                for w in sorted(warnings, key=lambda w: six.text_type(w.path)):
                    warnings_by_message[w.message].append(w)
                lines.append(
                    u'* {{[В апстриме ((https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{namespace_id}/upstreams/list/{upstream_id}/edit/ {upstream_id})) ' \
                    u'необходимо настроить attempts rate limiter в следующих модулях balancer2:'.format(
                        namespace_id=ns_id,
                        upstream_id=id))
                assert len(warnings_by_message) == 1
                for msg, ww in sorted(six.iteritems(warnings_by_message)):
                    for w in ww:
                        lines.append(' * %%{}%%'.format(' -> '.join(w.path)))
                lines.append(u'  ]}')
            message = '\n'.join(lines)
            issues[ns_id] = collections.OrderedDict([
                ('assignee', ''),
                ('tags', sorted(tags)),
                ('invitee', sorted(responsible)),
                ('context', collections.OrderedDict([
                    ('namespace_id', ns_id),
                    ('message', message),
                ]))
            ])

    meta = {
        'queue': 'AWACSCONFIGS',
        'parent_issue_id': 'AWACSCONFIGS-465',
        'tags': [],
        'author': 'nanny-robot',
        'followers': [],
        'summary_template': u'Настроить balancer2.attempt_rate_limiter в awacs-балансере {{namespace_id}}',
        'description_template': u'''
XXX

{{message}}
        '''.strip()
    }
    yml = yaml.dump(collections.OrderedDict([
        ('meta', meta),
        ('issues', issues),
    ]), default_flow_style=False, Dumper=AwacsYamlDumper, allow_unicode=True)
    with open('./tasks.yml', 'w') as f:
        f.write(yml)
    print(yml)


if __name__ == '__main__':
    cli()
