# coding: utf-8
import traceback

import nanny_rpc_client
import six
from awacs.model.balancer.generator import validate_config
from awacs.model.balancer.vector import (Vector, BalancerVersion, UpstreamVersion,
                                         BackendVersion, EndpointSetVersion, KnobVersion, CertVersion, DomainVersion)
from awacs.lib.strutils import to_full_id
from awacs.wrappers import main  # NOQA
from awacs.wrappers.base import Holder

from infra.awacs.tools.awacstoolslib.util import clone_pb, clone_pb_dict


def get_balancer_config_holder(d, namespace_pb, namespace_id, balancer_id):
    try:
        balancer_state_pb = d.get_balancer_state(namespace_id, balancer_id)
    except nanny_rpc_client.exceptions.NotFoundError:
        return None, 'Balancer {}:{} is not found'.format(namespace_id, balancer_id)
    active_vector = balancer_state_to_active_vector(namespace_id, balancer_id, balancer_state_pb)
    if not active_vector.balancer_version:
        return None, 'No active configuration for balancer {}:{}, nothing to check'.format(namespace_id, balancer_id)
    (balancer_spec_pb,
     domain_spec_pbs,
     upstream_spec_pbs,
     backend_spec_pbs,
     endpoint_set_spec_pbs,
     knob_spec_pbs,
     cert_spec_pbs) = vector_to_specs(d, active_vector)

    try:
        holder = vector_to_config_holder(active_vector,
                                         namespace_pb,
                                         clone_pb(balancer_spec_pb),
                                         clone_pb_dict(domain_spec_pbs),
                                         clone_pb_dict(upstream_spec_pbs),
                                         clone_pb_dict(backend_spec_pbs),
                                         clone_pb_dict(endpoint_set_spec_pbs),
                                         clone_pb_dict(knob_spec_pbs),
                                         clone_pb_dict(cert_spec_pbs))
    except Exception:
        return None, traceback.format_exc()
    return holder, None


def get_upstream_config_holder(d, namespace_id, upstream_id):
    try:
        upstream_pb = d.get_upstream(namespace_id, upstream_id)
    except nanny_rpc_client.exceptions.NotFoundError:
        return None, 'Upstream {}:{} is not found'.format(namespace_id, upstream_id)

    return get_upstream_config_holder_from_pb(upstream_pb), None


def get_upstream_config_holder_from_pb(upstream_pb):
    return Holder(upstream_pb.spec.yandex_balancer.config)


def vector_to_specs(d, vector):
    """
    :type d: App
    """
    balancer_spec_pb = d.get_balancer_rev(vector.balancer_version.version)
    domain_spec_pbs = {}
    upstream_spec_pbs = {}
    backend_spec_pbs = {}
    endpoint_set_spec_pbs = {}
    knob_spec_pbs = {}
    cert_spec_pbs = {}

    for full_domain_id, domain_version in six.iteritems(vector.domain_versions):
        if domain_version.deleted:
            continue
        domain_spec_pbs[domain_version] = d.get_domain_rev(domain_version.version)

    for full_upstream_id, upstream_version in six.iteritems(vector.upstream_versions):
        if upstream_version.deleted:
            continue
        upstream_spec_pbs[upstream_version] = d.get_upstream_rev(upstream_version.version)

    for full_backend_id, backend_version in six.iteritems(vector.backend_versions):
        if backend_version.deleted:
            continue
        backend_spec_pbs[backend_version] = d.get_backend_rev(backend_version.version)

    for full_cert_id, cert_version in six.iteritems(vector.cert_versions):
        if cert_version.deleted:
            continue
        cert_spec_pbs[cert_version] = d.get_cert_rev(cert_version.version)

    for full_endpoint_set_id, endpoint_set_version in six.iteritems(vector.endpoint_set_versions):
        if endpoint_set_version.deleted:
            continue
        endpoint_set_spec_pbs[endpoint_set_version] = d.get_endpoint_set_rev(endpoint_set_version.version)

    return balancer_spec_pb, domain_spec_pbs, upstream_spec_pbs, backend_spec_pbs, endpoint_set_spec_pbs, knob_spec_pbs, cert_spec_pbs


def vector_to_config_holder(vector,
                            namespace_pb,
                            balancer_spec_pb,
                            domain_spec_pbs,
                            upstream_spec_pbs,
                            backend_spec_pbs,
                            endpoint_set_spec_pbs,
                            knob_spec_pbs,
                            cert_spec_pbs):
    return validate_config(namespace_pb=namespace_pb,
                           namespace_id=vector.balancer_version.balancer_id[0],
                           balancer_version=vector.balancer_version,
                           balancer_spec_pb=balancer_spec_pb,
                           upstream_spec_pbs=upstream_spec_pbs,
                           backend_spec_pbs=backend_spec_pbs,
                           endpoint_set_spec_pbs=endpoint_set_spec_pbs,
                           knob_spec_pbs=knob_spec_pbs,
                           cert_spec_pbs=cert_spec_pbs,
                           domain_spec_pbs=domain_spec_pbs,
                           ).balancer


def balancer_state_to_active_vector(namespace_id, balancer_id, balancer_state_pb):
    balancer_active_version = None
    upstream_active_versions = {}
    backend_active_versions = {}
    endpoint_set_active_versions = {}
    knob_active_versions = {}
    cert_active_versions = {}

    for rev_pb in balancer_state_pb.balancer.statuses:
        v = BalancerVersion.from_rev_status_pb((namespace_id, balancer_id), rev_pb)
        if rev_pb.active.status == 'True':
            balancer_active_version = v

    for maybe_full_upstream_id, upstream_state_pb in six.iteritems(balancer_state_pb.upstreams):
        upstream_full_id = to_full_id(namespace_id, maybe_full_upstream_id)
        for rev_pb in upstream_state_pb.statuses:
            v = UpstreamVersion.from_rev_status_pb(upstream_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                upstream_active_versions[upstream_full_id] = v

    for maybe_full_backend_id, backend_state_pb in six.iteritems(balancer_state_pb.backends):
        backend_full_id = to_full_id(namespace_id, maybe_full_backend_id)
        for rev_pb in backend_state_pb.statuses:
            v = BackendVersion.from_rev_status_pb(backend_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                backend_active_versions[backend_full_id] = v

    for maybe_full_endpoint_set_id, endpoint_set_state_pb in six.iteritems(balancer_state_pb.endpoint_sets):
        for rev_pb in endpoint_set_state_pb.statuses:
            endpoint_set_full_id = to_full_id(namespace_id, maybe_full_endpoint_set_id)
            v = EndpointSetVersion.from_rev_status_pb(endpoint_set_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                endpoint_set_active_versions[endpoint_set_full_id] = v

    for knob_id, knob_state_pb in six.iteritems(balancer_state_pb.knobs):
        for rev_pb in knob_state_pb.statuses:
            knob_full_id = to_full_id(namespace_id, knob_id)
            v = KnobVersion.from_rev_status_pb(knob_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                knob_active_versions[knob_full_id] = v

    for cert_id, cert_state_pb in six.iteritems(balancer_state_pb.certificates):
        for rev_pb in cert_state_pb.statuses:
            cert_full_id = to_full_id(namespace_id, cert_id)
            v = CertVersion.from_rev_status_pb(cert_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                cert_active_versions[cert_full_id] = v

    domain_active_versions = {}
    for domain_id, domain_state_pb in six.iteritems(balancer_state_pb.domains):
        for rev_pb in domain_state_pb.statuses:
            domain_full_id = to_full_id(namespace_id, domain_id)
            v = DomainVersion.from_rev_status_pb(domain_full_id, rev_pb)
            if rev_pb.active.status == 'True':
                domain_active_versions[domain_full_id] = v

    active_vector = Vector(balancer_active_version,
                           upstream_active_versions,
                           domain_active_versions,
                           backend_active_versions,
                           endpoint_set_active_versions,
                           knob_active_versions,
                           cert_active_versions)

    return active_vector
