# coding: utf-8
from __future__ import print_function

import collections
import json
import pickle
import pprint
import random
import re
import string
import sys
from datetime import datetime, timedelta

import click
import google.protobuf.field_mask_pb2
import nanny_rpc_client
import pytz
import requests
import six
from awacs import yamlparser
from infra.awacs.proto import modules_pb2

from infra.awacs.tools.awacsls import awacsmap
from infra.awacs.tools.awacstoolslib.listers import (list_namespace_ids_w_sd, list_namespace_ids_w_yp_es,
                                                     list_namespace_ids_with_large_rps)
from infra.awacs.tools.awacstoolslib.util import cli, create_awacs_backend_href, create_awacs_upstream_href


ABC_API_URL = 'https://abc-back.yandex-team.ru/api/v3/'
ABC_API_URL_V4 = 'https://abc-back.yandex-team.ru/api/v4/'
STAFF_API_URL = 'https://staff-api.yandex-team.ru/v3/'


class ApiClient:
    def __init__(self, token):
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(token)


class AbcClient(ApiClient):
    def __init__(self, abc_token):
        ApiClient.__init__(self, abc_token)

    def get_abc_service(self, abc_service_id):
        url = ABC_API_URL + 'services/{}/'.format(abc_service_id)
        resp = self.session.get(url)
        resp.raise_for_status()
        return resp.json()

    def get_members(self, abc_service_id, params=None):
        # TODO
        # url = ABC_API_URL_V4 + 'services/responsibles/?service={}'.format(abc_service_id)
        # url = ABC_API_URL_V4 + 'roles/?service={}'.format(abc_service_id)

        url = ABC_API_URL_V4 + 'services/members/'

        ps = {'service': abc_service_id}
        if params is not None:
            ps.update(params)

        resp = self.session.get(url, params=ps)
        resp.raise_for_status()
        return resp.json()


class StaffClient(ApiClient):
    def __init__(self, token):
        ApiClient.__init__(self, token)

    def get_person(self, login, params=None):
        # https://staff-api.yandex-team.ru/v3/persons?login=robot-srch-releaser&_fields=official.is_robot
        url = STAFF_API_URL + 'persons/'

        ps = {'login': login}
        if params is not None:
            ps.update(params)

        resp = self.session.get(url, params=ps)
        resp.raise_for_status()
        return resp.json()


def get_label(labels, key, default=None):
    for a in labels.attributes:
        if a.key == key:
            return a.value
    return default


def get_disk_request_by_mount_path(disk_volume_requests, mount_path):
    for r in disk_volume_requests:
        if get_label(r.labels, key='mount_path') == mount_path:
            return r


@cli.command('ls_pods')
@click.pass_obj
@click.option('--cluster', required=True, type=click.Choice(('SAS', 'MAN', 'VLA', 'MYT', 'IVA')))
def ls_pods(app, cluster):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    op = app.op
    awacs_client = app.awacs_client
    cluster = cluster.upper()

    stats = collections.Counter()

    def get_total():
        rv = 0
        for k, v in stats.items():
            rv += k * v
        return rv / 1000

    if op and op.data.get('nanny_service_ids', {}).get(cluster):
        nanny_service_ids = op.data['nanny_service_ids'][cluster]
        click.echo('Got {} Nanny service ids from opcache'.format(cluster))
    else:
        nanny_service_ids = []
        click.echo('Reading {} Nanny service ids'.format(cluster))
        for balancer_pb in awacs_client.iter_all_balancers(skip_incomplete=True, yp_cluster_in=(cluster,)):
            nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
            nanny_service_ids.append(nanny_service_id)
        if op:
            op.data.setdefault('nanny_service_ids', {})[cluster] = nanny_service_ids
            op.save()
        click.echo('Completed reading {} Nanny service ids'.format(cluster))

    n = len(nanny_service_ids)
    if op:
        pods_by_service = op.data.get('pods_by_service', {})
    else:
        pods_by_service = {}
    for i, nanny_service_id in enumerate(nanny_service_ids):
        field_mask_pb = google.protobuf.field_mask_pb2.FieldMask()
        field_mask_pb.paths.append('meta.id')
        field_mask_pb.paths.append('spec.resource_requests')
        field_mask_pb.paths.append('spec.disk_volume_requests')
        if nanny_service_id not in pods_by_service:
            pods_by_service[nanny_service_id] = {}
            for pod_pb in app.nanny_client.iter_pods(cluster, nanny_service_id, field_mask_pb=field_mask_pb):
                has_awacs_volume = get_disk_request_by_mount_path(
                    pod_pb.spec.disk_volume_requests, '/awacs') is not None
                pods_by_service[nanny_service_id][pod_pb.meta.id] = {
                    'vcpu_guarantee': int(pod_pb.spec.resource_requests.vcpu_guarantee),
                    'has_awacs_volume': has_awacs_volume,
                }
        for pod in pods_by_service[nanny_service_id].values():
            pod_cpu = pod['vcpu_guarantee']
            stats[pod_cpu] += 1
        if i % 10 == 0:
            click.echo(json.dumps(stats, indent=2, sort_keys=True))
            click.echo('Total: {}, processed {}/{}'.format(get_total(), i + 1, n))
            if op:
                op.data['pods_by_service'] = pods_by_service
                op.save()

    click.echo(json.dumps(stats, indent=2, sort_keys=True))
    click.echo('Total: {}'.format(get_total()))
    click.echo()
    total_pods = 0
    total_pods_w_awacs_vol = 0
    total_services = 0
    services_wo_awacs_vol = set()
    total_services_w_awacs_vol = 0
    for service_id, pods in sorted(pods_by_service.items()):
        all_pods_have_awacs_vol = True
        for pod_id, pod in pods.items():
            # click.echo(pod_id)
            total_pods += 1
            if pod['has_awacs_volume']:
                total_pods_w_awacs_vol += 1
            else:
                all_pods_have_awacs_vol = False
        total_services += 1
        if all_pods_have_awacs_vol:
            total_services_w_awacs_vol += 1
        else:
            services_wo_awacs_vol.add(service_id)

    click.echo('total_pods_w_awacs_vol: {}, total_pods: {}, {}%'.format(
        total_pods_w_awacs_vol, total_pods,
        float(total_pods_w_awacs_vol) / total_pods * 100))
    click.echo('total_services_w_awacs_vol: {}, total_services: {}, {}%'.format(
        total_services_w_awacs_vol, total_services,
        float(total_services_w_awacs_vol) / total_services * 100))
    click.echo('services_wo_awacs_vol')
    for s_id in services_wo_awacs_vol:
        click.echo('* ' + s_id)


@cli.command()
@click.pass_obj
@click.option('--condition', required=True)
@click.option('--abc-token', envvar='ABC_TOKEN')
@click.option('--staff-token', envvar='STAFF_TOKEN')
def ls(app, condition, abc_token=None, staff_token=None):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    awacs_client = app.awacs_client
    if condition == 'HL':
        abcs = {}
        total_abc_rpss = collections.defaultdict(int)
        ns_ids = set()
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        for ns_id in sorted(list_namespace_ids_with_large_rps(app, 500)):
            ns_ids.add(ns_id)
            ns_pb = awacs_client.get_namespace(ns_id)
            abcs[ns_id] = ns_pb.meta.abc_service_id
            total_abc_rpss[ns_pb.meta.abc_service_id] += rps_data[ns_id]

        for ns_id in reversed(sorted(ns_ids, key=rps_data.get)):
            print('{} {} {}'.format(string.ljust(ns_id, 50),
                                    string.ljust(str(int(rps_data[ns_id])) + ' RPS', 20),
                                    'https://abc.yandex-team.ru/services/{}'.format(abcs[ns_id])))
    elif condition == 'X_YANDEX_HTTPS':
        # awacsmap.Processor(awacs_client, app.l3mgr_client, app.nanny_client).process() #pickle_path='./awacsmap-tmp.pickle')
        awacsmap.Processor(awacs_client, app.l3mgr_client, app.nanny_client).process_pickled()
        # awacsmap.Processor(awacs_client, app.l3mgr_client, app.nanny_client).process()
    elif condition == 'SWATOPS_270_CANDIDATES':
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        blacklisted_infixes = ('taxi', 'eda', 'lavka')
        threshold_rps = 500

        ns_ids = set()
        for ns_id in sorted(list_namespace_ids_w_yp_es(app)):
            if any([infix in ns_id for infix in blacklisted_infixes]):
                continue
            if ns_id.endswith('maps.yandex.net') and not ns_id.startswith('front-'):
                continue
            if ns_id not in rps_data:
                continue
            if int(rps_data[ns_id]) > threshold_rps:
                continue
            ns_ids.add(ns_id)
        for ns_id in sorted(ns_ids):
            print(ns_id)
        print('#', len(ns_ids))
    elif condition == 'NAMESPACES_WITH_SD':
        for ns_id in sorted(list_namespace_ids_w_sd(app)):
            click.echo(ns_id)
    elif condition == 'SHARED_L7':
        def grep(t, what):
            rv = []
            for line in t.splitlines():
                if what in line:
                    rv.append(line)
            return '\n'.join(rv)

        def indent(t, shift=2):
            rv = []
            for line in t.splitlines():
                rv.append(' ' * shift + line)
            return '\n'.join(rv)

        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        for ns_id in list_namespace_ids_with_large_rps(app, 500):
            total = 0
            l3_balancer_ids = set()
            for l3_balancer_pb in awacs_client.list_l3_balancers(ns_id):
                l3_balancer_ids.add(l3_balancer_pb.meta.id)
                total += 1
            if total > 1:
                print('{} {} L3 balancers, {} RPS'.format(string.ljust(ns_id, 50), total,
                                                          int(rps_data[ns_id])))
                print('L3 balancer ids: ' + ' '.join(l3_balancer_ids))
                for balancer_pb in awacs_client.list_balancers(ns_id):
                    if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.EASY_MODE:
                        print()
                        print('* L7', balancer_pb.meta.id)
                        print(indent(grep(balancer_pb.spec.yandex_balancer.yaml, 'ips'), 4))
    elif condition == 'NOT_EASY_MODE':
        suffixes = {'maps.yandex.net'}
        for s in ('eda', 'lavka', 'taxi'):
            suffixes.add('{}.yandex.net'.format(s))
            suffixes.add('{}.tst.yandex.net'.format(s))
        suffixes = tuple(suffixes)

        abcs = {}
        ns_ids = set()
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        for ns_id in sorted(list_namespace_ids_with_large_rps(app, 500)):
            if ns_id.endswith(suffixes):
                continue
            easy_mode = True
            for balancer_pb in awacs_client.list_balancers(ns_id):
                if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.EASY_MODE:
                    easy_mode = False
                    break
            if easy_mode:
                for upstream_pb in awacs_client.list_upstreams(ns_id):
                    if upstream_pb.spec.yandex_balancer.mode != upstream_pb.spec.yandex_balancer.EASY_MODE2:
                        easy_mode = False
                        break
            if not easy_mode:
                ns_ids.add(ns_id)
                ns_pb = awacs_client.get_namespace(ns_id)
                abcs[ns_id] = ns_pb.meta.abc_service_id

        for ns_id in reversed(sorted(ns_ids, key=rps_data.get)):
            print('{} {} {}'.format(string.ljust(ns_id, 50),
                                    string.ljust(str(int(rps_data[ns_id])) + ' RPS', 20),
                                    'https://abc.yandex-team.ru/services/{}'.format(abcs[ns_id])))
    elif condition == 'HL_NOT_TOP_LEVEL_EASY_MODE':
        filters_blacklist = {'id_prefix_in', 'id_prefix', 'id_suffix', 'not:', 'and:', 'or:'}
        # https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/turbopages_service_balancer/show/
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        for ns_id in sorted(list_namespace_ids_with_large_rps(app, 500)):
            has_rps_limiter = True
            for balancer_pb in awacs_client.list_balancers(ns_id):
                yaml = balancer_pb.spec.yandex_balancer.yaml
                is_easy_mode = balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.EASY_MODE
                if not is_easy_mode and any(f in yaml for f in filters_blacklist):
                    has_rps_limiter = False
                    break
            if not has_rps_limiter:
                print('{} {} RPS'.format(string.ljust(ns_id, 50),
                                         int(rps_data[ns_id])))
    elif condition == 'NAMESPACES_WITH_YP_ES':
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()
        blacklisted_infixes = ('taxi', 'eda', 'lavka')
        for ns_id in sorted(list_namespace_ids_w_yp_es(app)):
            if not any([x in ns_id for x in blacklisted_infixes]):
                continue
            if 'tst' not in ns_id:
                continue
            if int(rps_data.get(ns_id, -1)) < 100:
                click.echo(ns_id)
    elif condition == 'NAMESPACES_WITH_DIFFERENT_L7':
        import difflib
        from colorama import Fore, Back

        def color_diff(diff):
            for line in diff:
                if line.startswith('+'):
                    yield Fore.GREEN + line + Fore.RESET
                elif line.startswith('-'):
                    yield Fore.RED + line + Fore.RESET
                elif line.startswith('^'):
                    yield Fore.BLUE + line + Fore.RESET
                else:
                    yield line

        def get_diff(a, b):
            a = a.splitlines(1)
            b = b.splitlines(1)
            diff = difflib.unified_diff(a, b, n=10)
            diff = color_diff(diff)
            return ''.join(diff)

        for ns_pb in awacs_client.iter_all_namespaces():
            namespace_id = ns_pb.meta.id
            bs_pbs = []
            for bs_pb in awacs_client.list_balancers(namespace_id=namespace_id):
                bs_pbs.append(bs_pb)

            if len(bs_pbs) == 0:
                continue

            def find_eq_group(ind):
                inds = set()
                for i in range(len(bs_pbs)):
                    if bs_pbs[ind].spec.yandex_balancer.config == bs_pbs[i].spec.yandex_balancer.config:
                        inds.add(i)
                return inds

            eq_groups = []

            group = set()
            rest = set(range(len(bs_pbs)))

            while True:
                rest -= group
                if len(rest) == 0:
                    break
                group = find_eq_group(rest.pop())
                eq_groups.append(group)

            if len(eq_groups) == 1:
                continue

            eq_groups.sort(key=lambda group: len(group))

            print('namespace https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}'.format(namespace_id))

            for i, group in enumerate(eq_groups):
                group = sorted(group)
                print('group {}:'.format(i+1))
                for j in group:
                    balancer_id = bs_pbs[j].meta.id
                    url_template = 'https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/balancers/list/{}/show/'
                    print(url_template.format(namespace_id, balancer_id))

            for i in range(1, len(eq_groups)):
                print('group {} - group {} diff:'.format(i, i+1))
                a = bs_pbs[list(eq_groups[i-1])[0]].spec.yandex_balancer.yaml
                b = bs_pbs[list(eq_groups[i])[0]].spec.yandex_balancer.yaml
                print(get_diff(a, b))

            print('')

    elif condition == 'L7_MACRO_VERSIONS_W_IDS':
        def get_id(b_pb):
            return b_pb.meta.namespace_id + '/' + b_pb.meta.id

        c = collections.defaultdict(list)
        for balancer_pb in awacs_client.iter_all_balancers():
            if not balancer_pb.spec.yandex_balancer.config.HasField('l7_macro'):
                c['NOT_L7_MACRO'].append(get_id(balancer_pb))
                continue
            version = balancer_pb.spec.yandex_balancer.config.l7_macro.version or '0.0.1'
            c[version].append(get_id(balancer_pb))
        for k, vs in sorted(c.items()):
            if k in ('0.1.1', 'NOT_L7_MACRO'):
                continue
            print(k)
            for v in vs:
                print('-', v)
    elif condition == 'INACTIVE_L3':
        total = 0
        for l3_balancer_pb in awacs_client.iter_all_l3_balancers():
            if l3_balancer_pb.spec.incomplete:
                continue
            if l3_balancer_pb.l3_status.active.status != 'True':
                print(l3_balancer_pb.meta.namespace_id, l3_balancer_pb.meta.id)
            total += 1
        print('Total L3', total)
    elif condition == 'L3MGR_RS':
        for l3_balancer_pb in awacs_client.iter_all_l3_balancers():
            if l3_balancer_pb.spec.incomplete:
                continue
            print(app.l3mgr_client.get_service(l3_balancer_pb.spec.l3mgr_service_id))
    elif condition == 'CERT_EXPIRATIONS':
        today = datetime.utcnow().date()
        month = timedelta(days=30)
        for cert_pb in awacs_client.iter_all_certs():
            if cert_pb.spec.incomplete:
                continue
            valid_for = cert_pb.spec.fields.validity.not_after.ToDatetime().date() - today
            fg = 'green'
            if valid_for <= month:
                fg = 'red'
            click.echo(click.style('{:<55} {:<55} {}'.format(cert_pb.meta.namespace_id, cert_pb.meta.id,
                                                             valid_for), fg=fg))
    elif condition == 'L7_MACRO_COMMENTS':
        for balancer_pb in awacs_client.iter_all_balancers():
            spec_pb = balancer_pb.spec
            if not spec_pb.yandex_balancer.config.HasField('l7_macro'):
                continue
            if '#' in spec_pb.yandex_balancer.yaml:
                print(balancer_pb.meta.namespace_id, balancer_pb.meta.id, 'yaml probably contains comments')
    elif condition == 'MATCHER_MAP':
        for upstream_pb in awacs_client.iter_all_upstreams():
            spec_pb = upstream_pb.spec
            if 'matcher_map' in spec_pb.yandex_balancer.yaml:
                print(upstream_pb.meta.namespace_id, upstream_pb.meta.id)
    elif condition == 'GENCFG':
        rv = set()
        for aspects_set_pb in awacs_client.iter_all_balancer_aspects_sets():
            ns_id = aspects_set_pb.meta.namespace_id
            content_pb = aspects_set_pb.content.cluster.content
            if not content_pb.active_conf_id:
                continue
            if content_pb.platform == content_pb.GENCFG:
                rv.add(ns_id)
        print('\n'.join(sorted(rv)))
    elif condition == 'BALANCER_TAGS':
        balancer_pbs = awacs_client.iter_all_balancers()
        for balancer_pb in balancer_pbs:
            nanny_service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
            if not nanny_service_id:
                continue
            tags_pb = balancer_pb.spec.config_transport.nanny_static_file.instance_tags
            if not tags_pb.prj:
                continue
            print(balancer_pb.meta.namespace_id, balancer_pb.meta.id, tags_pb.itype, tags_pb.ctype, tags_pb.prj)
    elif condition == 'RPS_FROM_YASM':
        from infra.yasm.yasmapi import GolovanRequest
        import time
        import os.path

        HOST = 'ASEARCH'
        period = 5

        # result 1
        # period = 5
        # pkl_path = './rps-from-yasm.pkl'
        # dt = datetime.strptime('2020-09-18 18:00:00', '%Y-%m-%d %H:%M:%S')
        # st = int(time.mktime(dt.timetuple()))
        # TOTAL 6712910.0453

        # result 2
        # period = 5
        # pkl_path = './rps-from-yasm-2.pkl'
        # dt = datetime.strptime('2020-03-20 18:00:00', '%Y-%m-%d %H:%M:%S')
        # st = int(time.mktime(dt.timetuple()))
        # et = st + 900
        # TOTAL 5712892.57459

        # period = 3600
        # pkl_path = './rps-from-yasm-3.pkl'
        # dt = datetime.strptime('2020-03-20 18:00:00', '%Y-%m-%d %H:%M:%S')
        # st = int(time.mktime(dt.timetuple()))
        # et = st + 3600 - 1
        # TOTAL 4298163.25231

        # period = 3600
        # pkl_path = './rps-from-yasm-4.pkl'
        # dt = datetime.strptime('2020-09-18 18:00:00', '%Y-%m-%d %H:%M:%S')
        # st = int(time.mktime(dt.timetuple()))
        # et = st + 3600 - 1
        # TOTAL 6924063.98333

        # period = 3600
        # pkl_path = './rps-from-yasm-5.pkl'
        # dt = datetime.strptime('2020-09-25 18:00:00', '%Y-%m-%d %H:%M:%S')
        # st = int(time.mktime(dt.timetuple()))
        # et = st + 3600 - 1
        # TOTAL 6600329.67889

        period = 3600
        pkl_path = './rps-from-yasm-6.pkl'
        dt = datetime.strptime('2020-09-23 18:00:00', '%Y-%m-%d %H:%M:%S')
        st = int(time.mktime(dt.timetuple()))
        et = st + 3600 - 1
        # TOTAL 6481321

        period = 3600
        pkl_path = './rps-from-yasm-9.pkl'
        dt = datetime.strptime('2021-03-12 18:00:00', '%Y-%m-%d %H:%M:%S')
        st = int(time.mktime(dt.timetuple()))
        et = st + 3600 - 1
        # TOTAL 9533950, 9.5kk

        period = 3600
        pkl_path = './rps-from-yasm-10.pkl'
        dt = datetime.strptime('2021-03-26 18:00:00', '%Y-%m-%d %H:%M:%S')
        st = int(time.mktime(dt.timetuple()))
        et = st + 3600 - 1
        # TOTAL 9215181, 9.2kk

        period = 3600
        pkl_path = './rps-from-yasm-11.pkl'  # /with new tags
        dt = datetime.strptime('2021-03-26 18:00:00', '%Y-%m-%d %H:%M:%S')
        st = int(time.mktime(dt.timetuple()))
        et = st + 3600 - 1
        # TOTAL 9214518.19306

        uniq_tagsets = set()
        with open('./tags-310321.txt') as f:
            for line in f.readlines():
                n_id, b_id, itype, ctype, prj = line.strip().split()
                uniq_tagsets.add((n_id, itype, ctype, prj))

        def get_total(itype, ctype, prj):
            s = 'itype={};ctype={};prj={}:{}'.format(itype, ctype, prj,
                                                     'balancer_report-report-service_total-succ_summ')
            signals = [s]
            exc = None
            for attempt in (1, 2, 3, 4):
                try:
                    raw_data = list(GolovanRequest(HOST, period, st, et, signals))
                except Exception as exc:
                    time.sleep(2 ** attempt)
            if exc is not None:
                raise
            time.sleep(random.randint(0, 1))

            xs = []
            min_ts = sys.maxsize
            max_ts = -sys.maxsize
            for ts, values in raw_data:
                min_ts = min(ts, min_ts)
                max_ts = max(ts, max_ts)
                xs.append(values[s])
            xs_max = max(xs)
            if xs_max is None:
                avg_rps = 0
            else:
                avg_rps = sum(xs) / float(max_ts - min_ts + period) if xs else 0
            return avg_rps

        init_i = 0
        rv = collections.defaultdict(int)
        if os.path.exists(pkl_path):
            with open(pkl_path, 'r') as f:
                init_i, init_rv = pickle.load(f)
            rv.update(init_rv)
        total = sum(rv.values())

        for i, (n_id, itype, ctype, prj) in enumerate(sorted(uniq_tagsets)):
            if i <= init_i:
                continue
            x = get_total(itype, ctype, prj)
            print(i, n_id, int(x * 100) / 100.)
            rv[n_id] += x
            total += x
            if i % 10 == 0:
                print('total', total)
                with open(pkl_path, 'wb') as f:
                    pickle.dump((i, rv), f)
        print('TOTAL', total)
    elif condition == 'L3':
        total = 0
        for l3_balancer_pb in awacs_client.iter_all_l3_balancers():
            total += 1
        print('Total L3', total)
    elif condition == 'CERT_STORAGES':
        c = collections.Counter()
        for cert_pb in awacs_client.iter_all_certs():
            if cert_pb.spec.storage.type == cert_pb.spec.storage.NANNY_VAULT:
                c['NANNY_VAULT'] += 1
            else:
                c['YAV'] += 1
            print('.')
        print('count', c)
        print('total', sum(c.values()))
    elif condition == 'CERT_EXTERNAL':
        c = 0
        namespace_ids = set()
        full_cert_ids = set()
        for cert_pb in awacs_client.iter_all_certs():
            ca_name = cert_pb.spec.certificator.ca_name
            if ca_name != 'InternalCA' and 'Russian Federation' in cert_pb.spec.fields.subject:
                full_cert_ids.add((cert_pb.meta.namespace_id, cert_pb.meta.id))
                namespace_ids.add(cert_pb.meta.namespace_id)
                c += 1
        for ns_id, c_id in sorted(full_cert_ids):
            print(ns_id, c_id)
    elif condition == 'W_ALERTS':
        c = 0
        t = 0
        x = 0
        nottaxi = 0
        for ns_pb in awacs_client.iter_all_namespaces():
            date = ns_pb.order.status.last_transition_time.ToDatetime().date()
            if date.year == 2020 and date.month >= 3:
                x += 1
                ns_id = ns_pb.meta.id
                nottaxi += not ('taxi' in ns_id)

            t += 1
            if ns_pb.spec.HasField('alerting'):
                c += 1
            else:
                date = ns_pb.order.status.last_transition_time.ToDatetime().date()
                if date.year == 2020 and date.month >= 3:
                    print(ns_pb.meta.id, date)
        print('c', c)
        print('total', t)
        print('x', x)
        print('nottaxi', nottaxi)
    elif condition == 'WO_ALERTS':
        rps_data = awacs_client.get_yesterday_max_rps_stats_by_namespace()

        c = collections.Counter()
        for ns_pb in awacs_client.iter_all_namespaces():
            ns_id = ns_pb.meta.id
            if any(skip_id in ns_id for skip_id in ('taxi', 'maps.yandex.net', 'tst', 'test')):
                continue
            if ns_pb.spec.HasField('alerting'):
                continue
            b_pbs = awacs_client.list_balancers(ns_id)
            if not b_pbs:
                continue
            lbl = ''
            if all(b_pb.spec.yandex_balancer.mode == b_pb.spec.yandex_balancer.QUICK_START_MODE for b_pb in b_pbs):
                lbl = 'QS'
            c[(ns_id, lbl)] = int(rps_data.get(ns_id, -1))
        for k, v in c.most_common(100):
            print(k[0], v, k[1])
        print('total', len(c))

    elif condition == 'SWAT-6777':
        matcher = re.compile('^(.+[.])?yandex[.](' + '|'.join([
            str('az'), str('by'), str('co.il'), str('com'), str('com.am'),
            str('com.ge'), str('com.tr'), str('ee'), str('eu'), str('fi'),
            str('fr'), str('kg'), str('kz'), str('lt'), str('lv'),
            str('md'), str('pl'), str('ru'), str('tj'), str('tm'),
            str('ua'), str('uz'),
        ]) + ')$')

        def get_names(cert_pb):
            rv = set(cert_pb.spec.fields.subject_alternative_names)
            rv.add(cert_pb.spec.fields.subject_common_name)
            return rv

        def match(names):
            for name in names:
                if matcher.match(name):
                    return True
            return False

        for cert_pb in awacs_client.iter_all_certs():
            names = get_names(cert_pb)
            if match(names):
                print(cert_pb.meta.namespace_id, '// names:', ', '.join(sorted(names)))
    elif condition == 'SWATOPS-92':
        for backend_pb in awacs_client.iter_all_backends():
            for pb in backend_pb.spec.selector.nanny_snapshots:
                if pb.snapshot_id:
                    print(backend_pb.meta)
                    raise Exception('wow!')
    elif condition == 'UPSTREAM_LINES':
        upstream_pbs = awacs_client.iter_all_upstreams()
        rv = collections.defaultdict(set)
        for upstream_pb in upstream_pbs:
            yml = upstream_pb.spec.yandex_balancer.yaml
            n = len(yml.splitlines())
            id_ = (upstream_pb.meta.namespace_id, upstream_pb.meta.id)
            rv[n].add(id_)
            rv = max(rv, n)
        for n, ids in sorted(rv.items())[-100:]:
            print(n, ids)
    elif condition == 'BALANCER_LINES':
        balancer_pbs = awacs_client.iter_all_balancers()
        rv = collections.defaultdict(set)
        for balancer_pb in balancer_pbs:
            yml = balancer_pb.spec.yandex_balancer.yaml
            n = len(yml)
            id_ = (balancer_pb.meta.namespace_id, balancer_pb.meta.id)
            rv[n].add(id_)
            rv = max(rv, n)
        for n, ids in sorted(rv.items())[-100:]:
            print(n, ids)
    elif condition == 'TO_BE_CLEANED_UP':
        with open(u'./list.txt') as f:
            ns_ids = [line.strip().split()[-1] for line in f.readlines()]

        dirty_ns_ids = set()
        for ns_id in ns_ids:
            balancer_pbs = awacs_client.list_balancers(ns_id)
            for balancer_pb in balancer_pbs:
                if balancer_pb.spec.yandex_balancer.mode != balancer_pb.spec.yandex_balancer.FULL_MODE:
                    continue
                yml = balancer_pb.spec.yandex_balancer.yaml
                m = re.search('.*port.*', yml)
                if 'get_port_var' not in balancer_pb.spec.yandex_balancer.yaml:
                    if m:
                        print(m.group(0))
                    dirty_ns_ids.add(ns_id)
        for ns_id in sorted(dirty_ns_ids):
            print('*', ns_id)
    elif condition == 'ALL':
        for ns_id in awacs_client.list_namespace_ids():
            print(ns_id)
    elif condition == 'ALL_WITH_RPS_LIMITER':
        blacklisted_ns_ids = {'tap-testing', 's.yandex-team.ru'}
        ns_ids = list(awacs_client.list_namespace_ids())
        for i, ns_id in enumerate(ns_ids):
            if ns_id in blacklisted_ns_ids:
                continue
            has_rps_limiter = False
            for balancer_pb in awacs_client.list_balancers(ns_id):
                yml = balancer_pb.spec.yandex_balancer.yaml
                if 'rps_limiter' in yml:
                    has_rps_limiter = True
                    break
            if not has_rps_limiter:
                for upstream_pb in awacs_client.list_upstreams(ns_id):
                    yml = upstream_pb.spec.yandex_balancer.yaml
                    if 'rps_limiter' in yml:
                        has_rps_limiter = True
                        break
            if not has_rps_limiter:
                print(i, '/', len(ns_ids), ns_id, file=sys.stderr)
                continue
            print('XXX', ns_id)
    elif condition == 'ALL_WITH_L7':
        for ns_id in awacs_client.list_namespace_ids():
            has_l7 = False
            for _ in awacs_client.list_balancers(ns_id):
                has_l7 = True
            if has_l7:
                print(ns_id)
    elif condition == 'SWAT-6507':
        is_quick_start = lambda b_pb: b_pb.spec.yandex_balancer.config.HasField('quick_start_balancer_macro')
        is_easy = lambda b_pb: b_pb.spec.yandex_balancer.config.HasField('l7_macro')
        for balancer_pb in awacs_client.iter_all_balancers():
            if balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.EASY_MODE:
                mode = "EASY"
            elif balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.QUICK_START_MODE:
                mode = 'QUICK_START'
            elif balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.FULL_MODE:
                mode = 'FULL'
            else:
                raise AssertionError
            if is_quick_start(balancer_pb):
                if mode != 'QUICK_START':
                    print('{}:{} starts with quick_start_balancer_macro, '
                          'mode: {}'.format(balancer_pb.meta.namespace_id, balancer_pb.meta.id, mode))
            if is_easy(balancer_pb):
                if mode != 'EASY':
                    print('{}:{} starts with l7_macro, '
                          'mode: {}'.format(balancer_pb.meta.namespace_id, balancer_pb.meta.id, mode))
    elif condition == 'SLBPING':
        rv = {
            'slb_ping_macro.errordoc': 0,
            'slb_ping_macro.use_shared_backends': 0,
            'slb_ping_macro.gbp': 0,
            'balancer2': 0,
            'errordocument': 0,
        }
        for upstream_pb in awacs_client.iter_all_upstreams():
            id_ = upstream_pb.meta.id
            if id_ not in ('slbping', 'slb_ping'):
                continue
            yml = upstream_pb.spec.yandex_balancer.yaml
            if 'slb_ping_macro' in yml:
                if 'errordoc: true' in yml:
                    rv['slb_ping_macro.errordoc'] += 1
                elif 'use_shared_backends: true' in yml:
                    rv['slb_ping_macro.use_shared_backends'] += 1
                elif 'include_backends' in yml or 'instances' in yml:
                    rv['slb_ping_macro.gbp'] += 1
                else:
                    raise AssertionError()
            elif 'balancer2' in yml:
                rv['balancer2'] += 1
            elif 'errordocument' in yml:
                rv['errordocument'] += 1
            else:
                print(yml)
                raise AssertionError()
            pprint.pprint(rv)

    elif condition == 'UNIMPORTED_CERTS':
        nanny = app.nanny_client

        NANNY_SECRET_TYPE = 'SECRET'
        YAV_SECRET_TYPE = 'VAULT_SECRET'

        index = {}
        namespace_ids = list(awacs_client.list_namespace_ids())
        for i, ns_id in enumerate(namespace_ids, start=1):
            index[ns_id] = set()
            try:
                cert_pbs = awacs_client.list_certs(ns_id)
                balancer_pbs = awacs_client.list_balancers(ns_id)

                cert_storage_texts = set()
                for pb in cert_pbs:
                    cert_storage_texts.add(six.text_type(pb.spec.storage))
                first_unimported_secret = ''
                secrets_found = False
                index[ns_id].update(cert_storage_texts)
                for pb in balancer_pbs:
                    service_id = pb.spec.config_transport.nanny_static_file.service_id
                    if not service_id:
                        # not ready
                        continue
                    service_runtime_attrs_data = nanny.get_service_runtime_attrs(service_id)
                    volumes = service_runtime_attrs_data['content']['instance_spec']['volume']
                    for volume in volumes:
                        name = volume['name']
                        if not name.startswith('secrets'):
                            continue
                        if volume['type'] == NANNY_SECRET_TYPE:
                            secrets_found = True
                            data = volume.get('secretVolume')
                            secret_id = data['keychainSecret']['secretId']
                            keychain_id = data['keychainSecret']['keychainId']
                            if (not cert_storage_texts or
                                not any(secret_id in t for t in cert_storage_texts) or
                                not any(keychain_id in t for t in cert_storage_texts)):
                                if not first_unimported_secret:
                                    first_unimported_secret = name
                            index[ns_id].add((keychain_id, secret_id))
                        elif volume['type'] == YAV_SECRET_TYPE:
                            secrets_found = True
                            data = volume.get('vaultSecretVolume')
                            secret_id = data['vaultSecret']['secretId']
                            if not cert_storage_texts or not any(secret_id in t for t in cert_storage_texts):
                                if not first_unimported_secret:
                                    first_unimported_secret = name
                            index[ns_id].add((secret_id,))
                        else:
                            raise AssertionError()
                if first_unimported_secret:
                    print('UNIMPORTED', first_unimported_secret, ns_id)
                else:
                    if not secrets_found:
                        print('OK NO_SECRETS', ns_id)
                    else:
                        print('OK IMPORTED', ns_id)
            except Exception as e:
                print('FAILED_TO_PROCESS', e, ns_id)
            finally:
                index[ns_id] = sorted(index[ns_id])
                sys.stderr.write('Processed {}, {} of {}\n'.format(ns_id, i, len(namespace_ids)))

        with open('./certs-index.json', 'w') as f:
            json.dump(index, f)

    elif condition == 'SWAT-6425':
        for backend_pb in awacs_client.iter_all_backends():
            if backend_pb.spec.selector.type != backend_pb.spec.selector.YP_ENDPOINT_SETS:
                continue
            backend_rev_pbs = list(awacs_client.list_backend_revs(namespace_id=backend_pb.meta.namespace_id,
                                                                  backend_id=backend_pb.meta.id))
            types = [rev_pb.spec.selector.type for rev_pb in backend_rev_pbs]
            if len(types) > 1 and backend_pb.spec.selector.YP_ENDPOINT_SETS_SD in types:
                for backend_rev_pb in backend_rev_pbs:
                    comment = backend_rev_pb.meta.comment
                    if u'awacsctl' in comment and backend_rev_pb.spec.selector.type == backend_rev_pb.spec.selector.YP_ENDPOINT_SETS:
                        print(backend_pb.meta.namespace_id + '/' + backend_pb.meta.id, 'types', types)
                        print('* ' + comment)
    elif condition == 'SWAT-6183':
        OK_ABC_STATES = ('develop', 'supported')
        if abc_token is None:
            raise click.ClickException('ABC_TOKEN is required for this condition')
        abc_client = AbcClient(abc_token)
        stats = collections.defaultdict(list)
        with click.progressbar(awacs_client.iter_all_namespaces()) as bar:
            for meta_pb in bar:
                try:
                    state = abc_client.get_abc_service(meta_pb.abc_service_id)['state']
                except requests.HTTPError as e:
                    if e.response.status_code == 404:
                        stats['not_found'].append((meta_pb.id, meta_pb.abc_service_id))
                    else:
                        click.echo(e, meta_pb.id, meta_pb.abc_service_id)
                    continue
                if state not in OK_ABC_STATES:
                    stats[state].append((meta_pb.id, meta_pb.abc_service_id))
        for key in stats:
            click.echo(key)
            for _id, abc_service_id in stats[key]:
                click.echo('{}: {}'.format(_id, abc_service_id))
            click.echo()
    elif condition == 'SHAWSHANK':
        for balancer_pb in awacs_client.iter_all_balancers():
            if balancer_pb.spec.container_spec.outbound_tunnels:
                print('/'.join((balancer_pb.meta.namespace_id, balancer_pb.meta.id)))
    elif condition == 'JUST_PARSE':
        for upstream_pb in awacs_client.iter_all_upstreams():
            if upstream_pb.meta.namespace_id in ('romanovich_knoss_fast_testing', 'knoss_fast_testing', 'knoss_fast'):
                continue
            yml = upstream_pb.spec.yandex_balancer.yaml
            try:
                yamlparser.parse(modules_pb2.Holder, yml)
            except Exception as e:
                print(six.text_type(e))
                print(create_awacs_upstream_href(upstream_pb.meta.namespace_id, upstream_pb.meta.id))
    elif condition == 'NANNY_SERVICE_IDS':
        for balancer_pb in awacs_client.iter_all_balancers():
            print(balancer_pb.spec.config_transport.nanny_static_file.service_id)
    elif condition == 'PAUSED':
        for balancer_pb in awacs_client.iter_all_balancers():
            if balancer_pb.meta.transport_paused.value:
                print('{}:{} paused since {} by {}'.format(balancer_pb.meta.namespace_id, balancer_pb.meta.id,
                                                           balancer_pb.meta.transport_paused.mtime.ToDatetime(),
                                                           balancer_pb.meta.transport_paused.author))
    elif condition == 'SWAT-6825':
        for endpoint_set_pb in awacs_client.iter_all_endpoint_sets():
            ns_id = endpoint_set_pb.meta.namespace_id
            endpoint_set_id = endpoint_set_pb.meta.id
            for instance_pb in endpoint_set_pb.spec.instances:
                if instance_pb.host == u'please-change-me':
                    backend_pb = awacs_client.get_backend(ns_id, endpoint_set_id)
                    if (u'validated' in six.text_type(backend_pb.statuses) or
                        u'validated' in six.text_type(backend_pb.l3_statuses)):
                        print(create_awacs_backend_href(ns_id, endpoint_set_id))
    elif condition == 'SWAT-6200':
        nanny_client = app.nanny_client

        cache = {}

        def _has_move_policy_and_yplite(service_id):
            try:
                policy_pb = nanny_client.get_replication_policy(service_id)
            except nanny_rpc_client.exceptions.NotFoundError:
                # No policy specified
                return False
            return policy_pb.spec.replication_method == policy_pb.spec.MOVE

        def has_move_policy_and_yplite(service_id):
            if service_id in cache:
                return cache[service_id]
            v = _has_move_policy_and_yplite(service_id)
            cache[service_id] = v
            return v

        for backend_pb in awacs_client.iter_all_backends():
            if backend_pb.spec.selector.type != backend_pb.spec.selector.NANNY_SNAPSHOTS:
                continue
            services = []
            for nanny_snapshot_pb in backend_pb.spec.selector.nanny_snapshots:
                if has_move_policy_and_yplite(nanny_snapshot_pb.service_id):
                    services.append(nanny_snapshot_pb.service_id)
            if services:
                print(create_awacs_backend_href(backend_pb.meta.namespace_id, backend_pb.meta.id))
                print("Services: {}".format(' '.join(services)))
    elif condition == 'YP_ENDPOINT_SETS':
        def find_man_instance_hostname(instance_pbs):
            for instance_pb in instance_pbs:
                if 'man' in instance_pb.host:
                    return instance_pb.host
            return None

        MOSCOW_TZ = pytz.timezone('Europe/Moscow')
        UTC_TZ = pytz.utc
        start_ctime = UTC_TZ.localize(datetime(2019, 11, 26, 0, 0, 0))
        end_ctime = UTC_TZ.localize(datetime(2019, 11, 26, 23, 0, 0))

        for backend_pb in awacs_client.iter_all_backends():
            if backend_pb.spec.selector.type != backend_pb.spec.selector.YP_ENDPOINT_SETS:
                continue
            man_endpoint_set_id = None
            for es_pb in backend_pb.spec.selector.yp_endpoint_sets:
                if es_pb.cluster == 'man':
                    man_endpoint_set_id = es_pb.endpoint_set_id
            if not man_endpoint_set_id:
                continue
            try:
                for es_rev_pb in awacs_client.list_endpoint_set_revs(namespace_id=backend_pb.meta.namespace_id,
                                                                     backend_id=backend_pb.meta.id):
                    if not find_man_instance_hostname(es_rev_pb.spec.instances):
                        ctime = es_rev_pb.meta.ctime.ToDatetime().replace(tzinfo=pytz.utc)
                        if start_ctime < ctime < end_ctime:
                            print(ctime.astimezone(MOSCOW_TZ).strftime('%Y-%m-%d %H:%M:%S %Z%z'),
                                  man_endpoint_set_id)
                            print(create_awacs_backend_href(backend_pb.meta.namespace_id, backend_pb.meta.id))
            except nanny_rpc_client.exceptions.NotFoundError:
                pass
    elif condition == 'L7_MACRO_VERSIONS':
        c = collections.Counter()
        for balancer_pb in awacs_client.iter_all_balancers():
            if not balancer_pb.spec.yandex_balancer.config.HasField('l7_macro'):
                c['NOT_L7_MACRO'] += 1
                continue
            version = balancer_pb.spec.yandex_balancer.config.l7_macro.version
            c[version] += 1
            print('.')
        for k, v in sorted(c.items()):
            print(k, v)
        print('total', sum(c.values()))
    elif condition == 'L7_CTL_VERSIONS':
        c = collections.Counter()
        for balancer_pb in awacs_client.iter_all_balancers():
            version = balancer_pb.spec.ctl_version
            c[version] += 1
            print('.')
        for k, v in sorted(c.items()):
            print(k, v)
        print('total', sum(c.values()))
    elif condition == 'L7_CTL_VERSIONS_W_IDS':
        c = collections.defaultdict(set)
        for balancer_pb in awacs_client.iter_all_balancers():
            version = balancer_pb.spec.ctl_version
            c[version].add(balancer_pb.meta.namespace_id)
            print('.')
        for k, v in sorted(c.items()):
            if k != 5:
                print('===', k)
                print('* ' + '\n* '.join(sorted(v)))
    elif condition == 'BACKEND_TYPES':
        c = collections.Counter()
        for backend_pb in awacs_client.iter_all_backends():
            if backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS_SD:
                c['YP_ENDPOINT_SETS_SD'] += 1
            if backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS:
                c['YP_ENDPOINT_SETS'] += 1
            if backend_pb.spec.selector.type == backend_pb.spec.selector.MANUAL:
                c['MANUAL'] += 1
            if backend_pb.spec.selector.type == backend_pb.spec.selector.GENCFG_GROUPS:
                c['GENCFG_GROUPS'] += 1
            if backend_pb.spec.selector.type == backend_pb.spec.selector.NANNY_SNAPSHOTS:
                c['NANNY_SNAPSHOTS'] += 1
            print('.')
        print('count', c)
        print('total', sum(c.values()))
        print('noctl', c['MANUAL'] + c['YP_ENDPOINT_SETS_SD'])
    elif condition == 'YP_ES_LOCATIONS':
        c = collections.Counter()
        for backend_pb in awacs_client.iter_all_backends():
            if (backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS_SD or
                backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS):
                for yp_es_pb in backend_pb.spec.selector.yp_endpoint_sets:
                    c[yp_es_pb.cluster] += 1
            print('.')
        print('count', c)
        print('total', sum(c.values()))
    elif condition == 'INVALID_BACKENDS':
        for backend_pb in awacs_client.iter_all_backends():
            invalid = False
            invalid_msg = ''
            ns_id = backend_pb.meta.namespace_id
            backend_id = backend_pb.meta.id
            for attr in ('statuses', 'l3_statuses', 'dns_record_statuses'):
                for pb in getattr(backend_pb, attr):
                    for _, status_pb in six.iteritems(pb.validated):
                        if status_pb.status != 'True':
                            invalid = True
                            invalid_msg = 'rev {} is not valid in {}'.format(pb.id, attr)
            if invalid:
                click.secho(create_awacs_backend_href(ns_id, backend_id), fg='white', bg='red')
                click.echo('{}:{} is not valid ({})'.format(ns_id, backend_id,
                                                            invalid_msg))
                click.echo()
    elif condition == 'FAILED_BACKENDS':
        c = 0
        for backend_pb in awacs_client.iter_all_backends():
            ns_id = backend_pb.meta.namespace_id
            backend_id = backend_pb.meta.id
            if backend_pb.spec.selector.type == backend_pb.spec.selector.YP_ENDPOINT_SETS_SD:
                continue
            if backend_pb.resolver_status.last_attempt.succeeded.status == 'False':
                print('{}:{}: not resolved: {}'.format(ns_id, backend_id,
                                                       backend_pb.resolver_status.last_attempt.succeeded.message))
                c += 1
        print('count', c)
    elif condition == 'DUPLICATE_BALANCER_IDS':
        seen_balancer_ids = collections.defaultdict(set)
        for balancer_pb in awacs_client.iter_all_balancers():
            balancer_id = balancer_pb.meta.id
            ns_id = balancer_pb.meta.namespace_id
            seen_balancer_ids[balancer_id].add(ns_id)
        for balancer_id, namespace_ids in six.iteritems(seen_balancer_ids):
            if len(namespace_ids) > 1:
                print('{} is present in the following namespaces: {}'.format(
                    balancer_id, ', '.join(sorted(namespace_ids))))
    elif condition == 'YP_LITE':
        for aspects_set in awacs_client.iter_all_balancer_aspects_sets():
            ns_id = aspects_set.meta.namespace_id
            content_pb = aspects_set.content.cluster.content
            if not content_pb.active_conf_id:
                continue
            if content_pb.platform == content_pb.YP_LITE:
                print(ns_id)
    elif condition == 'QS':
        rv = set()
        for balancer_pb in awacs_client.iter_all_balancers():
            if balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.QUICK_START_MODE:
                ns_id = balancer_pb.meta.namespace_id
                rv.add(ns_id)
        for id_ in sorted(rv):
            print(id_)
        print()
        print('Total #:', len(rv))
    elif condition == 'TLEM':
        rv = set()
        for balancer_pb in awacs_client.iter_all_balancers():
            if balancer_pb.spec.yandex_balancer.mode == balancer_pb.spec.yandex_balancer.EASY_MODE:
                ns_id = balancer_pb.meta.namespace_id
                rv.add(ns_id)
        for id_ in sorted(rv):
            print(id_)
        print()
        print('Total #:', len(rv))
    elif condition == 'COUNT_UPSTREAMS':
        c = 0
        for _ in awacs_client.iter_all_upstreams():
            c += 1
        print('Total # of upstreams', c)
    elif condition == 'SWAT-6428':
        from collections import defaultdict
        not_valid_tags = []
        intersection_by_tags = defaultdict(lambda: {'b_ids': [], 'count': 0})

        for balancer_aspect_pb in awacs_client.iter_all_balancer_aspects_sets():
            tags = balancer_aspect_pb.content.cluster.content.tags
            if tags.itype and tags.prj:
                locations = balancer_aspect_pb.content.cluster.content.locations or ['undefined']
                for location in locations:
                    tags_set = (str(tags.itype), str(tags.ctype), str(sorted(tags.prj)), str(sorted(tags.metaprj)),
                                str(location))
                    intersection_by_tags[tags_set]['b_ids'].append(balancer_aspect_pb.meta.balancer_id)
                    intersection_by_tags[tags_set]['count'] += 1
            else:
                locations = balancer_aspect_pb.content.cluster.content.locations or ['undefined']
                not_valid_tags.append((balancer_aspect_pb.meta.balancer_id, locations, tags))

        click.echo('[##] Balancers Tags Intersection:')
        for i, (tag, info) in enumerate(
            sorted(intersection_by_tags.items(), key=lambda x: x[1]['count'], reverse=True)):
            if info['count'] > 1:
                click.echo('\t[{}] Tag: {} has {} balancers: '.format(i + 1, tag, info['count']))
                for b_id in info['b_ids']:
                    click.echo("\t\t{}".format(b_id))

        click.echo('[##] Balancers Tags Not Valid:')
        for i, line in enumerate(not_valid_tags):
            click.echo("[{}]\t{}".format(i + 1, ", ".join([str(l).replace('\n', '') for l in line])))
    elif condition == 'AWACS-484':
        import os

        if abc_token is None:
            raise click.ClickException('ABC_TOKEN is required for this condition')
        if staff_token is None:
            raise click.ClickException('STAFF_TOKEN is required for this condition')

        def generate_input_for_startrek(data):
            class literal_unicode(six.text_type):
                pass

            def literal_unicode_representer(dumper, data):
                return dumper.represent_scalar(u'tag:yaml.org,2002:str', data, style='|')

            yaml.add_representer(literal_unicode, literal_unicode_representer)

            output = {
                'meta': {
                    'author': 'nanny-robot',
                    'description_template': literal_unicode(u'''\
Изменение настроек необходимо, чтобы мы могли заменить тип бэкенда на "YP endpoint sets SD (fast)". \
**НЕ МЕНЯЙТЕ** тип бэкенда самостоятельно, мы это сделаем автоматически.
Про типы бэкендов подробнее можно прочитать \
((https://wiki.yandex-team.ru/cplb/awacs/terminology/backend-types/ здесь)). \
Кратко: "YP endpoint sets SD (fast)" работает быстрее, так как резолвом IP-адресов занимается \
непосредственно L7-балансер, поэтому при переезде бэкендов аваксу не нужно делать и выкатывать новый конфиг, \
и l7-балансер видит новые адреса в течение минуты.

Ранее в awacs переопределение портов эндпоинтсетов использовалось для обхода невозможности указания в Yandex Deploy \
нескольких endpoint set в одном deploy unit. Теперь такой функционал появился.

Пожалуйста, сделайте следующее:
1. в Yandex Deploy добавьте новые эндпоинтсеты для тех случаев, где вы использовали переопределение портов в awacs.
2. В настройках бэкенда awacs:
  1. Добавьте новый эндпоинтсет с Port Policy Keep.
  2. Удалите соответствующий эндпоинтсет с Port Policy Shift или Override.
  3. Уберите кастомные веса для всех остальных эндпоинтсетов (Weight Override заменить на Keep).

<{ В картинках
===== Добавляем новый endpoint set в Деплое =====
[[https://jing.yandex-team.ru/files/anttsov/1.png]]
---
---
===== Выставляем нужный порт =====
[[https://jing.yandex-team.ru/files/anttsov/2.png]]
---
---
===== Копируем endpoint set id =====
[[https://jing.yandex-team.ru/files/anttsov/3.png]]
---
---
===== Изначальное состояние в awacs. !!Красным!! подчеркнуты настройки, которые следует изменить. \
В бэкенде два эндпоинтсета с одинаковыми id, во втором из них переопределен порт, также переопределены веса. =====
[[https://jing.yandex-team.ru/files/anttsov/4.png]]
---
---
===== Итоговое состояние в awacs. !!(зел)Зеленым!! подчеркнуты настройки, которые обязательно должны быть. \
Во втором эндпоинтсете изменен id, убрано переопределение весов =====
[[https://jing.yandex-team.ru/files/anttsov/5.png]]
}>

{{message}}
'''),
                    'followers': ['anttsov', 'romanovich'],
                    'parent_issue_id': 'AWACS-484',
                    'queue': 'AWACSCONFIGS',
                    'summary_template':
                        u'Избавиться от переопределения портов и весов в бэкендах балансера {{namespace_id}}',
                    'tags': [],
                },
                'issues': {}
            }

            for namespace, v in data.items():
                invitee = v['logins']
                backends = v['backends']
                message = (
                    u'Список бэкендов с переопределенными портами и/или весами в неймспейсе '
                    u'((https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/show/ {})):'
                ).format(namespace, namespace)

                for backend in backends:
                    tmpl = u'\n * {{[((https://nanny.yandex-team.ru/ui/#/awacs/namespaces/list/{}/backends/list/{}/edit/ {}))]}}'
                    message += tmpl.format(namespace, backend, backend)

                output['issues'][namespace] = {
                    'assignee': '',
                    'tags': [],
                    'invitee': invitee,
                    'context': {
                        'namespace_id': namespace,
                        'message': literal_unicode(message),
                    }
                }

            return output

        def collect_data():
            abc_client = AbcClient(abc_token)
            staff_client = StaffClient(staff_token)
            cnt = 0
            cnt_custom_weight = 0
            data = {}
            namespaces = {}
            is_robot_dict = {}

            def is_robot(login):
                if login in is_robot_dict:
                    return is_robot_dict[login]
                staff_resp = staff_client.get_person(backend_pb.meta.author, params={'_fields': 'official.is_robot'})
                res = staff_resp['result'][0]['official']['is_robot']
                is_robot_dict[login] = res
                return res

            def unique(seq):
                seen = set()
                return [x for x in seq if not (x in seen or seen.add(x))]

            for backend_pb in awacs_client.iter_all_backends():
                if backend_pb.spec.selector.type != backend_pb.spec.selector.YP_ENDPOINT_SETS:
                    continue

                deploy_is_used = False
                custom_weight = False
                custom_port = False
                for v in backend_pb.spec.selector.yp_endpoint_sets:
                    if re.match(r'[\w\-]+\.[\w\-]+', v.endpoint_set_id):
                        deploy_is_used = True
                    if v.port.policy in (v.port.OVERRIDE, v.port.SHIFT):
                        custom_port = True
                    if v.weight.policy == v.weight.OVERRIDE:
                        if not custom_weight:
                            cnt_custom_weight += 1
                        custom_weight = True

                if not deploy_is_used:
                    continue

                port = backend_pb.spec.selector.port
                custom_port = custom_port or port.policy in (port.OVERRIDE, port.SHIFT)

                if not (custom_port or custom_weight):
                    continue

                def backend_rev_is_active():
                    for status in backend_pb.statuses:
                        if status.id != backend_pb.meta.version:
                            continue
                        for k in status.active:
                            if status.active[k].status == 'True':
                                return True
                    return False

                if not backend_rev_is_active():
                    continue

                cnt += 1

                ns_id = backend_pb.meta.namespace_id
                ns_pb = namespaces.get(ns_id)
                if ns_pb is None:
                    ns_pb = awacs_client.get_namespace(namespace_id=ns_id)
                    namespaces[ns_id] = ns_pb

                author_is_robot = is_robot(backend_pb.meta.author)

                if ns_id in data:
                    if not author_is_robot:
                        data[ns_id]['logins'].insert(0, backend_pb.meta.author)
                    data[ns_id]['backends'].append(backend_pb.meta.id)
                else:
                    fields = ['person.login', 'person.is_robot', 'role.scope.slug']
                    abc_members_resp = abc_client.get_members(
                        ns_pb.meta.abc_service_id, params={'fields': ','.join(fields)})

                    def get_logins(group):
                        logins = []
                        for res in abc_members_resp['results']:
                            role = res['role']
                            if role['scope']['slug'] == group and not res['person']['is_robot']:
                                logins.append(res['person']['login'])
                        return logins

                    def uniq_rand(seq, n):
                        seq = unique(seq)
                        res = []
                        m = min(n, len(seq))
                        while len(res) < m:
                            res.append(random.choice(seq))
                            res = unique(res)
                        return res

                    nlogins = 4

                    logins = []
                    if not author_is_robot:
                        logins.append(backend_pb.meta.author)
                    logins.extend(uniq_rand(get_logins('administration'), nlogins - 1))
                    logins.extend(uniq_rand(get_logins('dutywork'), nlogins - 1))
                    logins.extend(uniq_rand(get_logins('development'), nlogins - 1))

                    if len(logins) == 1:
                        r = abc_client.get_abc_service(ns_pb.meta.abc_service_id)
                        print('namespace={}, abc_id={}, name={}'.format(ns_id, r['id'], r['name']['en']))
                        # print(json.dumps(r, ensure_ascii=False, encoding='utf8', indent=4))

                    data[ns_id] = dict(
                        logins=unique(logins)[:nlogins],
                        backends=[backend_pb.meta.id])

                    if len(logins) < 5:
                        print('{}: number of logins = {}'.format(ns_id, len(logins)))

            for k, v in data.items():
                v['logins'] = unique(v['logins'])
                assert len(v['backends']) > 0

            print('cnt:', cnt)
            print('cnt_custom_weight:', cnt_custom_weight)
            return data

        path = 'infra/awacs/tools/startreker/tasks/awacs-484/'
        if not os.path.exists(path):
            os.mkdir(path)

        data_path = path + 'data.yaml'

        if os.path.exists(data_path):
            with open(data_path) as f:
                data = yaml.load(f.read(), Loader=yaml.FullLoader)
        else:
            data = collect_data()
            with open(data_path, 'w') as f:
                yaml.safe_dump(data, f)

        inp = generate_input_for_startrek(data)
        with open(path + 'input.yaml', 'w') as f:
            yaml.dump(inp, f, encoding='utf8', allow_unicode=True)

    else:
        raise AssertionError()


if __name__ == '__main__':
    cli()
