# coding: utf-8
import collections
import json

import click

from infra.awacs.tools.awacsplaybookbuilder.const import BLACKLISTED_NS_IDS, MANUAL_NS_IDS
from infra.awacs.tools.awacstoolslib.listers import (list_namespace_ids_with_large_rps,
                                                     list_namespace_ids_with_yp_lite_balancers)
from infra.awacs.tools.awacstoolslib.util import cli, wait_for_confirmation


def list_non_all_dynamic_gencfg_ns_ids(yp_lite_namespace_ids, processable_namespace_ids):
    rv = set()
    with open('./gencfg-from-pirogov.txt') as f:  # provided by pirogov@, slightly outdated
        for line in f:
            ns_id = line.split()[-1].strip()
            rv.add(ns_id)
    # some of the gencfg-from-pirogov namespaces might be
    # * already removed
    # * already migrated to YP.lite
    # let's account for this fact:
    return (rv - yp_lite_namespace_ids) & processable_namespace_ids


def split_off_taxi_ns_ids(ns_ids):
    rv = {ns_id for ns_id in ns_ids if 'taxi' in ns_id}
    return rv, ns_ids - rv


def split_off_maps_ns_ids(ns_ids):
    rv = {ns_id for ns_id in ns_ids if 'maps.yandex.net' in ns_id}
    rv.add('core-pht-proxy.maps.yandex.ru')
    rv.add('core-stv-ugc-proxy.maps.yandex.ru')
    rv.add('static-pano.maps.yandex.ru')
    rv.discard('saas-dm-maps.yandex.net')
    rv.discard('saas-searchproxy-maps.yandex.net')
    return rv, ns_ids - rv


def split_off_front_maps_ns_ids(ns_ids):
    rv = {ns_id for ns_id in ns_ids
          if ns_id.endswith('.slb.maps.yandex.net')}
    return rv, ns_ids - rv


def validate_stages(ns_ids, stages):
    for i, stage_x in sorted(stages.items()):
        for j, stage_y in sorted(stages.items()):
            assert i == j or not (stage_x & stage_y), '{} intersects w/ {}: {}'.format(i, j, stage_x & stage_y)

    assert set.union(*stages.values()) == ns_ids, (
        'ns_ids - set.union(*stages): {}\n'
        'set.union(*stages) - ns_ids: {}'.format(ns_ids - set.union(*stages.values()),
                                                 set.union(*stages.values()) - ns_ids))


@cli.command()
@click.pass_obj
@click.option('--playbook-id', required=True)
def build(app, playbook_id):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    :type playbook_id: str
    """
    assert not app.is_playbook_configured()
    assert app.is_op_configured()

    all_namespace_ids = set(app.awacs_client.list_namespace_ids())
    if not BLACKLISTED_NS_IDS.issubset(all_namespace_ids):
        print('BLACKLISTED_NS_IDS - all_namespace_ids:', BLACKLISTED_NS_IDS - all_namespace_ids)
        raise AssertionError()

    ns_ids = all_namespace_ids - BLACKLISTED_NS_IDS

    if 'yp_lite_namespace_ids' in app.op.data:
        yp_lite_namespace_ids = set(app.op.data['yp_lite_namespace_ids'])
        click.echo('Got YP.lite ns ids from op cache')
    else:
        click.echo('Reading YP.lite ns ids')
        yp_lite_namespace_ids = list_namespace_ids_with_yp_lite_balancers(app,
                                                                          namespace_id_in=all_namespace_ids)
        app.op.data['yp_lite_namespace_ids'] = sorted(yp_lite_namespace_ids)
        app.op.save()
        click.echo('Done reading YP.lite ns ids, {} of them'.format(len(yp_lite_namespace_ids)))

    # XXX
    # ns_ids = yp_lite_namespace_ids
    # /XXX

    ns_ids = ns_ids - BLACKLISTED_NS_IDS

    non_all_dynamic_gencfg_ns_ids = list_non_all_dynamic_gencfg_ns_ids(
        yp_lite_namespace_ids=yp_lite_namespace_ids,
        processable_namespace_ids=ns_ids)

    large_rps_ns_ids = list_namespace_ids_with_large_rps(app, threshold=15000)

    all_ns_ids = set(ns_ids)
    taxi_ns_ids, ns_ids = split_off_taxi_ns_ids(ns_ids)
    maps_ns_ids, ns_ids = split_off_maps_ns_ids(ns_ids)
    front_maps_ns_ids, _ = split_off_front_maps_ns_ids(maps_ns_ids)

    STAGE_1 = (non_all_dynamic_gencfg_ns_ids - large_rps_ns_ids - maps_ns_ids - taxi_ns_ids -
               MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_2 = (ns_ids - non_all_dynamic_gencfg_ns_ids - maps_ns_ids - taxi_ns_ids -
               large_rps_ns_ids - MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_3 = (taxi_ns_ids - MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_4 = (maps_ns_ids - MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_4_FRONT = (front_maps_ns_ids - large_rps_ns_ids - MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_5 = (large_rps_ns_ids - taxi_ns_ids - maps_ns_ids - MANUAL_NS_IDS - BLACKLISTED_NS_IDS) & all_ns_ids
    STAGE_6 = MANUAL_NS_IDS & all_ns_ids
    STAGE_X = BLACKLISTED_NS_IDS & all_ns_ids

    STAGES = [STAGE_1, STAGE_2, STAGE_3, STAGE_4, STAGE_5, STAGE_6]
    validate_stages(all_ns_ids, {
        'STAGE_1': STAGE_1,
        'STAGE_2': STAGE_2,
        'STAGE_3': STAGE_3,
        'STAGE_4': STAGE_4,
        'STAGE_5': STAGE_5,
        'STAGE_6': STAGE_6,
        'STAGE_X': STAGE_X,
    })

    with open('./playbooks/{}-playbook.json'.format(playbook_id), 'w') as f:
        json.dump({
            'STAGE_1': sorted(STAGE_1),
            'STAGE_2': sorted(STAGE_2),
            'STAGE_3': sorted(STAGE_3),
            'STAGE_4': sorted(STAGE_4),
            'STAGE_4_FRONT': sorted(STAGE_4_FRONT),
            'STAGE_5': sorted(STAGE_5),
            'STAGE_6': sorted(STAGE_6),
            'STAGE_UNION': sorted(set.union(*STAGES)),
            'STAGE_X': sorted(STAGE_X),
        }, f, sort_keys=True, indent=2)

    if not wait_for_confirmation('Playbook is ready. Proceed building index?'):
        return

    click.echo('Building index')
    with open('./playbooks/{}-index.json'.format(playbook_id), 'w') as f:
        index = collections.defaultdict(dict)

        for ns_id in set.union(*STAGES):
            balancer_pbs = app.awacs_client.list_balancers(ns_id)
            for balancer_pb in balancer_pbs:
                if balancer_pb.spec.incomplete:
                    continue
                b_id = balancer_pb.meta.id
                location = (balancer_pb.meta.location.yp_cluster or
                            balancer_pb.meta.location.gencfg_dc or
                            u'XDC').lower()
                service_id = balancer_pb.spec.config_transport.nanny_static_file.service_id
                index[ns_id].setdefault(location, []).append([b_id, service_id])
        json.dump(index, f, indent=2, sort_keys=True)
    click.echo('Done building index')


@cli.command()
@click.pass_obj
@click.option('--stage', required=True)
@click.option('--location', required=True)
def ls(app, stage, location):
    """
    :type app: infra.awacs.tools.awacstoolslib.app.App
    """
    assert app.is_playbook_configured()

    for ns_id, balancer_id in app.playbook.list_stage_balancer_full_ids(stage=stage, location=location):
        click.echo(ns_id + ':' + balancer_id)


if __name__ == '__main__':
    cli()
